/* ************************************************************************
 * 
 * MOENGAGE CONFIDENTIAL
 * __________________
 * 
 *  [2014] - [2015] MoEngage Inc. 
 *  All Rights Reserved.
 * 
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */

package com.moe.pushlibrary.utils;

import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.widget.Toast;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;
import java.util.Set;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * A Utility class for
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.0
 */
public final class MoEHelperUtils {

  private MoEHelperUtils() {
    // Utility class cannot be instantiated
  }

  /**
   * Convert the information contained in a JSONObject into a Bundle
   *
   * @param jsonObject The JSONObject which needs to be converted into a bundle
   * @return returns a populated bundle using the JSONObject that was passed
   */
  public static Bundle convertJSONObjecttoBundle(JSONObject jsonObject) {
    Bundle newBundle = null;

    try {
      @SuppressWarnings("unchecked") Iterator<String> keys = jsonObject.keys();
      newBundle = new Bundle();
      while (keys.hasNext()) {
        String nextKey = keys.next();
        Object nextValue = jsonObject.get(nextKey);
        if (nextValue instanceof Integer) {
          newBundle.putInt(nextKey, (Integer) nextValue);
        } else if (nextValue instanceof Boolean) {
          newBundle.putBoolean(nextKey, (Boolean) nextValue);
        } else if (nextValue instanceof String) {
          newBundle.putString(nextKey, (String) nextValue);
        } else if (nextValue instanceof Byte) {
          newBundle.putByte(nextKey, (Byte) nextValue);
        } else if (nextValue instanceof Character) {
          newBundle.putChar(nextKey, (Character) nextValue);
        } else if (nextValue instanceof Double) {
          newBundle.putDouble(nextKey, (Double) nextValue);
        } else if (nextValue instanceof Float) {
          newBundle.putFloat(nextKey, (Float) nextValue);
        } else if (nextValue instanceof Long) {
          newBundle.putLong(nextKey, (Long) nextValue);
        } else if (nextValue instanceof Short) {
          newBundle.putShort(nextKey, (Short) nextValue);
        }
      }
    } catch (Exception e) {
      return newBundle;
    }
    return newBundle;
  }

  /**
   * Copies the specified coupon code to the Clipboard.
   *
   * @param mContext An instance of the application Context
   * @param couponcode The coupon code to be added to the Clipboard
   */
  public static void copyCouponCodeToClipboard(Context mContext, String couponcode) {
    copyToClipboard(mContext, couponcode);
    showToast("Coupon code copied to clipboard", mContext);
  }

  /**
   * Copies the specified coupon code to the Clipboard.
   *
   * @param context An instance of the application Context
   * @param copyText The coupon code to be added to the Clipboard
   * @param message Message to be shown after copy
   */
  public static void copyTextToClipboardAndShowToast(Context context, String copyText,
      String message) {
    copyToClipboard(context, copyText);
    showToast(message, context);
  }

  /**
   * Shows a Toast message
   *
   * @param message The message which needs to be shown
   * @param mContext An instance of the application Context
   */
  public static void showToast(String message, Context mContext) {
    if (!TextUtils.isEmpty(message)) {
      Toast.makeText(mContext, message, Toast.LENGTH_SHORT).show();
    }
  }

  private static void copyToClipboard(Context mContext, String couponCode) {
    android.content.ClipboardManager clipboard =
        (android.content.ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
    android.content.ClipData clip =
        android.content.ClipData.newPlainText("coupon code", couponCode);
    clipboard.setPrimaryClip(clip);
  }

  /**
   * Logs the contents of the specified bundle
   *
   * @param extras The Bundle which needs to be dumped for debugging
   */
  public static void dumpIntentExtras(@NonNull Bundle extras) {
    // TODO: 04/07/16 temporary fix for null check
    if (extras == null) return;
    Set<String> ketSet = extras.keySet();
    if (null == ketSet || ketSet.isEmpty()) return;
    Logger.v("------Start of bundle extras------");
    for (String key : ketSet) {
      Object obj = extras.get(key);
      if (null != obj) {
        Logger.v("[ " + key + " = " + obj.toString() + " ]");
      }
    }
    Logger.v("-------End of bundle extras-------");
  }

  public static void dumpIntentExtras(Intent intent) {
    if (null != intent) {
      dumpIntentExtras(intent.getExtras());
    }
  }

  /**
   * Check if the activity is changing configuration.
   *
   * @param activity The activity which needs to be checked
   * @return true if it is changing configuration and false if the activity is
   * not changing configuration
   */
  public static boolean isChangingConfiguration(final Activity activity) {
    int sdk = android.os.Build.VERSION.SDK_INT;
    if (sdk <= android.os.Build.VERSION_CODES.HONEYCOMB) {
      return false;
    } else {
      return detectConfigChange(activity);
    }
  }

  /**
   * Available to SDK running on devices with Honeycomb or above it
   *
   * @param activity The activity instance which needs to be checked for
   * configuration change
   * @return true if it is changing configuration and false if the activity is
   * not changing configuration
   */
  @TargetApi(Build.VERSION_CODES.HONEYCOMB) private static boolean detectConfigChange(
      Activity activity) {
    if (null == activity) {
      return false;
    } else {
      return activity.isChangingConfigurations();
    }
  }

  public static Intent getLauncherActivityIntent(Context context) {
    PackageManager pm = context.getPackageManager();
    return pm.getLaunchIntentForPackage(context.getPackageName());
  }

  /**
   * Downloads an image and returns a bitmap object. The image should be less than 450dp for the
   * push notification. An aspect ratio of 2:1 is recommended. This should always be run in a
   * background
   * thread.
   *
   * According to http://developer.android.com/guide/appendix/media-formats.html, the supported
   * file
   * types are jpg and png.
   *
   * @param imageUrl The url where the image is found
   * @return An image in Bitmap form. If the image cannot be downloaded, or cannot be decoded into
   * a bitmap, then null is returned.
   */
  public static Bitmap downloadImageBitmap(String imageUrl) {
    if (imageUrl == null || imageUrl.length() == 0) {
      Logger.e(
          "Null or empty Url string passed to image bitmap download. Not attempting download.");
      return null;
    }
    Bitmap bitmap = null;
    try {
      InputStream inputStream = new java.net.URL(imageUrl).openStream();
      bitmap = BitmapFactory.decodeStream(inputStream);
      if (inputStream != null) {
        inputStream.close();
      }
    } catch (OutOfMemoryError e) {
      Logger.f(String.format("Out of Memory Error in image bitmap download for Url: %s.", imageUrl),
          e);
    } catch (UnknownHostException e) {
      Logger.f(String.format("Unknown Host Exception in image bitmap download for Url: %s. Device "
          + "may be offline.", imageUrl), e);
    } catch (MalformedURLException e) {
      Logger.f(String.format("Malformed URL Exception in image bitmap download for Url: %s. Image "
          + "Url may be corrupted.", imageUrl), e);
    } catch (Exception e) {
      Logger.f(String.format("Exception in image bitmap download for Url: %s", imageUrl), e);
    }
    return bitmap;
  }


  public static final int BASELINE_SCREEN_DPI = 160;

  public static int getPxFromDp(int dpi, int dp) {
    return (dpi * dp) / BASELINE_SCREEN_DPI;
  }

  public static boolean isValidResourceId(Context appContext, int resourceId) {
    try {
      appContext.getResources().getResourceName(resourceId);
      return true;
    } catch (Resources.NotFoundException e) {
    }catch (Exception e){
    }
    return false;
  }

  /**
   * Attribute key which holds the value of EVENT_ACTION
   */
  private static final String EVENT_ACTION = "EVENT_ACTION";
  /**
   * Attribute key which holds the value of EVENT_ATTRS
   */
  public static final String EVENT_ATTRS = "EVENT_ATTRS";

  public static final String EVENT_ATTRS_CUST = "EVENT_ATTRS_CUST";
  /**
   * Attribute key which holds the value of EVENT_G_TIME
   */
  public static final String EVENT_G_TIME = "EVENT_G_TIME";
  /**
   * Attribute key which holds the value of EVENT_L_TIME
   */
  public static final String EVENT_L_TIME = "EVENT_L_TIME";

  public static final String EVENT_NON_INTERACTIVE = "N_I_E";

  private static final String EXTRA_KEY_FROM_INBOX = "from";

  private static final String EXTRA_VALUE_FROM_INBOX = "inbox";

  public static boolean isFromInbox(Bundle extras) {
    if (null == extras) return false;
    return extras.containsKey(EXTRA_KEY_FROM_INBOX) && EXTRA_VALUE_FROM_INBOX.equals(
        extras.getString(EXTRA_KEY_FROM_INBOX));
  }



  public static JSONObject getDatapointJSON(String action, JSONObject actionAttrs) {
    if (actionAttrs.has(EVENT_G_TIME) && actionAttrs.has(EVENT_L_TIME)) {
      try {
        actionAttrs.put(EVENT_ACTION, action);
      } catch (Exception e) {
        Logger.f("MoEHelperUtils: getDataPointJson ", e);
      }
      return actionAttrs;
    }
    return getDatapointJSON(action, actionAttrs, Long.toString(MoEUtils.currentTime()),
        MoEUtils.getDateDataPointFormat());
  }

  public static JSONObject getDatapointJSON(String action, JSONObject actionAttrs, String gtime,
      String ltime) {
    return getDatapointJSON(action, actionAttrs, null, gtime, ltime);
  }

  public static JSONObject getDatapointJSON(String action, JSONObject generalAttrs,
      JSONObject customAttrs, String gtime, String ltime) {
    JSONObject jsonInteraction = new JSONObject();
    try {
      jsonInteraction.put(EVENT_ACTION, action);
      if (null != generalAttrs) {
        jsonInteraction.put(EVENT_ATTRS, generalAttrs.toString());
      }
      if (null != customAttrs && customAttrs.length() != 0) {
        jsonInteraction.put(EVENT_ATTRS_CUST, customAttrs.toString());
      }
      jsonInteraction.put(EVENT_G_TIME, gtime);
      jsonInteraction.put(EVENT_L_TIME, ltime);
      return jsonInteraction;
    } catch (Exception e) {
      Logger.f("MoEHelperUtils:getDataPointJson", e);
    }
    return null;
  }

  public static String getActionFromEvent(String actionDetails) {
    try {
      JSONObject actionObj = new JSONObject(actionDetails);
      return actionObj.getString(EVENT_ACTION);
    } catch (Exception e) {
      Logger.f("MoEHelperUtils:getActionFromEvent", e);
    }
    return null;
  }

  public static boolean hasPermission(Context appContext, String permission) {
    try {
      if (null == appContext || TextUtils.isEmpty(permission)) return false;
      PackageManager pm = appContext.getPackageManager();
      return (pm.checkPermission(permission, appContext.getPackageName())
          == PackageManager.PERMISSION_GRANTED);
    } catch (Exception ignored) {
      Logger.e("MoEHelperUtils: hasPermission ", ignored);
    }
    return false;
  }

  public static JSONObject transformEventAttributesForEvaluationPackage(JSONObject eventAttributes){
    JSONObject attributes = new JSONObject();
    try {
      if (eventAttributes.has(MoEHelperUtils.EVENT_ATTRS)){
        attributes = new JSONObject(eventAttributes.getString(MoEHelperUtils.EVENT_ATTRS));
      }
      if (eventAttributes.has(MoEHelperUtils.EVENT_ATTRS_CUST)){
        JSONObject customAttributes = new JSONObject(eventAttributes.getString(EVENT_ATTRS_CUST));
        if (customAttributes.has("timestamp")){
          JSONArray timestampAttrs = customAttributes.getJSONArray("timestamp");
          for (int index = 0; index < timestampAttrs.length(); index++){
            JSONObject timeAttrJSON = timestampAttrs.getJSONObject(index);
            Iterator iterator = timeAttrJSON.keys();
            while (iterator.hasNext()){
              String key = (String) iterator.next();
              Date date = new Date();
              date.setTime(timeAttrJSON.getLong(key));
              DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH);
              attributes.put(key, format.format(date));
            }
          }
        }
      }
    } catch (JSONException e) {
      Logger.e(  "MoEHelperUtils transformEventAttributesForEvaluationPackage() : ");
    }
    return attributes;
  }
}
