/* ************************************************************************
 *
 * MOENGAGE CONFIDENTIAL
 * __________________
 *
 *  [2014] - [2015] MoEngage Inc.
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.core;

import android.content.Context;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import com.moe.pushlibrary.exceptions.SDKNotInitializedException;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import javax.net.ssl.HttpsURLConnection;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author MoEngage (abhishek@moengage.com)
 * @version 3.0
 * @since 1.0
 */
final class MoERestClient {

  private static final String TAG = "MoERestClient";

  public enum RequestMethod {
    GET, POST
  }

  private static String ANDROID_ID = null;
  private static boolean androidIDRetrieved = false;

  private HashMap<String, String> params;
  private HashMap<String, String> headers;
  private JSONObject jsonBody = null;
  private String url;
  private JSONObject queryParams;

  private int responseCode;

  private String response;

  private String errorResponse;

  private String appId;

  private String path;

  public String getResponse() {
    return response;
  }

  int getResponseCode() {
    return responseCode;
  }

  /**
   * @throws SDKNotInitializedException
   */
  MoERestClient(String url, Context con) throws SDKNotInitializedException {
    this.url = url;
    params = new HashMap<>();
    if (!androidIDRetrieved) {
      androidIDRetrieved = true;
      ANDROID_ID = MoEUtils.getAndroidID(con);
    }
    path = getUrlPath(url);
    appId = MoEUtils.getAppId();
    if (TextUtils.isEmpty(appId)) throw new SDKNotInitializedException("App ID has not been set");
    queryParams = new JSONObject();
    initializeRestClient(con);
    headers = new HashMap<>();
    headers.put(MoEConstants.REQUEST_HEADER_APP_KEY, appId);
  }

  private void initializeRestClient(Context con) throws SDKNotInitializedException {
    //["os", "app_id", "os_ver", "sdk_ver", "model", "app_ver","device_ts","device_tz", "unique_id"]
    try {
      ConfigurationProvider provider = ConfigurationProvider.getInstance(con);
      String registrationId = provider.getFcmToken();
      String newUniqueID = provider.getCurrentUserId();
      String appVersion = Integer.toString(provider.getAppVersion());
      long millis = MoEUtils.currentTime();
      if (!TextUtils.isEmpty(registrationId) && !provider.isPushNotificationOptedOut()) {
        queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_FCM_PUSH_ID, registrationId);
      }
      if (!TextUtils.isEmpty(newUniqueID)) {
        queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_UUID, newUniqueID);
      }
      if (!TextUtils.isEmpty(appVersion)) {
        queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION, appVersion);
      }
      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_APP_ID, appId);

      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_OS, MoEConstants.GENERIC_PARAM_V2_VALUE_OS);
      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_SDK_VERSION,
          Integer.toString(MoEHelperConstants.LIB_VERSION));

      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE_OFFSET,
          String.valueOf(TimeZone.getDefault().getOffset(millis)));
      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_TIMESTAMP, String.valueOf(millis));
      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());
      queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_PUSH_SERVER, provider.getPushService());
      String miPushToken = provider.getMiPushToken();
      if (!TextUtils.isEmpty(miPushToken)
          && !provider.isPushNotificationOptedOut()) {
        queryParams.put(MoEConstants.PARAM_V2_KEY_MI_PUSH_TOKEN, miPushToken);
      }
      addIntegrationType();

      if (!provider.isDataTrackingOptedOut()) {
        if (!TextUtils.isEmpty(ANDROID_ID)) {
          queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, ANDROID_ID);
        }
        if (!SdkConfig.getConfig().isGaidTrackingOptedOut) {
          String gaid = provider.getStoredGAID();
          if (TextUtils.isEmpty(gaid)) {
            AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(con);
            if (adInfo != null) {
              gaid = adInfo.getId();
              provider.storeGAID(gaid);
            }
          }
          if (!TextUtils.isEmpty(gaid)) {
            queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
          }
        }
        queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION,
            String.valueOf(Build.VERSION.SDK_INT));
        queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL);
        queryParams.put(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
            provider.getAppVersionName());

        String nwType = MoEUtils.getNetworkType(con);
        if (!TextUtils.isEmpty(nwType)) {
          queryParams.put(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + path + " initializeRestClient() : ", e);
    }
  }

  /**
   * Add a request param
   *
   * @param name Name of the get param
   * @param Value Value of the get param
   */
  private void addParam(String name, String Value) {
    params.put(name, Value);
  }

  void addParam(HashMap<String, String> paramsMap) {
    for (Map.Entry<String, String> entry : paramsMap.entrySet()) {
      params.put(entry.getKey(), entry.getValue());
    }
  }

  /**
   * Add a request body
   *
   * @param requestBody The request body which needs to be added to the HTTP request
   */
  void addBody(JSONObject requestBody) {
    jsonBody = requestBody;
  }

  /**
   * Convert input stream to String
   *
   * @param inputStream The input stream from the API response entity
   * @return String representation of the API response jsonBody
   */
  private String convertStreamToString(InputStream inputStream) {
    if (inputStream == null) return null;
    BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
    StringBuilder sb = new StringBuilder();

    String line;
    try {
      while ((line = reader.readLine()) != null) {
        sb.append(line);
      }
    } catch (IOException e) {
      Logger.f(TAG + path + " executeRequest: IOException", e);
    } catch (Exception e) {
      Logger.f(TAG + path + " executeRequest: Exception", e);
    } finally {
      try {
        inputStream.close();
      } catch (IOException e) {
        Logger.f(TAG + path + " executeRequest: IOException", e);
      } catch (Exception e) {
        Logger.f(TAG + path +  " executeRequest: Exception", e);
      }
    }
    return sb.toString();
  }

  private static final String SCHEME_HTTPS = "https://";

  public void execute(RequestMethod method) throws IOException {
    URL finalURL = new URL(url);
    Logger.d(TAG + path + " executing API: " + finalURL.toString());
    InputStream in;
    HttpURLConnection urlConnection = null;
    if (url.startsWith(SCHEME_HTTPS)) {
      urlConnection = (HttpsURLConnection) urlConnection;
      urlConnection = (HttpsURLConnection) finalURL.openConnection();
    } else {
      urlConnection = (HttpURLConnection) finalURL.openConnection();
    }
    addConnectionTimeOut(urlConnection, 10);
    if (method == RequestMethod.POST) {
      addBody(urlConnection);
    } else {
      urlConnection.setRequestMethod("GET");
      addHeaders(urlConnection);
    }
    responseCode = urlConnection.getResponseCode();
    Logger.d(TAG + path + " ResponseCode: " + responseCode);
    if (200 != responseCode) {
      errorResponse = convertStreamToString(urlConnection.getErrorStream());
      Logger.f(TAG + path + " Response: API Failed: "
          + " response code :"
          + responseCode
          + "reason : "
          + errorResponse);
      if (!TextUtils.isEmpty(errorResponse)) {
        Logger.f(TAG + path + " with reason: " + errorResponse);
      }
      return;
    }
    in = urlConnection.getInputStream();
    response = convertStreamToString(in);
    urlConnection.disconnect();
    if (!TextUtils.isEmpty(response)) Logger.d(TAG + path + " Response: " + response);
  }

  private void addConnectionTimeOut(HttpURLConnection urlConnection, int timeOut) {
    urlConnection.setConnectTimeout(timeOut * 1000);
    urlConnection.setReadTimeout(timeOut * 1000);
  }

  private void addBody(HttpURLConnection connection) throws IOException {

    connection.setDoOutput(true);
    connection.setRequestProperty("Accept-Charset", "UTF-8");
    connection.setRequestProperty("Content-type", "application/json");
    addHeaders(connection);
    addParamsToBody();
    if (jsonBody != null) {
      Logger.d(TAG + path + " addBody: string: " + jsonBody);
    }
    // Write to the connection
    OutputStream output = connection.getOutputStream();
    if (jsonBody != null) {
      output.write(jsonBody.toString().getBytes("UTF-8"));
    }
    output.close();
  }

  private void addIntegrationType() {
    String integrationType = ConfigurationCache.getInstance().getIntegrationType();
    String integrationVersion = ConfigurationCache.getInstance().getIntegrationVersion();
    if (MoEUtils.isEmptyString(integrationType) || MoEUtils.isEmptyString(integrationVersion)) return;
    addParam(MoEConstants.GENERIC_PARAM_V2_KEY_INTEGRATION_TYPE, integrationType);
    addParam(MoEConstants.GENERIC_PARAM_V2_KEY_INTEGRATION_VERSION, integrationVersion);

  }

  private void addHeaders(HttpURLConnection urlConnection) {
    Set<Map.Entry<String, String>> headerMap = headers.entrySet();
    for (Map.Entry<String, String> header : headerMap) {
      Logger.v(
          TAG + path + " addHeaders(): " + header.getKey() + " : " + header.getValue());
      urlConnection.addRequestProperty(header.getKey(), header.getValue());
    }
  }

  private void addParamsToBody() {
    try {
      Set<Map.Entry<String, String>> mapParams = params.entrySet();
      for (Map.Entry<String, String> param : mapParams) {
        try {
          queryParams.put(param.getKey(), param.getValue());
        } catch (Exception e) {
          Logger.e(TAG + path + " addParamsToBody() ", e);
        }
      }
      if (jsonBody == null){
        jsonBody = new JSONObject();
      }
      jsonBody.put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, queryParams);
    } catch (JSONException e) {
      Logger.e(TAG + path + " addParamsToBody() : ", e);
    }
  }

  private String getUrlPath(String url){
    Uri uri = Uri.parse(url);
    return uri.getEncodedPath();
  }
}
