/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.core;

import android.annotation.SuppressLint;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.UserAttribute;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moe.pushlibrary.utils.ReflectionUtils;
import com.moengage.core.mipush.MiPushManager;
import com.moengage.core.model.DataTypes;
import com.moengage.core.model.MoEAttribute;
import com.moengage.core.model.RemoteConfig;
import java.security.MessageDigest;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import static android.Manifest.permission.ACCESS_WIFI_STATE;
import static android.Manifest.permission.READ_PHONE_STATE;
import static android.content.Context.CONNECTIVITY_SERVICE;
import static android.content.Context.TELEPHONY_SERVICE;
import static android.content.pm.PackageManager.FEATURE_TELEPHONY;
import static android.provider.Settings.Secure.ANDROID_ID;
import static android.provider.Settings.Secure.getString;

/**
 * @author MoEngage (abhishek@moenegage.com)
 * @version 5.0
 * @since 1.0
 */
public final class MoEUtils {

  private static final String TAG = "MoEUtils";

  @Nullable static String getOperatorName(Context context) {
    try {
      if (!SdkConfig.getConfig().isCarrierTrackingOptedOut) {
        if (MoEHelperUtils.hasPermission(context, READ_PHONE_STATE) && hasFeature(context,
            FEATURE_TELEPHONY)) {
          TelephonyManager telephonyManager =
              ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE));
          return telephonyManager.getSimOperatorName();
        }
      }
    } catch (Exception ignored) {
    }
    return null;
  }

  public static String convertBundletoJSONString(Bundle newBundle) {
    Set<String> keys = newBundle.keySet();
    JSONObject jsonObject = new JSONObject();
    for (String key : keys) {
      try {
        jsonObject.put(key, newBundle.get(key));
      } catch (Exception e) {
        Logger.f("MoEUtils:convertBundletoJSONString", e);
      }
    }
    return jsonObject.toString();
  }

  public static void showNormalDialogWithOk(String message, Context context) {
    if (null == context) return;
    AlertDialog.Builder builder = new AlertDialog.Builder(context);
    builder.setMessage(message).setPositiveButton("OK", new DialogInterface.OnClickListener() {
      public void onClick(DialogInterface dialog, int id) {
      }
    });
    AlertDialog dialog = builder.create();
    dialog.show();
  }

  public static void showCouponDialog(String message, final String couponcode,
      final Context context) {
    if (null == context) return;
    AlertDialog.Builder builder = new AlertDialog.Builder(context);
    builder.setMessage(message)
        .setPositiveButton("Copy Code", new DialogInterface.OnClickListener() {
          public void onClick(DialogInterface dialog, int id) {
            MoEHelperUtils.copyCouponCodeToClipboard(context, couponcode);
            Properties properties = new Properties();
            properties.addAttribute("coupon_code", couponcode);
            MoEHelper.getInstance(context)
                .trackEvent(MoEConstants.EVENT_ACTION_COUPON_CODE_COPY, properties);
          }
        });
    AlertDialog dialog = builder.create();
    dialog.show();
  }

  /*
   * Checks if user has enabled "Opt out of interest-based ads"
   *
   * @param context An instance of the application {@link Context}
   * @return return {@link AdvertisingIdClient.AdInfo}
   */
  @Nullable public static AdvertisingIdClient.AdInfo getAdvertisementInfo(Context context) {
    try {
      try {
        return AdvertisingIdClient.getAdvertisingIdInfo(context);
      } catch (Exception e) {
        Object adInfo = ReflectionUtils.invokeStatic(
            "com.google.android.gms.ads.identifier.AdvertisingIdClient", "getAdvertisingIdInfo",
            new Class[] { Context.class }, new Object[] { context });
        if (null != adInfo) {
          String advertisingId =
              (String) ReflectionUtils.invokeInstance(adInfo, "getId", null, null);
          boolean isLimit =
              ((Boolean) ReflectionUtils.invokeInstance(adInfo, "isLimitAdTrackingEnabled", null,
                  null)).booleanValue();
          return new AdvertisingIdClient.AdInfo(
              TextUtils.isEmpty(advertisingId) ? null : advertisingId, isLimit ? 1 : 0);
        } else {
          Logger.v(
              "It is advised that you add ----> com.google.android.gms:play-services-ads:7.5.0");
        }
      }
    } catch (Exception e) {
      Logger.f("MoEUtils:getAdvertisementInfo", e);
    }
    return null;
  }

  /**
   * Set the current exponential back off counter whihc needs to be used for
   * the device add call or the GCM registration
   *
   * @param context An instance of the Application Context
   * @param delayInSeconds The exponential backoff counter in seconds
   */
  public static void saveCurrentExponentialCounter(Context context, int delayInSeconds) {
    if (null == context) return;
    SharedPreferences sp = getSharedPrefs(context);
    sp.edit().putInt(MoEConstants.EXPONENTIAL_CONSTANT_MOE, delayInSeconds).apply();
  }

  /**
   * Get the current exponential backoff counter which needs to be used for
   * GCM registration or device add call
   *
   * @param context AN instance of the application context
   * @return the exponential back off counter in seconds
   */
  public static int getCurrentExponentialCounter(Context context) {
    if (null == context) return 1;
    SharedPreferences sp = getSharedPrefs(context);
    return sp.getInt(MoEConstants.EXPONENTIAL_CONSTANT_MOE, 1);
  }

  private static SharedPreferences getSharedPrefs(Context context) {
    if (null == context) return null;
    return context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
  }

  private static final String PREF_NAME = "pref_moe";

  public static boolean isRegistrationScheduled(Context context) {
    SharedPreferences sp = getSharedPrefs(context);
    return sp.getBoolean(MoEConstants.PREF_KEY_DEVICE_ADD_SCHEDULED, false);
  }

  public static void setRegistrationScheduled(Context context, boolean scheduled) {
    SharedPreferences sp = getSharedPrefs(context);
    sp.edit().putBoolean(MoEConstants.PREF_KEY_DEVICE_ADD_SCHEDULED, scheduled).apply();
  }

  /* Returns true if the application has the given feature. */
  public static boolean hasFeature(Context context, String feature) {
    return context.getPackageManager().hasSystemFeature(feature);
  }

  /* Returns true if the string is null, or empty (once trimmed). */
  public static boolean isNullOrEmpty(CharSequence text) {
    return TextUtils.isEmpty(text) || TextUtils.getTrimmedLength(text) == 0;
  }

  /* Returns true if the collection or has a size 0. */
  public static boolean isNullOrEmpty(Collection collection) {
    return collection == null || collection.size() == 0;
  }

  /*
   * @param map
   * @return Returns true if the map is null or empty, false otherwise.
   * */
  public static boolean isNullOrEmpty(Map map) {
    return map == null || map.size() == 0;
  }

  /*
   * @param context Application Context
   * @return Returns the system service for the given string.
   * */
  @SuppressWarnings("unchecked") public static <T> T getSystemService(Context context,
      String serviceConstant) {
    return (T) context.getSystemService(serviceConstant);
  }

  public static String getAndroidID(Context context) {
    if (!SdkConfig.getConfig().isAndroidIdTrackingOptedOut) {
      String androidId = getString(context.getContentResolver(), ANDROID_ID);
      if (!isNullOrEmpty(androidId) && !"9774d56d682e549c".equals(androidId) && !"unknown".equals(
          androidId) && !"000000000000000".equals(androidId)) {
        return androidId;
      }
    }
    return null;
  }

  @SuppressLint("MissingPermission")
  public static String getNetworkType(Context context) {
    try {
      if (MoEHelperUtils.hasPermission(context, ACCESS_WIFI_STATE)) {
        NetworkInfo wifiInfo =
            ((ConnectivityManager) context.getSystemService(CONNECTIVITY_SERVICE)).getNetworkInfo(
                1);
        if ((null != wifiInfo) && (wifiInfo.isConnectedOrConnecting())) {
          return "wifi";
        }
      }
      if (MoEHelperUtils.hasPermission(context, READ_PHONE_STATE) && hasFeature(context,
          FEATURE_TELEPHONY)) {
        TelephonyManager telephonyManager = getSystemService(context, TELEPHONY_SERVICE);
        int type = telephonyManager.getNetworkType();
        switch (type) {
          case TelephonyManager.NETWORK_TYPE_GPRS:
          case TelephonyManager.NETWORK_TYPE_EDGE:
          case TelephonyManager.NETWORK_TYPE_CDMA:
          case TelephonyManager.NETWORK_TYPE_1xRTT:
          case TelephonyManager.NETWORK_TYPE_IDEN: //api<8 : replace by 11
            return "2G";
          case TelephonyManager.NETWORK_TYPE_UMTS:
          case TelephonyManager.NETWORK_TYPE_EVDO_0:
          case TelephonyManager.NETWORK_TYPE_EVDO_A:
          case TelephonyManager.NETWORK_TYPE_HSDPA:
          case TelephonyManager.NETWORK_TYPE_HSUPA:
          case TelephonyManager.NETWORK_TYPE_HSPA:
          case TelephonyManager.NETWORK_TYPE_EVDO_B: //api<9 : replace by 14
          case TelephonyManager.NETWORK_TYPE_EHRPD:  //api<11 : replace by 12
          case TelephonyManager.NETWORK_TYPE_HSPAP:  //api<13 : replace by 15
            return "3G";
          case TelephonyManager.NETWORK_TYPE_LTE:    //api<11 : replace by 13
            return "4G";
          default:
            return "CouldNotDetermine";
        }
      }
    } catch (Exception e) {
      Logger.f("MoEUtils: getNetworkType", e);
    }
    return null;
  }

  /**
   * Call this for tracking activity see and activity stopped only<br>
   * <b>Note : Don't call from UI thread.</b>
   *
   * @param activityState activity state being tracked
   * @param activityName The name of the activity which is changing view state
   * @param context An instance of the application {@link Context}
   */
  static void trackActivityStates(String activityState, String activityName,
      Context context) {
    try {
      PayloadBuilder activityJson = new PayloadBuilder();
      activityJson.putAttrString(MoEConstants.EVENT_ACTIVITY_NAME, activityName);
      MoEHelper.getInstance(context).trackEvent(activityState, activityJson);
    } catch (Exception e) {
      Logger.f("MoEUtils :trackActivityStates", e);
    }
  }


  public static String getAPIRoute() {
    switch (SdkConfig.getConfig().dataRegion) {
      case REGION_EU:
        return MoEConstants.API_V2_EU;
      case REGION_INDIA:
        return MoEConstants.API_V2_INDIA;
      default:
        return MoEConstants.API_GENERAL_V2;
    }
  }

  @Nullable public static Bundle convertMapToBundle(Map<String, String> map) {
    if (map == null) return null;
    Bundle bundle = new Bundle();
    try {
      for (Map.Entry<String, String> entry : map.entrySet()) {
        bundle.putString(entry.getKey(), entry.getValue());
      }
    } catch (Exception e) {
      Logger.f("MoEUtils#convertMapToBundle : Exception", e);
    }
    return bundle;
  }

  @Nullable public static Bundle jsonToBundle(JSONObject json) {
    if (json == null) return null;
    try {
      Bundle bundle = new Bundle();
      Iterator iter = json.keys();
      while (iter.hasNext()) {
        String key = (String) iter.next();
        String value = json.getString(key);
        bundle.putString(key, value);
      }
      return bundle;
    } catch (JSONException e) {
      Logger.f("MoEUtils : jsonToBundle", e);
    }
    return null;
  }

  public static void updateTestDeviceState(Context context) {
    long registeredTime =
        ConfigurationProvider.getInstance(context).getVerificationRegistrationTime();
    if ((registeredTime + (3600 * 1000)) < System.currentTimeMillis()) {
      ConfigurationProvider.getInstance(context).setVerificationRegistration(false);
    }
  }

  @Nullable static UserAttribute getUserAttributePoJo(JSONObject userJSON) {
    UserAttribute userAttribute = null;
    try {
      Iterator jsonKeys = userJSON.keys();
      while (jsonKeys.hasNext()) {
        userAttribute = new UserAttribute();
        userAttribute.userAttributeName = (String) jsonKeys.next();
        userAttribute.userAttributeValue = userJSON.getString(userAttribute.userAttributeName);
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher : getUserAttributePoJo", e);
    }
    return userAttribute;
  }

  static boolean shouldSendUserAttribute(UserAttribute currentUserAttributes,
      UserAttribute savedUserAttributes){
    return currentUserAttributes == null
        || savedUserAttributes == null
        || !savedUserAttributes.equals(currentUserAttributes);
  }

  @Nullable static UserAttribute getSavedUserAttribute(Context context, String userAttributeName){
    return MoEDAO.getInstance(context).getUserAttributeByName(userAttributeName);
  }


  public static String getBatchId(){
    return String.valueOf(MoEUtils.currentTime()) + "-" + UUID.randomUUID().toString();
  }

  public static String getTimeInISO(long timeMillis){
    Date currentDate = new Date();
    currentDate.setTime(timeMillis);
    return ISO8601Utils.format(currentDate);
  }

  public static String getSha1ForString(String inputString){
    try {
      MessageDigest md = MessageDigest.getInstance("SHA-1");
      md.update(inputString.getBytes());
      return bytesToHex(md.digest());
    } catch (Exception e) {
      Logger.f(  "MoEUtils getSha1ForString() : Exception ", e);
    }
    return inputString;
  }

  private final static char[] hexArray = "0123456789ABCDEF".toCharArray();

  public static String bytesToHex(byte[] bytes) {
    char[] hexChars = new char[bytes.length * 2];
    for (int j = 0; j < bytes.length; j++) {
      int v = bytes[j] & 0xFF;
      hexChars[j * 2] = hexArray[v >>> 4];
      hexChars[j * 2 + 1] = hexArray[v & 0x0F];
    }
    return new String(hexChars);
  }

  @Nullable
  public static String getAppId(){
    String appId = SdkConfig.getConfig().appId;
    if (isEmptyString(appId)) return null;
    if (isDebugBuild) {
      appId += "_DEBUG";
    }
    return appId;

  }

  /**
   * Current time in epoch, milliseconds
   */
  public static long currentTime(){
    return System.currentTimeMillis();
  }

  public static String currentISOTime(){
    return getTimeInISO(MoEUtils.currentTime());
  }

  @Nullable @WorkerThread
  public static String getUserAttributeUniqueId(Context context) {
    try {
      MoEAttribute attribute = MoEDAO.getInstance(context).getAttributeByName(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID);
      if (attribute != null) return attribute.getValue();
      return ConfigurationProvider.getInstance(context).getUserAttributeUniqueId();
    } catch (Exception e) {
      Logger.f(TAG + " getUserAttributeUniqueId() : ");
    }
    return null;
  }


  @Nullable
  public static MoEAttribute convertJsonToAttributeObject(JSONObject attributeJson) throws
      JSONException {
    Iterator jsonKeys = attributeJson.keys();
    if (jsonKeys.hasNext()){
      String attributeName = (String) jsonKeys.next();
      Object attributeValue = attributeJson.get(attributeName);
      return new MoEAttribute(
          attributeName,
          attributeValue.toString(),
          MoEUtils.currentTime(),
          getDataTypeForObject(attributeValue).toString());
    }
    return null;
  }

  public static DataTypes getDataTypeForObject(Object value){
    if (value instanceof Integer){
      return DataTypes.INTEGER;
    }
    if (value instanceof Double){
      return DataTypes.DOUBLE;
    }
    if (value instanceof  Long){
      return DataTypes.LONG;
    }
    if (value instanceof Boolean){
      return DataTypes.BOOLEAN;
    }
    if (value instanceof Float){
      return DataTypes.FLOAT;
    }
    return DataTypes.STRING;
  }

  /**
   * Checks whether the given device is manufactured by Xiaomi or not.
   * @param manufacturer Device manufacturer.
   * @return true if it is a Xiaomi device, else false.
   */
  public static boolean isXiaomiDevice(String manufacturer){
    return MoEConstants.MANUFACTURER_XIAOMI.equals(manufacturer);
  }

  public static String deviceManufacturer(){
    return Build.MANUFACTURER;
  }

  public static boolean isDate(String attributeString) {
    try {
      DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", Locale.ENGLISH);
      long epoch = format.parse(attributeString).getTime();
      return epoch > -1;
    } catch (Exception e) {
      Logger.e(TAG + " isDate() : Exception: Could not convert string to date: " + attributeString);
    }
    try {
      DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.sss'Z'", Locale.ENGLISH);
      long epoch = format.parse(attributeString).getTime();
      return epoch > -1;
    } catch (Exception e) {
      Logger.e(TAG + " isDate() : Exception: Could not convert string to date: " + attributeString);
    }
    return false;
  }

  /**
   * Checks if the passed String has atleast one character.
   *
   * @param string String to be checked
   * @return true if string is empty else false.
   */
  public static boolean isEmptyString(String string){
    return string == null || string.trim().length() == 0;
  }

  public static boolean hasKeys(JSONObject jsonObject){
    return jsonObject != null && jsonObject.length() > 0;
  }

  public static int[] MONTH_NUMBERS = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 };

  public static String getDateDataPointFormat() {
    Calendar c = Calendar.getInstance();

    int hours = c.get(Calendar.HOUR_OF_DAY);
    int mins = c.get(Calendar.MINUTE);
    int secs = c.get(Calendar.SECOND);
    int day = c.get(Calendar.DAY_OF_MONTH);
    int month = MONTH_NUMBERS[c.get(Calendar.MONTH)];
    int year = c.get(Calendar.YEAR);

    StringBuilder dateString = new StringBuilder().append(day)
        .append(":")
        .append(month)
        .append(":")
        .append(year)
        .append(":")
        .append(hours)
        .append(":")
        .append(mins)
        .append(":")
        .append(secs);

    return dateString.toString();
  }
  public static boolean canEnableMiPush(RemoteConfig configuration) {
    return isXiaomiDevice(deviceManufacturer())
        && configuration.isPushAmpPlusEnabled
        && !isEmptyString(configuration.miAppId)
        && !isEmptyString(configuration.miAppKey)
        && MiPushManager.getInstance().hasMiPushModule();
  }

  public static Map<String, Object> jsonToMap(JSONObject payloadJson) {
    if (payloadJson == null || payloadJson.length() == 0) return new HashMap<>();
    Map<String, Object> map = new HashMap<>(payloadJson.length());
    Iterator iterator = payloadJson.keys();
    try {
      while (iterator.hasNext()) {
        String key = (String) iterator.next();
        map.put(key, payloadJson.get(key));
      }
    } catch (Exception e) {
      Logger.e(TAG + " jsonToMap() : Exception ", e);
    }
    return map;
  }

  public static long currentSeconds(){
    return System.currentTimeMillis()/1000;
  }

  public static long secondsFromIsoString(String isoString){
    if (!isoString.endsWith("Z")) isoString = isoString + "Z";
    return ISO8601Utils.parse(isoString).getTime()/1000;
  }

  public static String isoStringFromSeconds(long seconds){
    return ISO8601Utils.format(new Date(seconds *1000));
  }

  public static void logJsonArray(String tag, JSONArray jsonArray) {
    try {
      for (int i = 0; i < jsonArray.length(); i++) {
        JSONObject jsonObject = jsonArray.getJSONObject(i);
        Logger.v(tag + "\n" + jsonObject.toString(4));
      }
    } catch (JSONException e) {
      Logger.e(TAG + " logJsonArray() : ", e);
    }
  }

  public static <T> void logCollection(String tag, List<T> list){
    for (T item: list){
      Logger.v(tag + "\n" + item);
    }
  }

  static boolean isDebugBuild(Context context){
    return (0 != (context.getApplicationInfo().flags & ApplicationInfo.FLAG_DEBUGGABLE));
  }

  static boolean isDebugBuild = false;

}
