/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 * All rights reserved.
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.location;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import com.moengage.core.ConfigurationCache;
import com.moengage.core.Logger;
import com.moengage.core.SdkConfig;

/**
 * Location Services handler
 *
 * @author Umang Chamaria
 */
public final class GeoManager {

  private LocationHandler handler;

  private GeoManager() {
    //Intentionally made private
    loadHandler();
  }

  private static GeoManager instance;

  public static GeoManager getInstance() {
    if (instance == null) {
      synchronized (GeoManager.class){
        if (instance == null) instance = new GeoManager();
      }
    }
    return instance;
  }

  private void loadHandler() {
    try {
      handler = (LocationHandler) Class.forName("com.moengage.geofence.LocationHandlerImpl")
          .newInstance();
    } catch (ClassNotFoundException e) {
      Logger.e("Location Handler class Not Found Exception");
    } catch (Exception e) {
      Logger.e("Exception", e);
    }
  }

  @Nullable
  public LocationHandler getHandler(@NonNull Context context) {
    if (context == null) return null;
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return null;
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isGeofenceEnabled()) return null;
    if (SdkConfig.getConfig().isLocationServiceEnabled) return handler;
    return null;
  }

  public void updateFenceAndLocation(Context context) {
    LocationHandler handler = getHandler(context);
    if (handler != null) {
      handler.updateFenceAndLocation(context);
    }
  }

  public void removeGeoFences(Context context) {
    LocationHandler handler = getHandler(context);
    if (handler != null) {
      handler.removeGeoFences(context);
    }
  }

  public void scheduleBackgroundSync(Context context) {
    LocationHandler handler = getHandler(context);
    if (handler != null) {
      handler.scheduleBackgroundSync(context);
    }
  }
}
