package com.moengage.core.events;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.support.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.ConfigurationCache;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEDispatcher;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.json.JSONObject;

/**
 * Central point for all event related processing, i.e. tracking, blacklisting, trigger-events
 *
 * @author Umang Chamaria
 */

public class MoEEventManager {

  private static final String TAG = "MoEEventManager";

  private static MoEEventManager instance = null;

  private ConfigurationProvider configProvider = null;

  private Context context;

  private int eventCounter = 0;

  private List<String> inAppTriggerEvents;

  EventHandler eventHandler;

  private MoEEventManager(Context context) {
    this.context = context;
    configProvider = ConfigurationProvider.getInstance(context);
    getInAppSmartTriggerEvents();
    eventHandler = new EventHandler();
  }

  public static MoEEventManager getInstance(Context context) {
    if (instance == null) {
      synchronized (MoEEventManager.class) {
        if (instance == null) instance = new MoEEventManager(context);
      }
    }
    return instance;
  }

  int getEventCounter() {
    return eventCounter;
  }

  void incrementEventCounter() {
    eventCounter++;
  }

  public void setEventCounter(int counter) {
    eventCounter = counter;
  }

  public void trackEvent(String action, JSONObject attributes) {
    trackEvent(eventHandler.transformToEventModel(action, attributes));
  }

  public void trackEvent(String action, PayloadBuilder attributes){
    trackEvent(new Event(action, attributes.build()));
  }

  public void trackEvent(Event event) {
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    MoEDispatcher.getInstance(context).addTaskToQueue(new TrackEventTask(context, event));
  }

  /**
   * Checks whether a performed action is a smart trigger event or not
   *
   * @param action The action which needs to be checked
   * @return true if the action is a smart trigger event, false otherwise
   */
  boolean isSmartTriggerEvent(final String action) {
    return (inAppTriggerEvents != null && inAppTriggerEvents.contains(action)) || MoEHelperConstants
        .EVENT_APP_INSTALL.equals(action);
  }

  /**
   * Reads the trigger events list from storage and cache it.
   */
  public void getInAppSmartTriggerEvents() {
    try {
      String list = configProvider.getSmartTriggerList();
      if (list == null) {
        Logger.v("MoEEventManager:No smart triggers found");
        return;
      }
      String triggers[] = list.split(MoEConstants.EVENT_SEPARATOR);
      inAppTriggerEvents = new ArrayList<>(triggers.length);
      for (String event : triggers) {
        inAppTriggerEvents.add(event);
      }
    } catch (Exception e) {
      Logger.e("MoEEventManager: getInAppSmartTriggerEvents()", e);
    }
  }


  private boolean isFlushEvent(String eventName) {
    Set<String> flushEvents =
        ConfigurationCache.getInstance().getRemoteConfiguration().getFlushEventList();
    return flushEvents!= null && flushEvents.contains(eventName);
  }

  void flushIfRequired(@NonNull Event event) {
    if (event.eventName != null && isFlushEvent(event.eventName)) {
      Logger.v(TAG + " flushIfRequired() flush event : " + event.eventName);
      MoEHelper.getInstance(context).syncInteractionDataNow();
    }
  }

  @WorkerThread @RestrictTo(Scope.LIBRARY_GROUP)
  public void writeDataPointToStorage(Event event) {
    MoEDAO.getInstance(context).addEvent(event);
  }

}
