/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 * All rights reserved.
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.core;

import android.content.Context;
import android.net.Uri;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.text.TextUtils;
import com.moe.pushlibrary.exceptions.SDKNotInitializedException;
import com.moengage.core.MoERestClient.RequestMethod;
import com.moengage.core.rest.Request;
import com.moengage.core.rest.RequestBuilder;
import com.moengage.core.rest.RequestBuilder.RequestType;
import com.moengage.core.rest.Response;
import com.moengage.core.rest.RestClient;
import com.moengage.core.utils.RestUtils;
import com.moengage.inapp.InAppController;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.HashMap;
import javax.net.ssl.HttpsURLConnection;
import org.json.JSONObject;

/**
 * This class is responsible for all interactions with the MoE servers.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 5.0
 * @since 5.0
 */
public final class APIManager {

  private APIManager() {
    // Constructor intentionally made private. Utility class cannot be
    // instantiated
  }

  /*
   * @param con Instance of the Application {@link Context}
   * @param paramsMap The last updated time for the local inapp cache
   * @param API API to trigger
   * @param requestBody Request Body
   * @return true if the fetch was successful
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  @Nullable public static String fetchInAppCampaigns(Context con, String API,
      HashMap<String, String> paramsMap, JSONObject requestBody) {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()
          || !ConfigurationCache.getInstance()
          .getRemoteConfiguration().isInAppEnabled()) {
        return null;
      }
      MoERestClient client = new MoERestClient(API, con);
      client.addParam(paramsMap);
      if (requestBody != null) {
        client.addBody(requestBody);
      }
      client.execute(RequestMethod.POST);
      Logger.v("APIManager: Processing InApp Response - will parse and save data");
      if (!isHttpStatusOk(client.getResponseCode())) {
        InAppController.getInstance().trackAPIFailure(InAppController.SYNC_API_FAILURE);
        return null;
      }
      //update last updated time
      ConfigurationProvider.getInstance(con).setLastInappUpdateTime(System.currentTimeMillis());
      if (!TextUtils.isEmpty(client.getResponse())) {
        Logger.v("APIManager: fetchInAppCampaingn" + client.getResponse());
        return client.getResponse();
      }
    } catch (Exception e) {
      Logger.f("APIManager: fetchInAppCampaigns", e);
      InAppController.getInstance().trackAPIFailure(InAppController.SYNC_API_FAILURE);
    }
    return null;
  }

  /*
   * @param con Instance of the Application {@link Context}
   * @return returns the InAppMessage which needs to be shown for this auto
   * trigger event. If no in app is received then returns null
   */
  @Nullable static String logASmartEvent(Context con, String API, HashMap<String, String> paramsMap,
      JSONObject requestBody) {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()
          || !ConfigurationCache.getInstance()
          .getRemoteConfiguration().isInAppEnabled()) {
        return null;
      }
      MoERestClient client = new MoERestClient(API, con);
      client.addParam(paramsMap);
      if (requestBody != null) {
        client.addBody(requestBody);
      }
      client.execute(RequestMethod.POST);
      Logger.v("APIManager: Processing Smart event response");
      if (!isHttpStatusOk(client.getResponseCode())) {
        InAppController.getInstance().trackAPIFailure(InAppController.SMART_API_FAILURE);
        return null;
      }
      return client.getResponse();
    } catch (Exception e) {
      Logger.f("APIManager: logASmartEvent", e);
      InAppController.getInstance().trackAPIFailure(InAppController.SMART_API_FAILURE);
    }
    return null;
  }

  @Nullable static String fetchSingleInApp(Context context, String API,
      HashMap<String, String> requestParams) {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()
          || !ConfigurationCache
          .getInstance().getRemoteConfiguration().isInAppEnabled()) {
        return null;
      }
      MoERestClient client = new MoERestClient(API, context);
      client.addParam(requestParams);
      client.execute(RequestMethod.POST);
      if (!isHttpStatusOk(client.getResponseCode())) {
        InAppController.getInstance().trackAPIFailure(InAppController.SINGLE_API_FAILURE);
        return null;
      }
      return client.getResponse();
    } catch (Exception e) {
      Logger.f("APIManager: fetchInAppCampaigns", e);
      InAppController.getInstance().trackAPIFailure(InAppController.SINGLE_API_FAILURE);
    }
    return null;
  }

  static void uploadLogsToLogEntries(String api, JSONObject jsonObject) {
    try {
      URL url = new URL(api);
      HttpsURLConnection httpsURLConnection = (HttpsURLConnection) url.openConnection();
      httpsURLConnection.setDoOutput(true);
      httpsURLConnection.setRequestProperty("Accept-Charset", "UTF-8");
      httpsURLConnection.setRequestProperty("Content-type", "application/json");
      OutputStream output = httpsURLConnection.getOutputStream();
      output.write(jsonObject.toString().getBytes("UTF-8"));
      output.close();
      Logger.v(
          "API Manager : uploadLogsToLogEntries : response" + httpsURLConnection.getResponseCode());
    } catch (Exception e) {

    }
  }

  @Nullable
  static APIResponse registerUnregisterDeviceForIntegrationVerification(Context context, String API,
      HashMap<String, String> paramsMap) {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return null;
      MoERestClient client = new MoERestClient(API, context);
      if (paramsMap != null) {
        client.addParam(paramsMap);
      }
      client.execute(RequestMethod.POST);
      return new APIResponse(client.getResponse(), client.getResponseCode());
    } catch (Exception e) {
      Logger.f("APIManager : registerUnregisterDeviceForIntegrationVerification :", e);
    }
    return null;
  }

  @Nullable
  public static APIResponse deviceTriggerSyncRequest(Context context, String API, HashMap<String,
      String> paramsMap, JSONObject requestBody) {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isRealTimeTriggerEnabled()
          || !ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) {
        return null;
      }
      MoERestClient client = new MoERestClient(API, context);
      if (paramsMap != null) {
        client.addParam(paramsMap);
      }
      if (requestBody != null) {
        client.addBody(requestBody);
      }
      client.execute(RequestMethod.POST);
      return new APIResponse(client.getResponse(), client.getResponseCode());
    } catch (Exception e) {
      Logger.f("MoERestClient: deviceTriggerSyncRequest() : Exception ", e);
    }
    return null;
  }

  /**
   * Check if HTTP call was successful
   *
   * @param responseCode The HTTP response code
   * @return true if http status code is 200
   */
  static boolean isHttpStatusOk(int responseCode) {
    return 200 == responseCode;
  }


  @Nullable static Response reportAdd(String appId, String path, String requestId, JSONObject body)
      throws IOException, SDKNotInitializedException {
    Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
        .appendEncodedPath(path);

    RequestBuilder requestBuilder =
        RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST, appId)
            .addHeader(MoEConstants.REQUEST_HEADER_REQUEST_ID, requestId)
            .addBody(body);

    RestClient restClient = new RestClient(requestBuilder.build());
    return restClient.executeRequest();
  }

  @Nullable public static Response configApi(String appId, JSONObject requestBody)
      throws IOException, SDKNotInitializedException {
    Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
        .appendEncodedPath(MoEConstants.API_ENDPOINT_CONFIG_API)
        .appendEncodedPath(appId);

    Request request = RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST, appId)
        .addBody(requestBody)
        .build();

    RestClient restClient = new RestClient(request);
    return restClient.executeRequest();
  }
}
