package com.moe.pushlibrary.providers;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import com.moe.pushlibrary.providers.MoEDataContract.AttributeCacheEntity;
import com.moe.pushlibrary.providers.MoEDataContract.BatchDataEntity;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DTEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DatapointEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppMessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.UserAttributeEntity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.model.DataTypes;
import java.util.ArrayList;
import org.json.JSONObject;

class MoEDatabaseHelper extends SQLiteOpenHelper {

  private static final String TAG = "DatabaseHelper";

  /**
   * The database name
   */
  private static final String DATABASE_NAME = "MOEInteractions";
  /**
   * The current database version
   */
  private static final int DATABASE_VERSION = 15;

  private Context context;

  /**
   * @param context Application Context
   */
  public MoEDatabaseHelper(Context context) {
    super(context, DATABASE_NAME, null, DATABASE_VERSION);
    this.context = context;
  }

  /**
   * Table containing the Inbox messages
   */
  static final String TABLE_NAME_MSGS = "MESSAGES";
  /**
   * Table containing events
   */
  static final String TABLE_NAME_DATAPOINTS = "DATAPOINTS";
  /**
   * Table containing in app messages
   */
  static final String TABLE_NAME_INAPPS = "INAPPMSG";
  /**
   * Old table containing in app messages
   */
  static final String TABLE_NAME_OLD_INAPPS = "INAPPS";
  /**
   * Table to save user attributes
   */
  static final String TABLE_NAME_USER_ATTRIBUTES = "USERATTRIBUTES";
  /**
   * Table to save campaign list
   */
  static final String TABLE_NAME_CAMPAIGN_LIST = "CAMPAIGNLIST";
  /**
   * Table to save batched data
   */
  static final String TABLE_NAME_BATCH_DATA = "BATCH_DATA";
  /**
   * Table to save device triggers
   */
  static final String TABLE_NAME_DEVICE_TRIGGERS = "DEVICE_TRIGGERS";

  static final String TABLE_NAME_ATTRIBUTE_CACHE = "ATTRIBUTE_CACHE";

  @Override public void onCreate(SQLiteDatabase db) {
    // SQL statement to create interactions table
    String DDL_DATAPOINTS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_DATAPOINTS
        + " ( "
        + DatapointEntity._ID
        + " INTEGER PRIMARY KEY, "
        + DatapointEntity.GTIME
        + " INTEGER, "
        + DatapointEntity.DETAILS
        + " TEXT ); ";

    createMessagesTable(db);

    createInAppTable(db);
    createUserAttributeTable(db);
    createCampaignListTable(db);
    db.execSQL(DDL_DATAPOINTS);
    createBatchDataTable(db);
    createDeviceTriggerTable(db);
    createAttributeCacheTableIfRequired(db);
    Logger.i(": Database created");
  }

  private boolean tableExists(SQLiteDatabase db, final String tableName) {
    Cursor cursor = null;
    try {
      cursor = db.rawQuery(
          "select DISTINCT tbl_name from sqlite_master where tbl_name = '" + tableName + "'",
          null);
      return (cursor != null && cursor.getCount() > 0);
    } finally {
      if (null != cursor) {
        cursor.close();
      }
    }
  }

  public boolean isTableExists(SQLiteDatabase db, String tableName) {
    int count = 0;
    try {
      if (tableName == null || db == null || !db.isOpen()) {
        return false;
      }
      Cursor cursor = db.rawQuery("SELECT COUNT(*) FROM sqlite_master WHERE type = ? AND name = ?",
          new String[] { "table", tableName });
      if (!cursor.moveToFirst()) {
        return false;
      }
      count = cursor.getInt(0);
      cursor.close();
    } catch (Exception e) {
      Logger.f(": isTableExists Exception " + e.toString());
    }
    return count > 0;
  }

  /*
   * (non-Javadoc)
   *
   * @see
   * android.database.sqlite.SQLiteOpenHelper#onUpgrade(android.database
   * .sqlite.SQLiteDatabase, int, int)
   */
  @Override public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
    Logger.v(": Provider upgrading DB ");

    int upgradeTo = oldVersion + 1;
    while (upgradeTo <= newVersion) {
      switch (upgradeTo) {
        case 3:
          db.beginTransaction();
          try {
            createDataPointsTable(db);
            createMessagesTable(db);
            //creating the in-appv2 table here as the old table was dropped.
            createInAppTable(db);
            portDataFromv2(db);
            db.execSQL("DROP TABLE IF EXISTS moeints");
            db.execSQL("DROP TABLE IF EXISTS moemsgs");
            db.execSQL("DROP TABLE IF EXISTS moeinappmsgs");
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.f(": failed to port data. FROM V2.", e);
          } finally {
            db.endTransaction();
          }
          break;
        case 4:
          // DBv4 Never released
          break;
        case 5:
          db.beginTransaction();
          try {
            db.execSQL("DROP TABLE IF EXISTS CHATS");
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.f(": failed to port data.. FOR UBOX", e);
          } finally {
            db.endTransaction();
          }
          break;
        case 6:
          db.beginTransaction();
          try {
            if (isTableExists(db, TABLE_NAME_OLD_INAPPS)) {
              String ALTER_INAPP =
                  " ALTER TABLE " + TABLE_NAME_OLD_INAPPS + " ADD COLUMN TYPE INTEGER";
              db.execSQL(ALTER_INAPP);
            }
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.f(": failed to add column " + TABLE_NAME_OLD_INAPPS, e);
          } finally {
            db.endTransaction();
          }
          break;
        case 7:
          db.beginTransaction();
          try {
            createDataPointsTable(db);
            populateDatapoints(db);
            db.execSQL("DROP TABLE IF EXISTS EVENTS");
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.f(": failed to populate Datapoints ", e);
          } finally {
            db.endTransaction();
          }
          break;
        case 8:
          db.beginTransaction();
          try {
            if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
              String ALTER_INBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
                  MessageEntity.MSG_TAG + " TEXT";
              db.execSQL(ALTER_INBOX);
            }
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.f(": failed to add columns to " + TABLE_NAME_MSGS, e);
          } finally {
            db.endTransaction();
          }
          break;
        case 9:
          db.execSQL("DROP TABLE IF EXISTS INAPPS");
          createInAppTable(db);
          //this check is added for the mess up made by big basket.
          if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
            String ALTER_INBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
                MessageEntity.MSG_TAG + " TEXT";
            db.execSQL(ALTER_INBOX);
          }
          break;
        case 10:
          createUserAttributeTable(db);
          break;
        case 11:
          createCampaignListTable(db);
          break;
        case 12:
          createBatchDataTable(db);
          break;
        case 13:
          createDeviceTriggerTable(db);
          break;
        case 14:
          createAttributeCacheTableIfRequired(db);
          portUserAttributeUniqueId(db);
          break;
        case 15:
          addCampaignIdToMessageTable(db);
          break;
        default:
          Logger.v("Failed to upgrade from DB version" + oldVersion + "to DB version" +
              newVersion);
      }
      upgradeTo++;
    }
    ConfigurationProvider.getInstance(context).setNewDBVersion(DATABASE_VERSION);
    Logger.v(": Database Upgraded");
  }

  private void createDataPointsTable(SQLiteDatabase db) {
    String DDL_DATAPOINTS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_DATAPOINTS
        + " ( "
        + DatapointEntity._ID
        + " INTEGER PRIMARY KEY, "
        + DatapointEntity.GTIME
        + " INTEGER, "
        + DatapointEntity.DETAILS
        + " TEXT ); ";
    db.execSQL(DDL_DATAPOINTS);
  }

  private void createMessagesTable(SQLiteDatabase db) {
    String DDL_MSGS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_MSGS
        + " ( "
        + MessageEntity._ID
        + " INTEGER PRIMARY KEY, "
        + MessageEntity.MSG_DETAILS
        + " TEXT, "
        + MessageEntity.MSG_CLICKED
        + " INTEGER DEFAULT 0, "
        + MessageEntity.MSG_TTL
        + " INTEGER, "
        + MessageEntity.GTIME
        + " INTEGER, "
        + MessageEntity.MSG_TAG
        + " TEXT, "
        + MessageEntity.CAMPAIGN_ID
        + " TEXT "
        + ")";
    db.execSQL(DDL_MSGS);
  }

  private void createInAppTable(SQLiteDatabase db) {
    String DDL_INAPP = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_INAPPS
        + " ( "
        + InAppMessageEntity._ID
        + " INTEGER PRIMARY KEY, "
        + InAppMessageEntity.GTIME
        + " INTEGER, "
        + InAppMessageEntity.MSG_CAMPAIGN_ID
        + " TEXT, "
        + InAppMessageEntity.MSG_ALIGN_TYPE
        + " TEXT, "
        + InAppMessageEntity.MSG_INAPP_TYPE
        + " TEXT, "
        + InAppMessageEntity.MSG_TTL
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_MIN_DELAY
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_MAX_TIMES
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_SHOWN_COUNT
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_PERSISTENT
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_PRIORITY
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_CONTEXT
        + " TEXT, "
        + InAppMessageEntity.MSG_LAST_SHOWN
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_IS_CLICKED
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_HAS_ERRORS
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_AUTODISMISS_TIME
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_CANCELABLE
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_CONTENT
        + " TEXT, "
        + InAppMessageEntity.MSG_SHOW_ONLY_IN
        + " TEXT, "
        + InAppMessageEntity.MSG_STATUS
        + " TEXT, "
        + InAppMessageEntity.MSG_CONTAINER_STYLE
        + " TEXT );";
    db.execSQL(DDL_INAPP);
  }

  public void addMSGTagIfRequiredInbox(SQLiteDatabase db) {
    if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
      Logger.v(" : addMSGTagIfRequiredInbox : updating inbox table");
      String ALTER_UBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
          MessageEntity.MSG_TAG + " TEXT";
      db.execSQL(ALTER_UBOX);
    }
  }

  public void addUserAttributesTableIfRequired(SQLiteDatabase db) {
    if (!tableExists(db, TABLE_NAME_USER_ATTRIBUTES)) {
      Logger.v(" : DatabaseHelper : addUserAttributesTableIfRequired creating "
          + "missing  user attribute table");
      createUserAttributeTable(db);
    } else {
      Logger.v(" : DatabaseHelper : addUserAttributesTableIfRequired "
          + "user attribute table already present");
    }
  }

  public void addCampaignListTableIfRequired(SQLiteDatabase db) {
    if (!tableExists(db, TABLE_NAME_CAMPAIGN_LIST)) {
      Logger.v(" : DatabaseHelper : addCampaignListTableIfRequired creating "
          + "missing  campaign list table");
      createUserAttributeTable(db);
    } else {
      Logger.v(" : DatabaseHelper : addCampaignListTableIfRequired "
          + "campaign list table already present");
    }
  }

  private void createUserAttributeTable(SQLiteDatabase db) {
    String DDL_USER_ATTRIBUTES = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_USER_ATTRIBUTES
        + " ( "
        + UserAttributeEntity._ID
        + " INTEGER PRIMARY KEY, "
        + UserAttributeEntity.ATTRIBUTE_NAME
        + " TEXT, "
        + UserAttributeEntity.ATTRIBUTE_VALUE
        + " TEXT"
        + " ); ";
    db.execSQL(DDL_USER_ATTRIBUTES);
  }

  private void createCampaignListTable(SQLiteDatabase db) {
    String DDL_CAMPAIGN_LIST = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_CAMPAIGN_LIST
        + " ( "
        + CampaignListEntity._ID
        + " INTEGER PRIMARY KEY, "
        + CampaignListEntity.CAMPAIGN_ID
        + " TEXT, "
        + CampaignListEntity.CAMPAIGN_ID_TTL
        + " INTEGER"
        + " );";
    db.execSQL(DDL_CAMPAIGN_LIST);
  }

  private void createBatchDataTable(SQLiteDatabase db) {
    String DDL_BATCH_DATA = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_BATCH_DATA
        + " ( "
        + BatchDataEntity._ID
        + " INTEGER PRIMARY KEY, "
        + BatchDataEntity.BATCHED_DATA
        + " TEXT"
        + " );";
    db.execSQL(DDL_BATCH_DATA);
  }

  void addBatchDataTableIfRequired(SQLiteDatabase db) {
    if (!tableExists(db, TABLE_NAME_BATCH_DATA)) {
      Logger.v(" : DatabaseHelper : addBatchDataTableIfRequired creating "
          + "missing  campaign list table");
      createBatchDataTable(db);
    } else {
      Logger.v(": DatabaseHelper : addBatchDataTableIfRequired batch data table is "
          + "already present");
    }
  }

  void createDeviceTriggerTable(SQLiteDatabase db) {
    String DDL_DEVICE_TRIGGERS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_DEVICE_TRIGGERS
        + " ( "
        + DTEntity._ID
        + " INTEGER PRIMARY KEY, "
        + DTEntity.CAMPAIGN_ID
        + " TEXT, "
        + DTEntity.TRIGGER_EVENT_NAME
        + " TEXT, "
        + DTEntity.PAYLOAD
        + " TEXT, "
        + DTEntity.CAMPAIGN_PAYLOAD
        + " TEXT, "
        + DTEntity.CAMPAIGN_TYPE
        + " TEXT, "
        + DTEntity.MAX_COUNT
        + " INTEGER DEFAULT 0, "
        + DTEntity.MINIMUM_DELAY
        + " INTEGER DEFAULT 0, "
        + DTEntity.SHOULD_SHOW_OFFLINE
        + " INTEGER DEFAULT 0, "
        + DTEntity.MAX_SYNC_DELAY_TIME
        + " INTEGER DEFAULT 0, "
        + DTEntity.EXPIRY_TIME
        + " INTEGER, "
        + DTEntity.PRIORITY
        + " INTEGER DEFAULT 3, "
        + DTEntity.LAST_SHOW_TIME
        + " INTEGER DEFAULT 0, "
        + DTEntity.SHOW_COUNT
        + " INTEGER DEFAULT 0, "
        + DTEntity.LAST_UPDATED_TIME
        + " INTEGER DEFAULT 0, "
        + DTEntity.STATUS
        + " TEXT, "
        + DTEntity.SHOULD_IGNORE_DND
        + " INTEGER DEFAULT 0, "
        + DTEntity.DELAY_BEFORE_SHOWING_NOTIFICATION
        + " INTEGER DEFAULT 0 "
        + " ) ";
    db.execSQL(DDL_DEVICE_TRIGGERS);
  }

  void addDeviceTriggerTableIfRequired(SQLiteDatabase db) {
    if (!tableExists(db, TABLE_NAME_DEVICE_TRIGGERS)) {
      Logger.v(" : DatabaseHelper : addDeviceTriggerTableIfRequired creating "
          + "missing  device trigger table");
      createDeviceTriggerTable(db);
    } else {
      Logger.v(
          ": DatabaseHelper : addDeviceTriggerTableIfRequired device trigger table is "
              + "already present");
    }
  }

  void createAttributeCacheTableIfRequired(SQLiteDatabase database) {
    String DDL_ATTRIBUTE_CACHE = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_ATTRIBUTE_CACHE
        + " ( "
        + AttributeCacheEntity._ID
        + " INTEGER PRIMARY KEY, "
        + AttributeCacheEntity.ATTRIBUTE_NAME
        + " TEXT, "
        + AttributeCacheEntity.ATTRIBUTE_VALUE
        + " TEXT, "
        + AttributeCacheEntity.LAST_TRACKED_TIME
        + " INTEGER DEFAULT 0, "
        + AttributeCacheEntity.DATA_TYPE
        + " TEXT "
        + " ) ";
    database.execSQL(DDL_ATTRIBUTE_CACHE);
  }

  void addAttributeCacheTableIfRequired(SQLiteDatabase db) {
    if (!tableExists(db, TABLE_NAME_ATTRIBUTE_CACHE)) {
      Logger.v(" : DatabaseHelper : addAttributeCacheTableIfRequired() creating "
          + "missing  attribute cache table");
      createAttributeCacheTableIfRequired(db);
    } else {
      Logger.v(": DatabaseHelper : addAttributeCacheTableIfRequired() attribute cache"
          + " table is already present");
    }
  }

  void addCampaignIdToMessageTable(SQLiteDatabase db) {
    if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.CAMPAIGN_ID, db)) {
      Logger.v(TAG + " addMSGTagIfRequiredInbox()  updating inbox table");
      String ALTER_MESSAGES_TABLE = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
          MessageEntity.CAMPAIGN_ID + " TEXT";
      db.execSQL(ALTER_MESSAGES_TABLE);
    }
  }

  private void portUserAttributeUniqueId(SQLiteDatabase db) {
    try {
      Logger.v(TAG + " portUserAttributeUniqueId(): Will try to port USER_ATTRIBUTE_UNIQUE_ID "
          + "to another storage.");
      if (!tableExists(db, TABLE_NAME_USER_ATTRIBUTES)) {
        Logger.v(TAG + " portUserAttributeUniqueId() : User attribute table does not exist");
        return;
      }
      // fetch value from table
      db.beginTransaction();
      String QUERY_GET_USER_ATTRIBUTE_UNIQUE_ID = "SELECT "
          + UserAttributeEntity.ATTRIBUTE_NAME + ", "
          + UserAttributeEntity.ATTRIBUTE_VALUE + " "
          + "FROM " + TABLE_NAME_USER_ATTRIBUTES
          + " WHERE " + UserAttributeEntity.ATTRIBUTE_NAME + " = ?";
      Cursor cursor = db.rawQuery(QUERY_GET_USER_ATTRIBUTE_UNIQUE_ID,
          new String[] { MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID });
      if (cursor == null) {
        Logger.v(TAG + " portUserAttributeUniqueId(): USER_ATTRIBUTE_UNIQUE_ID not present. "
            + "Cursor is null");
        return;
      }
      if (!cursor.moveToFirst()) {
        Logger.v(TAG + " portUserAttributeUniqueId(): USER_ATTRIBUTE_UNIQUE_ID not present. "
            + "Cursor does not have a 1st element.");
        return;
      }
      String uniqueId = cursor.getString(1);
      cursor.close();
      // copy existing value to shared preference
      ConfigurationProvider.getInstance(context).saveUserAttributeUniqueId(uniqueId);
      // copy existing value to attribute cache table
      if (!tableExists(db, TABLE_NAME_ATTRIBUTE_CACHE)) {
        Logger.v(TAG + " portUserAttributeUniqueId() : Attribute Cache does not exist. Cannot "
            + "port data");
        return;
      }
      ContentValues contentValue = new ContentValues();
      contentValue.put(AttributeCacheEntity.ATTRIBUTE_NAME,
          MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID);
      contentValue.put(AttributeCacheEntity.ATTRIBUTE_VALUE, uniqueId);
      contentValue.put(AttributeCacheEntity.LAST_TRACKED_TIME, 0);
      contentValue.put(AttributeCacheEntity.DATA_TYPE, DataTypes.STRING.toString());
      db.insert(TABLE_NAME_ATTRIBUTE_CACHE, null, contentValue);
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.f(TAG + " portUserAttributeUniqueId(): Exception ", e);
    } finally {
      db.endTransaction();
    }
  }

  private void portDataFromv2(SQLiteDatabase db) {
    try {
      db.beginTransaction();

      //no need to port events data might cause issues

      String QUERY_GET_MSGS = "SELECT * FROM moemsgs";
      Cursor cursor = db.rawQuery(QUERY_GET_MSGS, null);
      if (null != cursor && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();
          values.put(MessageEntity.MSG_DETAILS, cursor.getString(1));
          values.put(MessageEntity.MSG_CLICKED, cursor.getInt(2));
          values.put(MessageEntity.MSG_TTL, Long.parseLong(cursor.getString(3)));
          values.put(MessageEntity.GTIME, Long.parseLong(cursor.getString(4)));
          long row = db.insert(TABLE_NAME_MSGS, null, values);
          Logger.v(":onUpgrade: Porting message data: " + row);
        } while (cursor.moveToNext());
        cursor.close();
        cursor = null;
      }

      //removed porting code for inapps
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.f(": portDatafromv2", e);
    } finally {
      if (null != db) {
        db.endTransaction();
      }
    }
  }

  private void populateDatapoints(SQLiteDatabase db) {
    if (null == db) {
      return;
    }
    try {
      Logger.d("Started porting DATA - FOR DATAPOINTS");
      db.beginTransaction();
      String QUERY_GET_EVENTS = "SELECT _id, action, attrs, gtime, ltime FROM EVENTS";
      Cursor cursor = db.rawQuery(QUERY_GET_EVENTS, null);
      if (null != cursor && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();
          JSONObject details = MoEHelperUtils.getDatapointJSON(cursor.getString(1),
              new JSONObject(cursor.getString(2)), Long.toString(cursor.getLong(3)),
              cursor.getString(4));
          if (null != details) {
            values.put(DatapointEntity.DETAILS, details.toString());
            values.put(DatapointEntity.GTIME, cursor.getLong(3));
            long row = db.insert(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS, null, values);
            Logger.v(":onUpgrade: Porting event data: " + row);
          }
        } while (cursor.moveToNext());
        cursor.close();
      }
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.f(": populateDatapoints", e);
    } finally {
      db.endTransaction();
    }
  }

  public boolean isFieldExist(String tableName, String fieldName, SQLiteDatabase db) {
    boolean isExist = false;
    Cursor res = db.rawQuery("PRAGMA table_info(" + tableName + ")", null);
    ArrayList<String> columnNameList = new ArrayList<>();
    try {
      if (res != null && res.moveToFirst()) {
        do {
          String columnName = res.getString(res.getColumnIndex("name"));
          columnNameList.add(columnName);
        } while (res.moveToNext());
      }
    } finally {
      if (res != null) {
        res.close();
      }
    }

    if (columnNameList.contains(fieldName)) {
      isExist = true;
    }
    return isExist;
  }
}