package com.moengage.core.analytics;

import android.app.Activity;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.ConfigurationCache;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoECoreEvaluator;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.model.TrafficSource;
import com.moengage.core.model.UserSession;
import java.util.UUID;

/**
 * @author Umang Chamaria
 * Date: 2019-05-21
 */
public class AnalyticsHelper {

  private static final String TAG = "AnalyticsHelper";

  private static AnalyticsHelper instance = null;

  private UserSession session;

  private MoECoreEvaluator evaluator;

  private SourceProcessor sourceProcessor;

  private AnalyticsHelper(Context context) {
    session = MoEDAO.getInstance(context).getLastSavedSession();
    evaluator = MoEDispatcher.getInstance(context)
        .getCoreEvaluator();
    sourceProcessor = new SourceProcessor();
  }

  public static AnalyticsHelper getInstance(Context context) {
    if (instance == null) {
      synchronized (AnalyticsHelper.class) {
        if (instance == null) instance = new AnalyticsHelper(context);
      }
    }
    return instance;
  }

  public void onEventTracked(Event event, Context context) {
    try {
      Logger.v(TAG
          + " onEventTracked() : Will update last interaction time if required. Event: "
          + event.details);
      // check if event is non-interactive
      if (!event.isInteractiveEvent) {
        Logger.v(
            TAG + " onEventTracked() : No operation required. Tracked event is non-interactive");
        return;
      }
      // user attribute tracked no action required.
      if (event.eventName.equals(MoEConstants.EVENT_ACTION_USER_ATTRIBUTE)) {
        Logger.v(TAG + " updateSession() : Need not update session info since user attribute is "
            + "tracked.");
        return;
      }
      // app in foreground no action required.
      if (MoEHelper.isAppInForeground()) {
        Logger.v(TAG + " updateSession() : App is in foreground no action required.");
        updateLastInteractionTime(MoEUtils.currentTime());
        return;
      }
      // no sessions created till now
      if (session == null) {
        Logger.v(TAG + " onEventTracked() : No previous session. Will create a new session");
        batchPreviousDataAndCreateNewSession(context, null, true);
        return;
      }
      // existing session expired create a new session.
      if (evaluator.hasSessionExpired(session.getLastInteractionTime(),
          ConfigurationCache.getInstance().getRemoteConfiguration().getSessionInActiveTime(),
          MoEUtils.currentTime())) {
        Logger.v(TAG + " onEventTracked() : Session has expired.");
        batchPreviousDataAndCreateNewSession(context, null, true);
        return;
      }
      //update last interaction
      updateLastInteractionTime(MoEUtils.currentTime());
    } catch (Exception e) {
      Logger.e(TAG + " onEventTracked() : Exception: ", e);
    }
  }

  void updateLastInteractionTime(long time) {
    if (session != null) {
      session.setLastInteractionTime(time);
    }
  }

  // update session if required
  @WorkerThread public void onAppOpen(Activity activity) {
    if (session != null) {
      Logger.v(TAG + " onAppOpen() : Current Session " + session.toString());
    }
    updateUserSessionIfRequired(activity);
  }

  public void onNotificationClicked(Context context, Bundle pushPayload,
      boolean isAppInBackground) {
    TrafficSource source = null;
    if (pushPayload.containsKey(MoEHelperConstants.GCM_EXTRA_WEB_URL)) {
      String urlString = pushPayload.getString(MoEHelperConstants.GCM_EXTRA_WEB_URL);
      if (!MoEUtils.isEmptyString(urlString)) {
        Uri uri = Uri.parse(urlString);
        source = sourceProcessor.getTrafficSourceFromUrl(uri,
            ConfigurationCache.getInstance().getRemoteConfiguration().getSourceExtraIdentifier());
      }
    }
    if (source == null || TrafficSource.isEmpty(source)) {
      source = sourceProcessor.getTrafficSourceFromExtras(pushPayload,
          ConfigurationCache.getInstance().getRemoteConfiguration().getSourceExtraIdentifier());
    }
    updateSessionIfRequired(context, source, isAppInBackground);
  }

  private void updateUserSessionIfRequired(Activity activity) {
    try {
      Context context = activity.getApplicationContext();
      TrafficSource currentSource = sourceProcessor.getTrafficSourceFromActivity(activity,
          ConfigurationCache.getInstance().getRemoteConfiguration().getSourceExtraIdentifier());
      updateSessionIfRequired(context, currentSource, MoEHelper.isAppInBackground());
    } catch (Exception e) {
      Logger.e(TAG + " onAppOpen() : Exception: ", e);
    }
  }

  private void updateSessionIfRequired(Context context, TrafficSource currentSource,
      boolean isAppInBackground) {
    if (session == null) {
      Logger.v(TAG
          + " updateSessionIfRequired() : No saved session for user will create a new session.");
      batchPreviousDataAndCreateNewSession(context, currentSource,
          MoEHelper.isAppInBackground());
      return;
    }
    // previous session expired
    boolean hasSessionExpired =
        evaluator.hasSessionExpired(session.getLastInteractionTime(),
            ConfigurationCache.getInstance().getRemoteConfiguration().getSessionInActiveTime(),
            MoEUtils.currentTime());
    if (hasSessionExpired) {
      Logger.v(TAG
          + " updateSessionIfRequired() : Previous session has expired. Will create a new session"
          + ".");
      batchPreviousDataAndCreateNewSession(context, currentSource,
          MoEHelper.isAppInBackground());
      return;
    }
    // source changed
    TrafficSource savedSource = session.getSource();
    boolean hasSourceChanged = evaluator.hasSourceChanged(savedSource, currentSource);
    if (hasSourceChanged) {
      Logger.v(TAG + " updateSessionIfRequired() : Source changed. will create a new session");
      batchPreviousDataAndCreateNewSession(context, currentSource,
          isAppInBackground);
    }
  }

  @WorkerThread private void batchPreviousDataAndCreateNewSession(Context context,
      TrafficSource currentSource, boolean isFromBackground) {
    MoEDispatcher.getInstance(context).getBatchHelper().createAndSaveBatches(context, session);
    MoEDispatcher.getInstance(context).sendInteractionData();
    createAndPersistNewSession(context, currentSource, isFromBackground);
  }

  private UserSession createAndPersistNewSession(Context context,
      TrafficSource currentSource, boolean isFromBackground) {
    session = createNewSession(currentSource, isFromBackground);
    Logger.v(TAG + " createAndPersistNewSession() : New session: " + session.toString());
    persistUserSession(context, session);
    return session;
  }

  private UserSession createNewSession(@Nullable TrafficSource currentSource,
      boolean isFromBackground) {
    long currentTime = MoEUtils.currentTime();
    UserSession session = new UserSession();
    session.setSessionId(UUID.randomUUID().toString());
    session.setStartTime(MoEUtils.getTimeInISO(currentTime));
    if (currentSource != null) {
      session.setSource(currentSource);
    }
    session.setLastInteractionTime(currentTime);
    session.setBackgroundInitiated(isFromBackground);
    return session;
  }

  private void persistUserSession(Context context, UserSession userSession) {
    MoEDAO.getInstance(context).saveUserSession(userSession);
  }

  public void onAppClose(Context context) {
    updateLastInteractionTime(MoEUtils.currentTime());
    persistUserSession(context, session);
  }

  @WorkerThread public void onLogout(Context context) {
    createAndPersistNewSession(context, null, MoEHelper.isAppInBackground());
  }

  @Nullable public UserSession getSession() {
    return session;
  }
}
