package com.moengage.core;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobParameters;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.moe.pushlibrary.models.BatchData;
import com.moengage.core.analytics.AnalyticsHelper;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.reports.ReportsBatchHelper;
import com.moengage.core.rest.Response;
import com.moengage.core.utils.JsonBuilder;
import com.moengage.core.utils.RestUtils;
import java.util.ArrayList;
import java.util.TimeZone;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Send report batches to the server.
 *
 * @author Umang Chamaria
 */
class SendInteractionDataTask extends SDKTask {
  @Nullable private OnJobComplete jobCompleteCallback;
  @Nullable private JobParameters jobParameters;
  private ConfigurationProvider provider;

  private final String TAG = "SendInteractionDataTask";

  SendInteractionDataTask(Context context) {
    this(context, null, null);
  }

  SendInteractionDataTask(Context context, @Nullable OnJobComplete jobCompleteCallback,
      @Nullable JobParameters parameters) {
    super(context);
    this.jobCompleteCallback = jobCompleteCallback;
    this.jobParameters = parameters;
    provider = ConfigurationProvider.getInstance(this.context);
  }

  @Override public TaskResult execute() {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return null;
      Logger.v(TAG + " executing task");
      // creating and saving batches
      MoEDispatcher.getInstance(context).getBatchHelper().createAndSaveBatches(context,
          AnalyticsHelper.getInstance(context).getSession());
      // set cached event counter to zero
      MoEEventManager.getInstance(context).setEventCounter(0);

      String appId = MoEUtils.getAppId(context);
      if (TextUtils.isEmpty(appId)) {
        Logger.e(TAG + " execute: App-id not present cannot make report add call.");
        return null;
      }
      sendInteractionData(appId);
      schedulePeriodicRetryIfRequired();
      releaseJobLockIfRequired();
      Logger.v(TAG + " : completed task");
      return null;
    } catch (Exception e) {
      Logger.f(TAG + " : execute() ", e);
    }
    return null;
  }

  private void releaseJobLockIfRequired() {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
        && jobCompleteCallback != null
        && jobParameters != null) {
      Logger.v(TAG + " releaseJobLockIfRequired() : Trying to release job lock.");
      jobCompleteCallback.jobCompleted(jobParameters, false);
    }
  }

  @Override public String getTaskTag() {
    return TAG_SEND_INTERACTION_DATA;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private void sendInteractionData(String appId) {
    boolean result = false;
    ArrayList<BatchData> batchedData = null;

    String endpoint =
        MoEConstants.API_ENDPOINT_REPORT_ADD + "/" + appId;
    if (shouldSendDataToTestServer()) {
      endpoint = MoEConstants.API_ENDPOINT_INTEGRATION_VERIFICATION_REPORT_ADD;
    }

    ReportsBatchHelper batchHelper = new ReportsBatchHelper();
    for (; ; ) {
      batchedData = MoEDAO.getInstance(context).getBatchedData(100);
      Logger.d(
          "SendInteractionDataTask : sendInteractionData:Fetching interaction data in batches");
      if (batchedData == null || batchedData.isEmpty()) {
        Logger.d("SendInteractionDataTask : sendInteractionData: Found Nothing to send");
        return;
      }
      for (BatchData data : batchedData) {
        BatchData batch = batchHelper.updateBatchIfRequired(context, data);
        try {
          String requestId = getRequestID(data.batchDataJson);
          JSONObject requestBody =
              cleanBody(data.batchDataJson)
                  .put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, getQueryParams());
          Response response = APIManager.reportAdd(appId, endpoint, requestId,
              requestBody);
          if (response != null && response.responseCode == 200) result = true;
        } catch (Exception e) {
          Logger.f("SendInteractionDataTask : API failed", e);
          result = false;
        }
        if (result) {
          Logger.d("SendInteractionDataTask : Batch sent successfully deleting batch");
          MoEDAO.getInstance(context).deleteBatch(batch);
        } else {
          scheduleRetryIfRequired();
          break;
        }
      }
      if (!result) break;
      batchedData.clear();
    }
  }

  private void scheduleRetryIfRequired() {
    int retryCount = provider.getImmediateRetryCount();
    switch (retryCount) {
      case 0:
        scheduleImmediateRetry(1);
        provider.setImmediateRetryCount(++retryCount);
        break;
      case 1:
        scheduleImmediateRetry(3);
        provider.setImmediateRetryCount(++retryCount);
        break;
      default:
        provider.setImmediateRetryCount(0);
    }
  }

  /**
   * Checks whether device is registered as a test device or not
   *
   * @return true if registered as test device, else false
   */
  private boolean shouldSendDataToTestServer() {
    boolean isDeviceRegisteredForVerification =
        provider.isDeviceRegisteredForVerification();
    long registrationTime =
        provider.getVerificationRegistrationTime();
    return isDeviceRegisteredForVerification && ((registrationTime + 3600 * 1000)
        > MoEUtils.currentTime());
  }

  private void scheduleImmediateRetry(int minutes) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      scheduleSyncJob(666666, minutes * 60 * 1000);
    } else {
      scheduleSyncAlarm(55555, minutes * 60 * 1000);
    }
  }

  @Nullable private String getRequestID(JSONObject batchData) {
    try {
      return batchData.getString(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    } catch (JSONException e) {
      Logger.e("SendInteractionDataTask: getRequestID(): Exception ", e);
    }
    return null;
  }

  private JSONObject cleanBody(JSONObject batchData) {
    batchData.remove(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    return batchData;
  }

  private void schedulePeriodicRetryIfRequired() {
    if (!ConfigurationProvider.getInstance(context).isBackgroundSyncEnabled()) return;
    long syncInterval =
        ConfigurationCache.getInstance().getRemoteConfiguration().getDataSyncRetryInterval();
    Logger.v(TAG + " schedulePeriodicRetryIfRequired() : Will schedule background retry.");
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      scheduleSyncJob(77777, syncInterval);
    } else {
      scheduleSyncAlarm(88888, syncInterval);
    }
  }

  private void scheduleSyncAlarm(int requestCode, long timeDelayMillis) {
    Intent alarmIntent = new Intent(context, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(context, requestCode, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP, MoEUtils.currentTime() + timeDelayMillis,
          pendingIntent);
    }
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP) private void scheduleSyncJob(int requestCode,
      long timeDelayMillis) {
    ComponentName serviceComponent = new ComponentName(context, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(requestCode, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY)
        .setOverrideDeadline(MoEUtils.currentTime() + 2 * timeDelayMillis)
        .setMinimumLatency(timeDelayMillis);
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  private JSONObject getQueryParams() throws JSONException {
    JsonBuilder builder = RestUtils.getDefaultParams(context);
    builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());

    String fcmToken = provider.getGCMToken();
    if (!TextUtils.isEmpty(fcmToken) && !provider.isPushNotificationOptedOut()) {
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GCM_ID, fcmToken);
    }

    String miPushToken = provider.getMiPushToken();
    if (!TextUtils.isEmpty(miPushToken) && !provider.isPushNotificationOptedOut()) {
      builder.putString(MoEConstants.PARAM_V2_KEY_MI_PUSH_TOKEN, miPushToken);
    }

    if (!provider.isDataTrackingOptedOut()) {
      String androidId = MoEUtils.getAndroidID(context);
      if (!TextUtils.isEmpty(androidId)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, androidId);
      }
      if (!provider.isAdIdCollectionProhibitted()) {
        String gaid = provider.getStoredGAID();
        if (TextUtils.isEmpty(gaid)) {
          AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!TextUtils.isEmpty(gaid)) {
          builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION,
          String.valueOf(Build.VERSION.SDK_INT));
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL);
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
          provider.getAppVersionName());

      String nwType = MoEUtils.getNetworkType(context);
      if (!TextUtils.isEmpty(nwType)) {
        builder.putString(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
      }
    }

    return builder.build();
  }
}
