package com.moengage.core.rest;

import android.support.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.rest.RequestBuilder.RequestType;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import java.net.URL;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.net.ssl.HttpsURLConnection;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class RestClient {

  private static final String TAG = "RestClient";

  private Request request;
  private String path;

  public RestClient(Request request) {
    this.request = request;
    this.path = request.uri.getEncodedPath();
  }

  @Nullable public Response executeRequest() {
    return execute();
  }

  private Response execute() {
    HttpURLConnection urlConnection = null;
    Response response = null;
    try {
      String urlString = request.uri.toString();
      URL url = new URL(urlString);
      Logger.v(TAG + path + " execute(): Request url: " + urlString);
      if (RestConstants.SCHEME_HTTPS.equals(request.uri.getScheme())) {
        urlConnection = (HttpsURLConnection) url.openConnection();
      } else {
        urlConnection = (HttpURLConnection) url.openConnection();
      }
      addHeaders(urlConnection, request.headersMap);
      setContentType(urlConnection, request.contentType);
      setRequestType(urlConnection, request.requestType);
      addConnectionTimeOut(urlConnection, request.timeOut);
      if (request.requestType == RequestType.POST
          || request.requestType == RequestType.PUT) {
        addBody(urlConnection, request.requestBody);
      }
      int responseCode = urlConnection.getResponseCode();
      Logger.v(TAG + path + " execute(): Response Code: " + responseCode);
      if (responseCode == 200) {
        String responseBody = convertStreamToString(urlConnection.getInputStream());
        Logger.v(TAG + path + " execute(): Response Body: " + responseBody);
        response = new Response(responseCode, responseBody);
      } else {
        String errorResponse = convertStreamToString(urlConnection.getErrorStream());
        Logger.f(TAG + path + " Response: API Failed: "
            + " response code :"
            + responseCode
            + "reason : "
            + errorResponse);
        response = new Response(responseCode, errorResponse);
      }
    } catch (Exception e) {
      Logger.e( TAG + path + " execute() : Exception: ", e);
    }finally {
      if (urlConnection != null) {
        urlConnection.disconnect();
      }
    }
    return response;
  }

  private void addConnectionTimeOut(HttpURLConnection urlConnection, int timeOut) {
    urlConnection.setConnectTimeout(timeOut * 1000);
    urlConnection.setReadTimeout(timeOut * 1000);
  }

  private void setContentType(HttpURLConnection urlConnection, String contentType) {
    urlConnection.setRequestProperty(CONTENT_TYPE, contentType);
  }

  private void setRequestType(HttpURLConnection urlConnection,
      RequestBuilder.RequestType requestType) throws ProtocolException {
    urlConnection.setRequestMethod(requestType.toString());
  }

  private void addBody(HttpURLConnection urlConnection, JSONObject requestBody) throws IOException {
    urlConnection.setDoOutput(true);
    urlConnection.setRequestProperty(CHARACTER_SET, ENCODING_CHARSET_UTF8);
    urlConnection.setRequestProperty(CONTENT_TYPE, RestConstants.DEFAULT_CONTENT_TYPE);
    OutputStream output = urlConnection.getOutputStream();
    if (requestBody != null) {
      Logger.v(TAG + path + " addBody(): Request Body: " + requestBody.toString());
      output.write(requestBody.toString().getBytes(ENCODING_CHARSET_UTF8));
    }
    output.close();
  }

  private void addHeaders(HttpURLConnection urlConnection, Map<String, String> headers) {
    Set<Entry<String, String>> headerMap = headers.entrySet();
    for (Map.Entry<String, String> header : headerMap) {
      Logger.v(TAG + path + " addHeaders(): " + header.getKey() + " : " + header.getValue());
      urlConnection.addRequestProperty(header.getKey(), header.getValue());
    }
  }

  private String convertStreamToString(InputStream inputStream) throws IOException {
    if (inputStream == null) return null;
    BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
    StringBuilder stringBuilder = new StringBuilder();

    String line;
    try {
      while ((line = reader.readLine()) != null) {
        stringBuilder.append(line);
      }
    } finally {
      inputStream.close();
    }
    return stringBuilder.toString();
  }

  private static final String ENCODING_CHARSET_UTF8 = "UTF-8";
  private static final String CHARACTER_SET = "Accept-Charset";
  private static final String CONTENT_TYPE = "Content-type";
}