/* ************************************************************************
 * 
 * MOENGAGE CONFIDENTIAL
 * __________________
 * 
 *  [2014] - [2015] MoEngage Inc. 
 *  All Rights Reserved.
 * 
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moe.pushlibrary.providers;

import android.content.ContentProvider;
import android.content.ContentProviderOperation;
import android.content.ContentProviderResult;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.OperationApplicationException;
import android.content.UriMatcher;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import com.moe.pushlibrary.providers.MoEDataContract.AttributeCacheEntity;
import com.moe.pushlibrary.providers.MoEDataContract.BatchDataEntity;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DTEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DatapointEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppMessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.UserAttributeEntity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.model.DataTypes;
import java.util.ArrayList;
import java.util.HashMap;
import org.json.JSONObject;

/**
 * This is a content provider for accessing the MoEngage data.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @since v5
 */
public class MoEProvider extends ContentProvider {

  private static final String TAG = "MoEProvider";

  /**
   * Denotes that the URI matched Messages
   */
  private static final int MESSAGES = 1;
  /**
   * Denotes that the URI matched a` single PromotionalMessage entry
   */
  private static final int MESSAGE_ID = 2;
  /**
   * Denotes that the URI matched Events
   */
  private static final int DATAPOINTS = 3;
  /**
   * Denotes that the URI matched a single Event entry
   */
  private static final int DATAPOINT_ID = 4;
  /**
   * Denotes that the URI matched InApps
   */
  private static final int INAPPS = 5;
  /**
   * Denotes that the URI matched a single InApp entry
   */
  private static final int INAPP_ID = 6;
  /**
   * Denotes that URI matched user attributes
   */
  private static final int USER_ATTRIBUTES = 9;
  /**
   * Denotes that URI matched a  user attribute
   */
  private static final int USER_ATTRIBUTES_ID = 10;
  /**
   * Denotes that URI matched campaign ids
   */
  private static final int CAMPAIGN_IDS = 11;
  /**
   * Denotes that URI matched a  campaign id
   */
  private static final int CAMPAIGN_ID = 12;
  /**
   * Denotes the uri matched to batches of data
   */
  private static final int DATA_BATCHES = 13;
  /**
   * Denotes the uri matched to a batch of data
   */
  private static final int DATA_BATCH = 14;

  private static final int DEVICE_TRIGGERS = 15;

  private static final int DEVICE_TRIGGER = 16;

  private static final int ATTRIBUTE_CACHE = 17;

  private static final int ATTRIBUTE_CACHE_ID = 18;

  /**
   * Instance of {@link UriMatcher} used for getting the type of the URI
   */
  private static UriMatcher sUriMatcher;

  private static HashMap<String, String> sMessageProjectionMap;
  private static HashMap<String, String> sEventProjectionMap;
  private static HashMap<String, String> sInAppProjectionMap;
  private static HashMap<String, String> sUserAttributeProjectionMap;
  private static HashMap<String, String> sCampaignListProjectionMap;
  private static HashMap<String, String> sBatchDataProjectionMap;
  private static HashMap<String, String> sDTProjectionMap;
  private static HashMap<String, String> sAttributeCacheMap;

  static {

    sMessageProjectionMap = new HashMap<>();
    sMessageProjectionMap.put(MessageEntity._ID, MessageEntity._ID);
    sMessageProjectionMap.put(MessageEntity.GTIME, MessageEntity.GTIME);
    sMessageProjectionMap.put(MessageEntity.MSG_DETAILS, MessageEntity.MSG_DETAILS);
    sMessageProjectionMap.put(MessageEntity.MSG_CLICKED, MessageEntity.MSG_CLICKED);
    sMessageProjectionMap.put(MessageEntity.MSG_TTL, MessageEntity.MSG_TTL);
    sMessageProjectionMap.put(MessageEntity.MSG_TAG, MessageEntity.MSG_TAG);

    sEventProjectionMap = new HashMap<>();
    sEventProjectionMap.put(DatapointEntity._ID, DatapointEntity._ID);
    sEventProjectionMap.put(DatapointEntity.GTIME, DatapointEntity.GTIME);
    sEventProjectionMap.put(DatapointEntity.DETAILS, DatapointEntity.DETAILS);

    sInAppProjectionMap = new HashMap<>();
    sInAppProjectionMap.put(InAppMessageEntity._ID, InAppMessageEntity._ID);
    sInAppProjectionMap.put(InAppMessageEntity.GTIME, InAppMessageEntity.GTIME);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_CAMPAIGN_ID, InAppMessageEntity.MSG_CAMPAIGN_ID);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_ALIGN_TYPE, InAppMessageEntity.MSG_ALIGN_TYPE);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_INAPP_TYPE, InAppMessageEntity.MSG_INAPP_TYPE);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_TTL, InAppMessageEntity.MSG_TTL);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_MIN_DELAY, InAppMessageEntity.MSG_MIN_DELAY);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_MAX_TIMES, InAppMessageEntity.MSG_MAX_TIMES);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_SHOWN_COUNT, InAppMessageEntity.MSG_SHOWN_COUNT);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_PERSISTENT, InAppMessageEntity.MSG_PERSISTENT);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_PRIORITY, InAppMessageEntity.MSG_PRIORITY);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_CONTEXT, InAppMessageEntity.MSG_CONTEXT);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_LAST_SHOWN, InAppMessageEntity.MSG_LAST_SHOWN);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_IS_CLICKED, InAppMessageEntity.MSG_IS_CLICKED);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_HAS_ERRORS, InAppMessageEntity.MSG_HAS_ERRORS);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_AUTODISMISS_TIME,
        InAppMessageEntity.MSG_AUTODISMISS_TIME);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_CANCELABLE, InAppMessageEntity.MSG_CANCELABLE);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_CONTENT, InAppMessageEntity.MSG_CONTENT);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_SHOW_ONLY_IN,
        InAppMessageEntity.MSG_SHOW_ONLY_IN);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_STATUS, InAppMessageEntity.MSG_STATUS);
    sInAppProjectionMap.put(InAppMessageEntity.MSG_CONTAINER_STYLE,
        InAppMessageEntity.MSG_CONTAINER_STYLE);

    sUserAttributeProjectionMap = new HashMap<>();
    sUserAttributeProjectionMap.put(UserAttributeEntity._ID, UserAttributeEntity._ID);
    sUserAttributeProjectionMap.put(UserAttributeEntity.GTIME, UserAttributeEntity.GTIME);
    sUserAttributeProjectionMap.put(UserAttributeEntity.ATTRIBUTE_NAME, UserAttributeEntity
        .ATTRIBUTE_NAME);
    sUserAttributeProjectionMap.put(UserAttributeEntity.ATTRIBUTE_VALUE, UserAttributeEntity
        .ATTRIBUTE_VALUE);

    sCampaignListProjectionMap = new HashMap<>();
    sCampaignListProjectionMap.put(CampaignListEntity._ID, CampaignListEntity._ID);
    sCampaignListProjectionMap.put(CampaignListEntity.CAMPAIGN_ID, CampaignListEntity.CAMPAIGN_ID);
    sCampaignListProjectionMap.put(CampaignListEntity.CAMPAIGN_ID_TTL, CampaignListEntity.CAMPAIGN_ID_TTL);

    sBatchDataProjectionMap = new HashMap<>();
    sBatchDataProjectionMap.put(BatchDataEntity._ID, BatchDataEntity._ID);
    sBatchDataProjectionMap.put(BatchDataEntity.BATCHED_DATA, BatchDataEntity.BATCHED_DATA);

    sDTProjectionMap = new HashMap<>();
    sDTProjectionMap.put(DTEntity._ID, DTEntity._ID);
    sDTProjectionMap.put(DTEntity.CAMPAIGN_ID, DTEntity.CAMPAIGN_ID);
    sDTProjectionMap.put(DTEntity.TRIGGER_EVENT_NAME, DTEntity.TRIGGER_EVENT_NAME );
    sDTProjectionMap.put(DTEntity.PAYLOAD, DTEntity.PAYLOAD);
    sDTProjectionMap.put(DTEntity.CAMPAIGN_PAYLOAD, DTEntity.CAMPAIGN_PAYLOAD);
    sDTProjectionMap.put(DTEntity.CAMPAIGN_TYPE, DTEntity.CAMPAIGN_TYPE);
    sDTProjectionMap.put(DTEntity.MAX_COUNT, DTEntity.MAX_COUNT);
    sDTProjectionMap.put(DTEntity.MINIMUM_DELAY, DTEntity.MINIMUM_DELAY);
    sDTProjectionMap.put(DTEntity.SHOULD_SHOW_OFFLINE, DTEntity.SHOULD_SHOW_OFFLINE);
    sDTProjectionMap.put(DTEntity.MAX_SYNC_DELAY_TIME, DTEntity.MAX_SYNC_DELAY_TIME);
    sDTProjectionMap.put(DTEntity.EXPIRY_TIME, DTEntity.EXPIRY_TIME);
    sDTProjectionMap.put(DTEntity.PRIORITY, DTEntity.PRIORITY);
    sDTProjectionMap.put(DTEntity.LAST_SHOW_TIME, DTEntity.LAST_SHOW_TIME);
    sDTProjectionMap.put(DTEntity.SHOW_COUNT, DTEntity.SHOW_COUNT);
    sDTProjectionMap.put(DTEntity.LAST_UPDATED_TIME, DTEntity.LAST_UPDATED_TIME);
    sDTProjectionMap.put(DTEntity.STATUS, DTEntity.STATUS);
    sDTProjectionMap.put(DTEntity.SHOULD_IGNORE_DND, DTEntity.SHOULD_IGNORE_DND);
    sDTProjectionMap.put(DTEntity.DELAY_BEFORE_SHOWING_NOTIFICATION, DTEntity
        .DELAY_BEFORE_SHOWING_NOTIFICATION);

    sAttributeCacheMap = new HashMap<>();
    sAttributeCacheMap.put(AttributeCacheEntity._ID, AttributeCacheEntity._ID);
    sAttributeCacheMap.put(AttributeCacheEntity.ATTRIBUTE_NAME, AttributeCacheEntity.ATTRIBUTE_NAME);
    sAttributeCacheMap.put(AttributeCacheEntity.ATTRIBUTE_VALUE, AttributeCacheEntity.ATTRIBUTE_VALUE);
    sAttributeCacheMap.put(AttributeCacheEntity.LAST_TRACKED_TIME, AttributeCacheEntity.LAST_TRACKED_TIME);
    sAttributeCacheMap.put(AttributeCacheEntity.DATA_TYPE, AttributeCacheEntity.DATA_TYPE);
  }

  /**
   * The database helper which is used to do all database related operations
   */
  class DatabaseHelper extends SQLiteOpenHelper {
    /**
     * The database name
     */
    private static final String DATABASE_NAME = "MOEInteractions";
    /**
     * The current database version
     */
    static final int DATABASE_VERSION = 14;

    /**
     * @param context Application Context
     */
    public DatabaseHelper(Context context) {
      super(context, DATABASE_NAME, null, DATABASE_VERSION);
    }

    /**
     * Table containing the Unified Inbox messages
     */
    static final String TABLE_NAME_UINBOX = "UINBOX";
    /**
     * Table containing the Inbox messages
     */
    static final String TABLE_NAME_MSGS = "MESSAGES";
    /**
     * Table containing events
     */
    static final String TABLE_NAME_DATAPOINTS = "DATAPOINTS";
    /**
     * Table containing in app messages
     */
    static final String TABLE_NAME_INAPPS = "INAPPMSG";
    /**
     * Old table containing in app messages
     */
    static final String TABLE_NAME_OLD_INAPPS = "INAPPS";
    /**
     * Table to save user attributes
     */
    static final String TABLE_NAME_USER_ATTRIBUTES = "USERATTRIBUTES";
    /**
     * Table to save campaign list
     */
    static final String TABLE_NAME_CAMPAIGN_LIST = "CAMPAIGNLIST";
    /**
     * Table to save batched data
     */
    static final String TABLE_NAME_BATCH_DATA = "BATCH_DATA";
    /**
     * Table to save device triggers
     */
    static final String TABLE_NAME_DEVICE_TRIGGERS = "DEVICE_TRIGGERS";

    static final String TABLE_NAME_ATTRIBUTE_CACHE = "ATTRIBUTE_CACHE";


    @Override public void onCreate(SQLiteDatabase db) {
      // SQL statement to create interactions table
      String DDL_DATAPOINTS = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_DATAPOINTS
          + " ( "
          + DatapointEntity._ID
          + " INTEGER PRIMARY KEY, "
          + DatapointEntity.GTIME
          + " INTEGER, "
          + DatapointEntity.DETAILS
          + " TEXT ); ";

      String DDL_MSGS = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_MSGS
          + " ( "
          + MessageEntity._ID
          + " INTEGER PRIMARY KEY, "
          + MessageEntity.MSG_DETAILS
          + " TEXT, "
          + MessageEntity.MSG_CLICKED
          + " INTEGER DEFAULT 0, "
          + MessageEntity.MSG_TTL
          + " INTEGER, "
          + MessageEntity.GTIME
          + " INTEGER, "
          + MessageEntity.MSG_TAG
          + " TEXT); ";

      createInAppTable(db);
      createUserAttributeTable(db);
      createCampaignListTable(db);
      db.execSQL(DDL_MSGS);
      db.execSQL(DDL_DATAPOINTS);
      createBatchDataTable(db);
      createDeviceTriggerTable(db);
      createAttributeCacheTableIfRequired(db);
      Logger.i("MoEProvider: Database created");
    }

    private boolean tableExists(SQLiteDatabase db, final String tableName) {
      Cursor cursor = null;
      try {
        cursor = db.rawQuery(
            "select DISTINCT tbl_name from sqlite_master where tbl_name = '" + tableName + "'",
            null);
        return (cursor != null && cursor.getCount() > 0);
      } finally {
        if (null != cursor) {
          cursor.close();
        }
      }
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * android.database.sqlite.SQLiteOpenHelper#onUpgrade(android.database
     * .sqlite.SQLiteDatabase, int, int)
     */
    @Override public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
      Logger.v("MoEProvider: Provider upgrading DB ");

      int upgradeTo = oldVersion + 1;
      while (upgradeTo <= newVersion) {
        switch (upgradeTo) {
          case 3:
            db.beginTransaction();
            try {
              createDataPointsTable(db);
              createMessagesTable(db);
              //creating the in-appv2 table here as the old table was dropped.
              createInAppTable(db);
              portDataFromv2(db);
              db.execSQL("DROP TABLE IF EXISTS moeints");
              db.execSQL("DROP TABLE IF EXISTS moemsgs");
              db.execSQL("DROP TABLE IF EXISTS moeinappmsgs");
              db.setTransactionSuccessful();
            } catch (Exception e) {
              Logger.f("MoEProvider: failed to port data. FROM V2.", e);
            } finally {
              db.endTransaction();
            }
            break;
          case 4:
            // DBv4 Never released
            break;
          case 5:
            db.beginTransaction();
            try {
              db.execSQL("DROP TABLE IF EXISTS CHATS");
              db.setTransactionSuccessful();
            } catch (Exception e) {
              Logger.f("MoEProvider: failed to port data.. FOR UBOX", e);
            } finally {
              db.endTransaction();
            }
            break;
          case 6:
            db.beginTransaction();
            try {
              if (isTableExists(db, TABLE_NAME_OLD_INAPPS)) {
                String ALTER_INAPP =
                    " ALTER TABLE " + TABLE_NAME_OLD_INAPPS + " ADD COLUMN TYPE INTEGER";
                db.execSQL(ALTER_INAPP);
              }
              db.setTransactionSuccessful();
            } catch (Exception e) {
              Logger.f("MoEProvider: failed to add column " + TABLE_NAME_OLD_INAPPS, e);
            } finally {
              db.endTransaction();
            }
            break;
          case 7:
            db.beginTransaction();
            try {
              createDataPointsTable(db);
              populateDatapoints(db);
              db.execSQL("DROP TABLE IF EXISTS EVENTS");
              db.setTransactionSuccessful();
            } catch (Exception e) {
              Logger.f("MoEProvider: failed to populate Datapoints ", e);
            } finally {
              db.endTransaction();
            }
            break;
          case 8:
            db.beginTransaction();
            try {
              if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
                String ALTER_INBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
                    MessageEntity.MSG_TAG + " TEXT";
                db.execSQL(ALTER_INBOX);
              }
              db.setTransactionSuccessful();
            } catch (Exception e) {
              Logger.f("MoEProvider: failed to add columns to " + TABLE_NAME_UINBOX + " / " +
                  TABLE_NAME_MSGS, e);
            } finally {
              db.endTransaction();
            }
            break;
          case 9:
            db.execSQL("DROP TABLE IF EXISTS INAPPS");
            createInAppTable(db);
            //this check is added for the mess up made by big basket.
            if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
              String ALTER_INBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
                  MessageEntity.MSG_TAG + " TEXT";
              db.execSQL(ALTER_INBOX);
            }
            break;
          case 10:
            createUserAttributeTable(db);
            break;
          case 11:
            createCampaignListTable(db);
            break;
          case 12:
            createBatchDataTable(db);
            break;
          case 13:
            createDeviceTriggerTable(db);
            break;
          case 14:
            createAttributeCacheTableIfRequired(db);
            portUserAttributeUniqueId(db);
            break;
          default:
            Logger.v("Failed to upgrade from DB version" + oldVersion + "to DB version" +
                newVersion);
        }
        upgradeTo++;
      }
      ConfigurationProvider.getInstance(getContext()).setNewDBVersion(DATABASE_VERSION);
      Logger.v("MoEProvider: Database Upgraded");
    }

    private void createDataPointsTable(SQLiteDatabase db) {
      String DDL_DATAPOINTS = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_DATAPOINTS
          + " ( "
          + DatapointEntity._ID
          + " INTEGER PRIMARY KEY, "
          + DatapointEntity.GTIME
          + " INTEGER, "
          + DatapointEntity.DETAILS
          + " TEXT ); ";
      db.execSQL(DDL_DATAPOINTS);
    }

    private void createMessagesTable(SQLiteDatabase db) {
      String DDL_MSGS = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_MSGS
          + " ( "
          + MessageEntity._ID
          + " INTEGER PRIMARY KEY, "
          + MessageEntity.MSG_DETAILS
          + " TEXT, "
          + MessageEntity.MSG_CLICKED
          + " INTEGER DEFAULT 0, "
          + MessageEntity.MSG_TTL
          + " INTEGER, "
          + MessageEntity.GTIME
          + " INTEGER )";
      db.execSQL(DDL_MSGS);
    }

    private void createInAppTable(SQLiteDatabase db) {
      String DDL_INAPP = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_INAPPS
          + " ( "
          + InAppMessageEntity._ID
          + " INTEGER PRIMARY KEY, "
          + InAppMessageEntity.GTIME
          + " INTEGER, "
          + InAppMessageEntity.MSG_CAMPAIGN_ID
          + " TEXT, "
          + InAppMessageEntity.MSG_ALIGN_TYPE
          + " TEXT, "
          + InAppMessageEntity.MSG_INAPP_TYPE
          + " TEXT, "
          + InAppMessageEntity.MSG_TTL
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_MIN_DELAY
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_MAX_TIMES
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_SHOWN_COUNT
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_PERSISTENT
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_PRIORITY
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_CONTEXT
          + " TEXT, "
          + InAppMessageEntity.MSG_LAST_SHOWN
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_IS_CLICKED
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_HAS_ERRORS
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_AUTODISMISS_TIME
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_CANCELABLE
          + " INTEGER DEFAULT 0, "
          + InAppMessageEntity.MSG_CONTENT
          + " TEXT, "
          + InAppMessageEntity.MSG_SHOW_ONLY_IN
          + " TEXT, "
          + InAppMessageEntity.MSG_STATUS
          + " TEXT, "
          + InAppMessageEntity.MSG_CONTAINER_STYLE
          + " TEXT );";
      db.execSQL(DDL_INAPP);
    }

    public void addMSGTagIfRequiredInbox(SQLiteDatabase db) {
      if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
        Logger.v("MoEProvider : addMSGTagIfRequiredInbox : updating inbox table");
        String ALTER_UBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
            MessageEntity.MSG_TAG + " TEXT";
        db.execSQL(ALTER_UBOX);
      }
    }

    public void addUserAttributesTableIfRequired(SQLiteDatabase db){
      if (!tableExists(db, TABLE_NAME_USER_ATTRIBUTES)){
        Logger.v("MoEProvider : DatabaseHelper : addUserAttributesTableIfRequired creating "
            + "missing  user attribute table");
        createUserAttributeTable(db);
      }else {
        Logger.v("MoEProvider : DatabaseHelper : addUserAttributesTableIfRequired "
            + "user attribute table already present");
      }
    }

    public void addCampaignListTableIfRequired(SQLiteDatabase db){
      if (!tableExists(db, TABLE_NAME_CAMPAIGN_LIST)){
        Logger.v("MoEProvider : DatabaseHelper : addCampaignListTableIfRequired creating "
            + "missing  campaign list table");
        createUserAttributeTable(db);
      }else {
        Logger.v("MoEProvider : DatabaseHelper : addCampaignListTableIfRequired "
            + "campaign list table already present");
      }
    }

    private void createUserAttributeTable(SQLiteDatabase db){
      String DDL_USER_ATTRIBUTES = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_USER_ATTRIBUTES
          + " ( "
          + UserAttributeEntity._ID
          + " INTEGER PRIMARY KEY, "
          + UserAttributeEntity.ATTRIBUTE_NAME
          + " TEXT, "
          + UserAttributeEntity.ATTRIBUTE_VALUE
          + " TEXT"
          + " ); ";
      db.execSQL(DDL_USER_ATTRIBUTES);
    }

    private void createCampaignListTable(SQLiteDatabase db){
      String DDL_CAMPAIGN_LIST = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_CAMPAIGN_LIST
          + " ( "
          + CampaignListEntity._ID
          + " INTEGER PRIMARY KEY, "
          + CampaignListEntity.CAMPAIGN_ID
          + " TEXT, "
          + CampaignListEntity.CAMPAIGN_ID_TTL
          + " INTEGER"
          + " );";
      db.execSQL(DDL_CAMPAIGN_LIST);
    }

    private void createBatchDataTable(SQLiteDatabase db){
      String DDL_BATCH_DATA = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_BATCH_DATA
          + " ( "
          + BatchDataEntity._ID
          + " INTEGER PRIMARY KEY, "
          + BatchDataEntity.BATCHED_DATA
          + " TEXT"
          + " );";
      db.execSQL(DDL_BATCH_DATA);
    }

    void addBatchDataTableIfRequired(SQLiteDatabase db){
      if (!tableExists(db, TABLE_NAME_BATCH_DATA)){
        Logger.v("MoEProvider : DatabaseHelper : addBatchDataTableIfRequired creating "
            + "missing  campaign list table");
        createBatchDataTable(db);
      }else {
        Logger.v("MoEProvider: DatabaseHelper : addBatchDataTableIfRequired batch data table is "
            + "already present");
      }
    }

    void createDeviceTriggerTable(SQLiteDatabase db) {
      String DDL_DEVICE_TRIGGERS = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_DEVICE_TRIGGERS
          + " ( "
          + DTEntity._ID
          + " INTEGER PRIMARY KEY, "
          + DTEntity.CAMPAIGN_ID
          + " TEXT, "
          + DTEntity.TRIGGER_EVENT_NAME
          + " TEXT, "
          + DTEntity.PAYLOAD
          + " TEXT, "
          + DTEntity.CAMPAIGN_PAYLOAD
          + " TEXT, "
          + DTEntity.CAMPAIGN_TYPE
          + " TEXT, "
          + DTEntity.MAX_COUNT
          + " INTEGER DEFAULT 0, "
          + DTEntity.MINIMUM_DELAY
          + " INTEGER DEFAULT 0, "
          + DTEntity.SHOULD_SHOW_OFFLINE
          + " INTEGER DEFAULT 0, "
          + DTEntity.MAX_SYNC_DELAY_TIME
          + " INTEGER DEFAULT 0, "
          + DTEntity.EXPIRY_TIME
          + " INTEGER, "
          + DTEntity.PRIORITY
          + " INTEGER DEFAULT 3, "
          + DTEntity.LAST_SHOW_TIME
          + " INTEGER DEFAULT 0, "
          + DTEntity.SHOW_COUNT
          + " INTEGER DEFAULT 0, "
          + DTEntity.LAST_UPDATED_TIME
          + " INTEGER DEFAULT 0, "
          + DTEntity.STATUS
          + " TEXT, "
          + DTEntity.SHOULD_IGNORE_DND
          + " INTEGER DEFAULT 0, "
          + DTEntity.DELAY_BEFORE_SHOWING_NOTIFICATION
          + " INTEGER DEFAULT 0 "
          + " ) ";
      db.execSQL(DDL_DEVICE_TRIGGERS);
    }

    void addDeviceTriggerTableIfRequired(SQLiteDatabase db){
      if (!tableExists(db, TABLE_NAME_DEVICE_TRIGGERS)){
        Logger.v("MoEProvider : DatabaseHelper : addDeviceTriggerTableIfRequired creating "
            + "missing  device trigger table");
        createDeviceTriggerTable(db);
      }else {
        Logger.v("MoEProvider: DatabaseHelper : addDeviceTriggerTableIfRequired device trigger table is "
            + "already present");
      }
    }

    void createAttributeCacheTableIfRequired(SQLiteDatabase database){
      String DDL_ATTRIBUTE_CACHE = "CREATE TABLE IF NOT EXISTS "
          + TABLE_NAME_ATTRIBUTE_CACHE
          + " ( "
          + AttributeCacheEntity._ID
          + " INTEGER PRIMARY KEY, "
          + AttributeCacheEntity.ATTRIBUTE_NAME
          + " TEXT, "
          + AttributeCacheEntity.ATTRIBUTE_VALUE
          + " TEXT, "
          + AttributeCacheEntity.LAST_TRACKED_TIME
          + " INTEGER DEFAULT 0, "
          + AttributeCacheEntity.DATA_TYPE
          + " TEXT "
          + " ) ";
      database.execSQL(DDL_ATTRIBUTE_CACHE);
    }

    void addAttributeCacheTableIfRequired(SQLiteDatabase db){
      if (!tableExists(db, TABLE_NAME_ATTRIBUTE_CACHE)){
        Logger.v("MoEProvider : DatabaseHelper : addAttributeCacheTableIfRequired() creating "
            + "missing  attribute cache table");
        createAttributeCacheTableIfRequired(db);
      }else {
        Logger.v("MoEProvider: DatabaseHelper : addAttributeCacheTableIfRequired() attribute cache"
            + " table is already present");
      }
    }

    private void portUserAttributeUniqueId(SQLiteDatabase db){
      try {
        Logger.v(TAG + " portUserAttributeUniqueId(): Will try to port USER_ATTRIBUTE_UNIQUE_ID "
            + "to another storage.");
        if (!tableExists(db, TABLE_NAME_USER_ATTRIBUTES)){
          Logger.v(TAG + " portUserAttributeUniqueId() : User attribute table does not exist");
          return;
        }
        // fetch value from table
        db.beginTransaction();
        String QUERY_GET_USER_ATTRIBUTE_UNIQUE_ID = "SELECT "
            + UserAttributeEntity.ATTRIBUTE_NAME + ", "
            + UserAttributeEntity.ATTRIBUTE_VALUE + " "
            + "FROM " + TABLE_NAME_USER_ATTRIBUTES
            + " WHERE " + UserAttributeEntity.ATTRIBUTE_NAME + " = ?";
        Cursor cursor = db.rawQuery(QUERY_GET_USER_ATTRIBUTE_UNIQUE_ID,
            new String[]{MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID});
        if (cursor == null){
          Logger.v(TAG + " portUserAttributeUniqueId(): USER_ATTRIBUTE_UNIQUE_ID not present. "
              + "Cursor is null");
          return;
        }
        if (!cursor.moveToFirst()){
          Logger.v(TAG + " portUserAttributeUniqueId(): USER_ATTRIBUTE_UNIQUE_ID not present. "
              + "Cursor does not have a 1st element.");
          return;
        }
        String uniqueId = cursor.getString(1);
        cursor.close();
        // copy existing value to shared preference
        ConfigurationProvider.getInstance(getContext()).saveUserAttributeUniqueId(uniqueId);
        // copy existing value to attribute cache table
        if (!tableExists(db, TABLE_NAME_ATTRIBUTE_CACHE)){
          Logger.v(TAG + " portUserAttributeUniqueId() : Attribute Cache does not exist. Cannot "
              + "port data");
          return;
        }
        ContentValues contentValue = new ContentValues();
        contentValue.put(AttributeCacheEntity.ATTRIBUTE_NAME, MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID);
        contentValue.put(AttributeCacheEntity.ATTRIBUTE_VALUE, uniqueId);
        contentValue.put(AttributeCacheEntity.LAST_TRACKED_TIME, 0);
        contentValue.put(AttributeCacheEntity.DATA_TYPE, DataTypes.STRING.toString());
        db.insert(TABLE_NAME_ATTRIBUTE_CACHE, null, contentValue);
        db.setTransactionSuccessful();
      } catch (Exception e) {
        Logger.f(TAG + " portUserAttributeUniqueId(): Exception ", e);
      }finally {
        db.endTransaction();
      }
    }
  }

  private void populateDatapoints(SQLiteDatabase db) {
    if (null == db) {
      return;
    }
    try {
      Logger.d("Started porting DATA - FOR DATAPOINTS");
      db.beginTransaction();
      String QUERY_GET_EVENTS = "SELECT _id, action, attrs, gtime, ltime FROM EVENTS";
      Cursor cursor = db.rawQuery(QUERY_GET_EVENTS, null);
      if (null != cursor && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();
          JSONObject details = MoEHelperUtils.getDatapointJSON(cursor.getString(1),
              new JSONObject(cursor.getString(2)), Long.toString(cursor.getLong(3)),
              cursor.getString(4));
          if (null != details) {
            values.put(DatapointEntity.DETAILS, details.toString());
            values.put(DatapointEntity.GTIME, cursor.getLong(3));
            long row = db.insert(DatabaseHelper.TABLE_NAME_DATAPOINTS, null, values);
            Logger.v("MoEProvider:onUpgrade: Porting event data: " + row);
          }
        } while (cursor.moveToNext());
        cursor.close();
      }
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.f("MoEProvider: populateDatapoints", e);
    } finally {
      db.endTransaction();
    }
  }
  private void portDataFromv2(SQLiteDatabase db) {
    try {
      db.beginTransaction();

      //no need to port events data might cause issues

      String QUERY_GET_MSGS = "SELECT * FROM moemsgs";
      Cursor cursor = db.rawQuery(QUERY_GET_MSGS, null);
      if (null != cursor && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();
          values.put(MessageEntity.MSG_DETAILS, cursor.getString(1));
          values.put(MessageEntity.MSG_CLICKED, cursor.getInt(2));
          values.put(MessageEntity.MSG_TTL, Long.parseLong(cursor.getString(3)));
          values.put(MessageEntity.GTIME, Long.parseLong(cursor.getString(4)));
          long row = db.insert(DatabaseHelper.TABLE_NAME_MSGS, null, values);
          Logger.v("MoEProvider:onUpgrade: Porting message data: " + row);
        } while (cursor.moveToNext());
        cursor.close();
        cursor = null;
      }

      //removed porting code for inapps
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.f("MoEProvider: portDatafromv2", e);
    } finally {
      if (null != db) {
        db.endTransaction();
      }
    }
  }

  private DatabaseHelper mOpenHelper = null;

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#onCreate()
   */
  @Override public boolean onCreate() {
    mOpenHelper = new DatabaseHelper(getContext());
    initializeUriMatcher();
    Logger.d("MoEProvider: Provider created");
    return true;
  }

  private void initializeUriMatcher() {
    sUriMatcher = new UriMatcher(UriMatcher.NO_MATCH);
    Context context = getContext();
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "messages", MESSAGES);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "messages/#", MESSAGE_ID);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "datapoints", DATAPOINTS);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "datapoints/#", DATAPOINT_ID);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "inapps", INAPPS);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "inapps/#", INAPP_ID);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "userattributes/", USER_ATTRIBUTES);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "userattributes/#", USER_ATTRIBUTES_ID);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "campaignlist/", CAMPAIGN_IDS);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "campaignlist/#", CAMPAIGN_ID);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "batchdata/", DATA_BATCHES);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "batchdata/#", DATA_BATCH);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "dtcampaign", DEVICE_TRIGGERS);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "dtcampaign/#", DEVICE_TRIGGER);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "attributecache/", ATTRIBUTE_CACHE);
    sUriMatcher.addURI(MoEDataContract.getAuthority(context), "attributecache/#", ATTRIBUTE_CACHE_ID);

  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#getType(android.net.Uri)
   */
  @Override public String getType(@NonNull Uri uri) {
    switch (sUriMatcher.match(uri)) {
      case MESSAGES:
        return MessageEntity.CONTENT_TYPE;
      case MESSAGE_ID:
        return MessageEntity.CONTENT_ITEM_TYPE;
      case DATAPOINTS:
        return DatapointEntity.CONTENT_TYPE;
      case DATAPOINT_ID:
        return DatapointEntity.CONTENT_ITEM_TYPE;
      case INAPPS:
        return InAppMessageEntity.CONTENT_TYPE;
      case INAPP_ID:
        return InAppMessageEntity.CONTENT_ITEM_TYPE;
      case USER_ATTRIBUTES:
        return UserAttributeEntity.CONTENT_TYPE;
      case USER_ATTRIBUTES_ID:
        return UserAttributeEntity.CONTENT_ITEM_TYPE;
      case CAMPAIGN_IDS:
        return CampaignListEntity.CONTENT_TYPE;
      case CAMPAIGN_ID:
        return CampaignListEntity.CONTENT_ITEM_TYPE;
      case DATA_BATCHES:
        return BatchDataEntity.CONTENT_TYPE;
      case DATA_BATCH:
        return BatchDataEntity.CONTENT_ITEM_TYPE;
      case DEVICE_TRIGGERS:
        return DTEntity.CONTENT_TYPE;
      case DEVICE_TRIGGER:
        return DTEntity.CONTENT_ITEM_TYPE;
      case ATTRIBUTE_CACHE:
        return AttributeCacheEntity.CONTENT_TYPE;
      case ATTRIBUTE_CACHE_ID:
        return AttributeCacheEntity.CONTENT_ITEM_TYPE;
      default:
        Logger.f("No Matching URI found");
        return null;
    }
  }

  public Cursor query(@NonNull Uri uri, String[] projection, String selection,
      String[] selectionArgs, String sortOrder) {
    // Get the database and run the query
    SQLiteDatabase db = mOpenHelper.getReadableDatabase();
    updateIfRequired(db);
    SQLiteQueryBuilder qb = new SQLiteQueryBuilder();
    // If no sort order is specified use the default
    String orderBy = sortOrder;
    String limit = uri.getQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT);
    switch (sUriMatcher.match(uri)) {
      case MESSAGE_ID:
        qb.appendWhere(MessageEntity._ID + "=" + uri.getPathSegments().get(1));
      case MESSAGES:
        qb.setProjectionMap(sMessageProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_MSGS);
        if (TextUtils.isEmpty(sortOrder)) {
          orderBy = MessageEntity.DEFAULT_SORT_ORDER;
        }
        break;
      case DATAPOINT_ID:
        qb.appendWhere(DatapointEntity._ID + "=" + uri.getPathSegments().get(1));
      case DATAPOINTS:
        qb.setProjectionMap(sEventProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_DATAPOINTS);
        break;
      case INAPP_ID:
        qb.appendWhere(InAppMessageEntity._ID + "=" + uri.getPathSegments().get(1));
      case INAPPS:
        qb.setProjectionMap(sInAppProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_INAPPS);
        if (TextUtils.isEmpty(sortOrder)) {
          orderBy = InAppMessageEntity.DEFAULT_SORT_ORDER;
        }
        break;
      case USER_ATTRIBUTES:
        qb.setProjectionMap(sUserAttributeProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_USER_ATTRIBUTES);
        break;
      case CAMPAIGN_IDS:
        qb.setProjectionMap(sCampaignListProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_CAMPAIGN_LIST);
        break;
      case DATA_BATCHES:
        qb.setProjectionMap(sBatchDataProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_BATCH_DATA);
        break;
      case DEVICE_TRIGGERS:
        qb.setProjectionMap(sDTProjectionMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS);
        break;
      case ATTRIBUTE_CACHE:
        qb.setProjectionMap(sAttributeCacheMap);
        qb.setTables(DatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE);
        break;
      default:
        Logger.f("Unknown URI " + uri);
    }
    Cursor c = null;
    try {
      c = qb.query(db, projection, selection, selectionArgs, null, null, orderBy, limit);
    }catch (Exception e){

    }
    // Tell the cursor what uri to watch, so it knows when its source data
    // changes
    return c;
  }

  public Uri insert(Uri uri, ContentValues values) {
    if (null == values) return null;
    SQLiteDatabase db = mOpenHelper.getWritableDatabase();
    updateIfRequired(db);
    long rowId = 0;
    Uri newUri = null;
    switch (sUriMatcher.match(uri)) {
      case MESSAGES:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_MSGS, null, values);
        if (rowId > 0) {
          newUri = ContentUris.withAppendedId(MessageEntity.getContentUri(getContext()), rowId);
        }
        break;
      case DATAPOINTS:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_DATAPOINTS, null, values);
        if (rowId > 0) {
          newUri = ContentUris.withAppendedId(DatapointEntity.getContentUri(getContext()), rowId);
        }
        break;
      case INAPPS:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_INAPPS, null, values);
        if (rowId > 0) {
          newUri =
              ContentUris.withAppendedId(InAppMessageEntity.getContentUri(getContext()), rowId);
        }
        break;
      case USER_ATTRIBUTES:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, null, values);
        if (rowId>0){
          newUri = ContentUris.withAppendedId(UserAttributeEntity.getContentUri(getContext()),
              rowId);
        }
        break;
      case CAMPAIGN_IDS:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, null, values);
        if (rowId > 0) {
          newUri =
              ContentUris.withAppendedId(UserAttributeEntity.getContentUri(getContext()), rowId);
        }
        break;
      case DATA_BATCHES:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_BATCH_DATA, null, values);
        if (rowId > 0){
          newUri = ContentUris.withAppendedId(BatchDataEntity.getContentUri(getContext()), rowId);
        }
        break;
      case DEVICE_TRIGGERS:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, null, values);
        if (rowId > 0){
          newUri = ContentUris.withAppendedId(DTEntity.getContentUri(getContext()), rowId);
        }
        break;
      case ATTRIBUTE_CACHE:
        rowId = db.insert(DatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, null, values);
        if (rowId > 0) {
          newUri =
              ContentUris.withAppendedId(AttributeCacheEntity.getContentUri(getContext()), rowId);
        }
        break;
      default:
        Logger.f("Unknown URI " + uri);
    }
    if (null != newUri) {
      Logger.v("MoEProvider: Added new record : " + newUri.toString());
      getContext().getContentResolver().notifyChange(uri, null);
    } else {
      Logger.f("MoEProvider: Failed to add new record");
    }
    return newUri;
  }

  public void updateIfRequired(SQLiteDatabase db) {
    if (MoEConstants.dbUpdateRequired) {
      mOpenHelper.addMSGTagIfRequiredInbox(db);
      mOpenHelper.addUserAttributesTableIfRequired(db);
      mOpenHelper.addCampaignListTableIfRequired(db);
      mOpenHelper.addBatchDataTableIfRequired(db);
      mOpenHelper.addDeviceTriggerTableIfRequired(db);
      mOpenHelper.addAttributeCacheTableIfRequired(db);
      MoEConstants.dbUpdateRequired = false;
    }
  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#applyBatch(java.util.ArrayList)
   */
  @Override @NonNull public ContentProviderResult[] applyBatch(
      @NonNull ArrayList<ContentProviderOperation> operations)
      throws OperationApplicationException {
    ContentProviderResult[] result = new ContentProviderResult[operations.size()];
    int i = 0;
    // Opens the database object in "write" mode.
    SQLiteDatabase db = mOpenHelper.getWritableDatabase();
    // Begin a transaction
    updateIfRequired(db);
    db.beginTransaction();
    try {
      for (ContentProviderOperation operation : operations) {
        // Chain the result for back references
        result[i++] = operation.apply(this, result, i);
      }
      db.setTransactionSuccessful();
    } catch (OperationApplicationException e) {
      Logger.f("MoEProvider : batch failed: ", e);
    }catch (Exception e){
      Logger.f("MoEProvider : batch failed: ", e);
    } finally {
      db.endTransaction();
    }
    return result;
  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#delete(android.net.Uri,
   * java.lang.String, java.lang.String[])
   */
  public int delete(Uri uri, String selection, String[] selectionArgs) {
    if (null == uri) return 0;
    SQLiteDatabase db = mOpenHelper.getWritableDatabase();
    updateIfRequired(db);
    int count = 0;
    switch (sUriMatcher.match(uri)) {
      case MESSAGES:
        count = db.delete(DatabaseHelper.TABLE_NAME_MSGS, selection, selectionArgs);
        break;
      case MESSAGE_ID:
        String noteId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_MSGS,
            MessageEntity._ID + "=" + noteId + (!TextUtils.isEmpty(selection) ? " AND " + selection
                : ""), selectionArgs);
        break;
      case DATAPOINTS:
        count = db.delete(DatabaseHelper.TABLE_NAME_DATAPOINTS, selection, selectionArgs);
        break;
      case DATAPOINT_ID:
        String eventId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_DATAPOINTS,
            DatapointEntity._ID + " = " + eventId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case INAPPS:
        count = db.delete(DatabaseHelper.TABLE_NAME_INAPPS, selection, selectionArgs);
        break;
      case INAPP_ID:
        String inappId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_INAPPS,
            DatapointEntity._ID + "=" + inappId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case USER_ATTRIBUTES:
        count = db.delete(DatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, selection, selectionArgs);
        break;
      case USER_ATTRIBUTES_ID:
        String attributeId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_USER_ATTRIBUTES,
            UserAttributeEntity._ID + "=" + attributeId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case CAMPAIGN_IDS:
        count = db.delete(DatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, selection, selectionArgs);
        break;
      case CAMPAIGN_ID:
        String campaignId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_CAMPAIGN_LIST,
            CampaignListEntity._ID + "=" + campaignId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case DATA_BATCHES:
        count = db.delete(DatabaseHelper.TABLE_NAME_BATCH_DATA, selection, selectionArgs);
        break;
      case DATA_BATCH:
        String batchId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_BATCH_DATA,
            BatchDataEntity._ID + "=" + batchId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case DEVICE_TRIGGERS:
        count = db.delete(DatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, selection, selectionArgs);
        break;
      case DEVICE_TRIGGER:
        String triggerId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS,
            DTEntity._ID + "=" + triggerId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case ATTRIBUTE_CACHE:
        count = db.delete(DatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, selection, selectionArgs);
        break;
      case ATTRIBUTE_CACHE_ID:
        String cachedAttributeId = uri.getPathSegments().get(1);
        count = db.delete(DatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE,
            AttributeCacheEntity._ID + "=" + cachedAttributeId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      default:
        Logger.f("Unknown URI " + uri);
    }
    getContext().getContentResolver().notifyChange(uri, null);
    Logger.v("MoEProvider: Deleted " + count + " record(s) for URI: " + uri.toString());
    return count;
  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#update(android.net.Uri,
   * android.content.ContentValues, java.lang.String, java.lang.String[])
   */
  public int update(Uri uri, ContentValues values, String selection, String[] selectionArgs) {

    SQLiteDatabase db = mOpenHelper.getWritableDatabase();
    updateIfRequired(db);
    int count = 0;
    switch (sUriMatcher.match(uri)) {
      case MESSAGES:
        count = db.update(DatabaseHelper.TABLE_NAME_MSGS, values, selection, selectionArgs);
        break;
      case MESSAGE_ID:
        String msgId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_MSGS, values,
            DatapointEntity._ID + "=" + msgId + (!TextUtils.isEmpty(selection) ? " AND " + selection
                : ""), selectionArgs);
        break;
      case DATAPOINTS:
        count = db.update(DatabaseHelper.TABLE_NAME_DATAPOINTS, values, selection, selectionArgs);
        break;
      case DATAPOINT_ID:
        String eventId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_DATAPOINTS, values,
            DatapointEntity._ID + "=" + eventId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case INAPPS:
        count = db.update(DatabaseHelper.TABLE_NAME_INAPPS, values, selection, selectionArgs);
        break;
      case INAPP_ID:
        String inAppId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_INAPPS, values,
            InAppMessageEntity._ID + "=" + inAppId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case USER_ATTRIBUTES:
        count = db.update(DatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, values, selection,
            selectionArgs);
        break;
      case USER_ATTRIBUTES_ID:
        String attributeId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, values,
            UserAttributeEntity._ID + "=" + attributeId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case CAMPAIGN_IDS:
        count =
            db.update(DatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, values, selection, selectionArgs);
        break;
      case CAMPAIGN_ID:
        String campaignId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, values,
            CampaignListEntity._ID + "=" + campaignId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case DATA_BATCHES:
        count =
            db.update(DatabaseHelper.TABLE_NAME_BATCH_DATA, values, selection, selectionArgs);
        break;
      case DATA_BATCH:
        String batchId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_BATCH_DATA, values,
            BatchDataEntity._ID + "=" + batchId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case DEVICE_TRIGGERS:
        count = db.update(DatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, values, selection, selectionArgs);
        break;
      case DEVICE_TRIGGER:
        String triggerId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, values,
            DTEntity._ID + "=" + triggerId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      case ATTRIBUTE_CACHE:
        count = db.update(DatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, values, selection, selectionArgs);
        break;
      case ATTRIBUTE_CACHE_ID:
        String cachedAttributeId = uri.getPathSegments().get(1);
        count = db.update(DatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, values,
            AttributeCacheEntity._ID + "=" + cachedAttributeId + (!TextUtils.isEmpty(selection) ? " AND "
                + selection : ""), selectionArgs);
        break;
      default:
        Logger.f("Unknown URI " + uri);
    }
    getContext().getContentResolver().notifyChange(uri, null);
    Logger.v("MoEProvider: Updated " + count + " record(s)");
    return count;
  }

  public boolean  isFieldExist(String tableName, String fieldName, SQLiteDatabase db) {
    boolean isExist = false;
    Cursor res = db.rawQuery("PRAGMA table_info(" + tableName + ")", null);
    ArrayList<String> columnNameList = new ArrayList<>();
    try {
      if (res != null && res.moveToFirst()) {
        do {
          String columnName = res.getString(res.getColumnIndex("name"));
          columnNameList.add(columnName);
        } while (res.moveToNext());
      }
    } finally {
      if (res != null) {
        res.close();
      }
    }

    if (columnNameList.contains(fieldName)) {
      isExist = true;
    }
    return isExist;
  }

  public boolean isTableExists(SQLiteDatabase db, String tableName) {
    int count = 0;
    try {
      if (tableName == null || db == null || !db.isOpen()) {
        return false;
      }
      Cursor cursor = db.rawQuery("SELECT COUNT(*) FROM sqlite_master WHERE type = ? AND name = ?",
          new String[] { "table", tableName });
      if (!cursor.moveToFirst()) {
        return false;
      }
      count = cursor.getInt(0);
      cursor.close();
    } catch (Exception e) {
      Logger.f("MoEProvider: isTableExists Exception " + e.toString());
    }
    return count > 0;
  }
}