package com.moengage.location;

import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import com.moengage.core.ConfigurationCache;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;

/**
 * Location Services handler
 * @author Umang Chamaria
 */
public final class GeoManager {

  private LocationHandler handler;

  private GeoManager() {
    //Intentionally made private
    loadHandler();
  }

  private static GeoManager _THIS;

  public static GeoManager getInstance() {
    if (null == _THIS) {
      _THIS = new GeoManager();
    }
    return _THIS;
  }

  private void loadHandler(){
    try {
      handler = (LocationHandler) Class.forName("com.moengage.locationlibrary.LocationHandlerImpl")
              .newInstance();
    } catch (ClassNotFoundException e) {
      Logger.e("Location Handler class Not Found Exception");
    } catch (Exception e) {
      Logger.e("Exception", e);
    }
  }

  @Nullable
  public LocationHandler getHandler(@NonNull Context context) {
    if (null == context) return null;
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isGeofenceEnabled()) return null;
    if (!ConfigurationProvider.getInstance(context).isManifestParsingEnabled() &&
        ConfigurationProvider.getInstance(context).getLocationServiceState()) return handler;
    if (ConfigurationProvider.getInstance(context).isManifestParsingEnabled()) return handler;
    return null;
  }

  public void updateFenceAndLocation(Context context){
    LocationHandler handler = getHandler(context);
      if (handler != null){
        handler.updateFenceAndLocation(context);
    }
  }

  public void removeGeoFences(Context context){
    LocationHandler handler = getHandler(context);
    if (handler != null){
      handler.removeGeoFences(context);
    }
  }

  public interface LocationHandler {

    /**
     * Sets geo-fence if application has location permission
     * @param context Application Context
     * @param response Response from geo-fence API
     */
    void setGeoFences(Context context, String response);

    /**
     * Informs server about geo-fence hit
     * @param context Application Context
     * @param intent Intent received from location services on geo-fence hit
     */
    void onGeoFenceHit(Context context, Intent intent);

    /**
     * Updates user location and fetches new fences
     * @param context Application Context
     */
    void updateFenceAndLocation(Context context);

    void removeGeoFences(Context context);
  }

  public enum TASK_TYPE {
    GET_GEOFENCE, GEOFENCE_HIT
  }
}
