package com.moengage.core.segmentation;

import android.support.annotation.Nullable;
import android.text.format.DateUtils;
import android.text.format.Time;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class FilterEvaluator {

  private static final String TAG = "FilterEvaluator";

  private JSONObject mEventAttributes;
  private JSONObject mConditions;
  private String mAttributeName;
  private String mOperator;
  private String mDataType;
  private Object mValue;
  private boolean mNegate;
  private boolean mCaseSensitive;

  public FilterEvaluator(JSONObject conditions, JSONObject eventAttributes) {
    mEventAttributes = eventAttributes;
    mConditions = conditions;
    try {
      mNegate = mConditions.optBoolean(FilterConstants.CONDITION_ATTR_NEGATE, false);
      mCaseSensitive =
          mConditions.optBoolean(FilterConstants.CONDITION_ATTR_CASE_SENSITIVE, false);
      mDataType = mConditions.getString(FilterConstants.CONDITION_ATTR_DATA_TYPE);
      mOperator = mConditions.getString(FilterConstants.CONDITION_ATTR_OPERATOR);
      mAttributeName = mConditions.getString(FilterConstants.CONDITION_ATTR_ATTRIBUTE_NAME);
      if (mConditions.has(FilterConstants.CONDITION_ATTR_ATTRIBUTE_VALUE)){
        mValue = mConditions.get(FilterConstants.CONDITION_ATTR_ATTRIBUTE_VALUE);
      }
    } catch (Exception e) {
      Logger.f(TAG + "FilterEvaluator() : ", e);
    }
  }

  public boolean evaluate() {
    try {
      if (mAttributeName == null || mOperator == null || mDataType == null) {
        return false;
      }
      if (mOperator.equals(FilterConstants.CONDITION_EXISTS)) {
        boolean result = checkIfAttributesHasAttribute();
        if (mNegate) return !result;
        return result;
      }
      if (!checkIfAttributesHasAttribute()) return false;
      switch (mDataType) {
        case FilterConstants.DATA_TYPE_BOOLEAN:
          return isConditionSatisfiedBoolean();
        case FilterConstants.DATA_TYPE_DOUBLE:
          return isConditionSatisfiedDouble();
        case FilterConstants.DATA_TYPE_STRING:
          return isConditionSatisfiedString();
        case FilterConstants.DATA_TYPE_DATE_TIME:
          return isConditionSatisfiedDateTime();
      }
    } catch (Exception e) {
      Logger.f(TAG + "evaluate() : ", e);
    }
    return false;
  }

  private boolean isConditionSatisfiedString() {
    try {
      Object userValue = getAttributeValue();
      boolean result = false;
      if (userValue == null || mValue == null) return false;

      if (mOperator.equals(FilterConstants.CONDITION_CONTAINS)) {
        String campaignValueString = (String) mValue;
        if (!mCaseSensitive) {
          campaignValueString = campaignValueString.toLowerCase();
        }
        if (userValue instanceof JSONArray) {
          JSONArray userValueArray = (JSONArray) userValue;
          for (int i = 0; i < userValueArray.length(); i++) {
            if (userValueArray.getString(i).contains((String) mValue)) {
              result = true;
              break;
            }
          }
          if (mNegate) return !result;
          return result;
        } else if (userValue instanceof String) {
          String userValueString = (String) userValue;
          if (!mCaseSensitive) {
            userValueString = userValueString.toLowerCase();
            campaignValueString = campaignValueString.toLowerCase();
          }
          result = userValueString.contains(campaignValueString);
          if (mNegate) return !result;
          return result;
        }
        return false;
      } else if (mOperator.equals(FilterConstants.CONDITION_IN)) {
        JSONArray campaignValueArray = (JSONArray) mValue;
        if (campaignValueArray == null) return false;
        if (userValue instanceof JSONArray) {
          JSONArray userValueArray = (JSONArray) userValue;
          for (int i =0; i< campaignValueArray.length(); i++){
            boolean shouldBreak = false;
            for (int j=0; j< userValueArray.length(); j++){
              String userString = userValueArray.getString(j);
              String campaignString = campaignValueArray.getString(i);
              if (!mCaseSensitive){
                userString = userString.toLowerCase();
                campaignString = campaignString.toLowerCase();
              }
              if (userString.equals(campaignString)){
                shouldBreak = true;
                result = true;
                break;
              }
            }
            if (shouldBreak) break;
          }
          if (mNegate) return !result;
          return result;
        } else if (userValue instanceof String) {
          String userValueString = (String) userValue;
          if (!mCaseSensitive) {
            userValueString = userValueString.toLowerCase();
          }
          for (int i = 0; i < campaignValueArray.length(); i++) {
            String campaignValueString = campaignValueArray.getString(i);
            if (!mCaseSensitive) {
              campaignValueString = campaignValueString.toLowerCase();
            }
            if (campaignValueString.equals(userValueString)) {
              result = true;
              break;
            }
          }
          if (mNegate) return !result;
          return result;
        }
      } else {
        if (!(userValue instanceof String) || !(mValue instanceof String)) return false;
        String campaignValueString = (String) mValue;
        String userValueString = (String) userValue;
        if (!mCaseSensitive) {
          userValueString = userValueString.toLowerCase();
          campaignValueString = campaignValueString.toLowerCase();
        }
        if (userValueString == null || campaignValueString == null) return false;
        switch (mOperator) {
          case FilterConstants.CONDITION_IS:
            result = userValueString.equals(campaignValueString);
            if (mNegate) return !result;
            return result;
          case FilterConstants.CONDITION_ENDS_WITH:
            result = userValueString.endsWith(campaignValueString);
            if (mNegate) return !result;
            return result;
          case FilterConstants.CONDITION_STARTS_WITH:
            result = userValueString.startsWith(campaignValueString);
            if (mNegate) return !result;
            return result;
          default:
            return false;
        }
      }
    } catch (Exception e) {
      Logger.f(TAG + "isConditionSatisfiedString() : ", e);
    }
    return false;
  }

  private boolean checkIfAttributesHasAttribute() throws JSONException {
    if (mEventAttributes.has(MoEHelperUtils.EVENT_ATTRS)){
      String eventAttrString = mEventAttributes.getString(MoEHelperUtils.EVENT_ATTRS);
      JSONObject eventAttr = new JSONObject(eventAttrString);
      if (eventAttr.has(mAttributeName)) return true;
    }
    if (mEventAttributes.has(MoEHelperUtils.EVENT_ATTRS_CUST)) {
      try {
        String eventCustAttrString  = mEventAttributes.getString(MoEHelperUtils.EVENT_ATTRS_CUST);
        JSONObject customAttributes = new JSONObject(eventCustAttrString);
        if (!customAttributes.has("timestamp")) return false;
        JSONArray customAttr = customAttributes.getJSONArray("timestamp");
        if (customAttr == null) return false;
        for (int i = 0; i < customAttr.length(); i++) {
          JSONObject attr = customAttr.getJSONObject(i);
          if (attr.has(mAttributeName)) return true;
        }
      } catch (Exception e) {
        Logger.f(TAG + "checkIfAttributesHasAttribute() : ", e);
        return false;
      }
    }
    return false;
  }

  private boolean isConditionSatisfiedDateTime() throws JSONException, ParseException {
    Object userValue = getAttributeValue();
    if (userValue == null) return false;
    if (mValue == null && !mOperator.equals(FilterConstants.CONDITION_TODAY)) return false;
    DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH);
    long campaignDate = 0;
    if (mValue != null){
      campaignDate = format.parse((String) mValue).getTime();
    }
    long userDate = (Long) userValue;
    switch (mOperator) {
      case FilterConstants.CONDITION_ON:
        Time time = new Time();
        time.set(campaignDate);

        int thenYear = time.year;
        int thenMonth = time.month;
        int thenMonthDay = time.monthDay;

        time.set(userDate);
        return (thenYear == time.year)
            && (thenMonth == time.month)
            && (thenMonthDay == time.monthDay);
      case FilterConstants.CONDITION_BETWEEN:
        if (!mConditions.has(FilterConstants.CONDITION_ATTR_ATTRIBUTE_VALUE_ONE)) return false;
        String upperBound = mConditions.getString(FilterConstants
            .CONDITION_ATTR_ATTRIBUTE_VALUE_ONE);
        long upperBoundDate = format.parse(upperBound).getTime();
        boolean result = userDate >= campaignDate && userDate < upperBoundDate;
        if (mNegate) return !result;
        return result;
      case FilterConstants.CONDITION_BEFORE:
        return campaignDate > userDate;
      case FilterConstants.CONDITION_AFTER:
        return userDate > campaignDate;
      case FilterConstants.CONDITION_TODAY:
        return DateUtils.isToday(userDate);
    }
    return false;
  }

  private boolean isConditionSatisfiedBoolean() throws JSONException {
    Object userValue = getAttributeValue();
    if (userValue == null || mValue == null) return false;
    boolean userValueBoolean = (Boolean) userValue;
    boolean campaignValueBoolean = (Boolean) mValue;
    switch (mOperator) {
      case FilterConstants.CONDITION_IS:
        boolean result = userValueBoolean == campaignValueBoolean;
        if (mNegate) return !result;
        return result;
      default:
        return false;
    }
  }

  private boolean isConditionSatisfiedDouble() throws JSONException {
    Object userValue = getAttributeValue();
    if (userValue == null) return false;
    boolean result = false;
    if (mOperator.equals(FilterConstants.CONDITION_IN)) {
      if (!(mValue instanceof JSONArray))return false;
      JSONArray campaignValueArray = (JSONArray) mValue;
      if (userValue instanceof JSONArray){
        JSONArray userValueArray = (JSONArray) userValue;
        for (int i =0; i< campaignValueArray.length(); i++){
          boolean shouldBreak = false;
          for (int j=0; j< userValueArray.length(); j++){
            if (userValueArray.getDouble(j) == campaignValueArray.getDouble(i)){
              shouldBreak = true;
              result = true;
              break;
            }
          }
          if (shouldBreak) break;
        }
        if (mNegate) return !result;
        return result;
      }else {
        double userValueDouble = Double.parseDouble(userValue.toString());
        for (int i=0; i<campaignValueArray.length(); i++){
          if (userValueDouble == campaignValueArray.getDouble(i)){
            result = true;
            break;
          }
        }
        if (mNegate) return !result;
        return result;
      }
    } else {
      double userValueLong = Double.parseDouble(userValue.toString());
      double campaignValueLong = Double.parseDouble(mValue.toString());
      switch (mOperator) {
        case FilterConstants.CONDITION_IS:
          result = userValueLong == campaignValueLong;
          if (mNegate) return !result;
          return result;
        case FilterConstants.CONDITION_GREATER_THAN:
          return userValueLong > campaignValueLong;
        case FilterConstants.CONDITION_LESS_THAN:
          return userValueLong < campaignValueLong;
        case FilterConstants.CONDITION_BETWEEN:
          if (!mConditions.has(FilterConstants.CONDITION_ATTR_ATTRIBUTE_VALUE_ONE)) return false;
          double upperBound = mConditions.getDouble(FilterConstants
              .CONDITION_ATTR_ATTRIBUTE_VALUE_ONE);
          result = userValueLong >= campaignValueLong && userValueLong < upperBound;
          if (mNegate) return !result;
          return result;
        default:
          return false;
      }
    }
  }

  @Nullable private Object getAttributeValue() throws JSONException {
    if (mEventAttributes.has(MoEHelperUtils.EVENT_ATTRS)){
      String eventAttrString = mEventAttributes.getString(MoEHelperUtils.EVENT_ATTRS);
      JSONObject eventAttr = new JSONObject(eventAttrString);
      if (eventAttr.has(mAttributeName)) return eventAttr.get(mAttributeName);
    }
    if (mEventAttributes.has(MoEHelperUtils.EVENT_ATTRS_CUST)){
      String eventCustAttrString  = mEventAttributes.getString(MoEHelperUtils.EVENT_ATTRS_CUST);
      JSONObject customAttributes = new JSONObject(eventCustAttrString);
      JSONArray customAttr = customAttributes.getJSONArray("timestamp");
      for (int i = 0; i < customAttr.length(); i++) {
        JSONObject attr = customAttr.getJSONObject(i);
        if (attr.has(mAttributeName)) return attr.get(mAttributeName);
      }
    }
    return null;
  }
}
