package com.moengage.push;

import android.content.Context;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDispatcher;
import com.moengage.push.PushManager.OnTokenReceivedListener;
import org.json.JSONObject;

/**
 * Class to handle push token processing, includes sending token to the server, provide callback
 * if required.
 *
 * @author Umang Chamaria
 * Date: 28/03/19
 */
public class TokenHandler {

  public static final String REQ_REFRESH = "MOE_REG_REFRESH";
  public static final String TOKEN_BY_MOE = "MoE";
  public static final String REG_ON_APP_OPEN = "REG_ON_APP_OPEN";
  public static final String REQ_DELETE_TOKEN = "MOE_DEL_TOK";
  public static final String REQ_REGISTRATION = "MOE_REG_REQ";

  private final String ATTR_REGISTRATION_BY = "registered_by";
  private final Object lock = new Object();
  private OnTokenReceivedListener tokenListener;
  private static final String TAG = "TokenHandler";
  private static TokenHandler instance = null;

  private TokenHandler(){}

  public static TokenHandler getInstance(){
    if (instance == null){
      synchronized (TokenHandler.class){
        if (instance == null) instance = new TokenHandler();
      }
    }
    return instance;
  }

  /**
   * Process Push Token
   *
   * @param context instance of {@link Context}
   * @param token Push Token
   * @param pushRegisteredBy Push registration is handled by App or MoEngage
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  public void processToken(Context context, String token, String pushRegisteredBy) {
    if (TextUtils.isEmpty(token)) return;
    Logger.v(TAG + " processToken() : Will try to process push token. Token: " + token + "\n"
        + "registered by: " + pushRegisteredBy);
    try {
      synchronized (lock) {
        token = ripMultiplexingExtras(token);
        if (tokenListener != null) tokenListener.onTokenReceived(token);
        String oldRegId = ConfigurationProvider.getInstance(context).getGCMToken();
        boolean update = tokenRefreshRequired(context, token);
        if (update) {
          PayloadBuilder payloadBuilder = new PayloadBuilder();
          payloadBuilder.putAttrString(ATTR_REGISTRATION_BY, pushRegisteredBy);
          payloadBuilder.setNonInteractive();
          MoEHelper.getInstance(context)
              .trackEvent(MoEHelperConstants.TOKEN_EVENT, payloadBuilder);
          trackDeviceAttributeForRegistration(context, pushRegisteredBy);
        }
        boolean shouldUpdateToken =
            update || !ConfigurationProvider.getInstance(context).hasSentFcmTokenToServer();
        if (shouldUpdateToken) {
          ConfigurationProvider.getInstance(context).setGCMToken(token);
          MoEDispatcher.getInstance(context).getDeviceAddManager().registerFcmToken(context);
        }
        Logger.v(TAG + " processToken() oldId: = "
            + oldRegId
            + " token = "
            + token
            + " --updating[true/false]: "
            + shouldUpdateToken);
      }
    } catch (Exception e) {
      Logger.f(TAG + " processToken() : Exception ", e);
    }
  }

  private static final String ID_PREFIX = "|ID|";

  private String ripMultiplexingExtras(String token) {
    return !TextUtils.isEmpty(token) && token.startsWith(ID_PREFIX) ? token.substring(7) : token;
  }

  private boolean tokenRefreshRequired(Context context, String newToken) {
    if (!TextUtils.isEmpty(newToken)) {
      String oldToken = ConfigurationProvider.getInstance(context).getGCMToken();
      return TextUtils.isEmpty(oldToken) || !newToken.equals(oldToken);
    } else {
      return false;
    }
  }

  private void trackDeviceAttributeForRegistration(Context context, String pushRegisteredBy) {
    try {
      JSONObject attribute = new JSONObject();
      attribute.put(MoEConstants.PUSH_REGISTRATION_ATTRIBUTE, pushRegisteredBy);
      MoEDispatcher.getInstance(context).setDeviceAttribute(attribute);
    } catch (Exception e) {
      Logger.f("PushManager: trackDeviceAttributeForRegistration() : ", e);
    }
  }

  void setTokenListener(OnTokenReceivedListener listener){
    tokenListener = listener;
  }
}
