package com.moengage.core;

import android.app.Activity;
import android.app.Fragment;
import android.app.job.JobParameters;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.analytics.AnalyticsHelper;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.executor.ITask;
import com.moengage.core.executor.OnTaskCompleteListener;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskProcessor;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.listeners.OnAppBackgroundListener;
import com.moengage.core.listeners.OnLogoutCompleteListener;
import com.moengage.core.pushamp.PushAmpManager;
import com.moengage.core.mipush.MiPushManager;
import com.moengage.core.remoteconfig.ConfigApiNetworkTask;
import com.moengage.core.reports.ReportsBatchHelper;
import com.moengage.core.userattributes.MoEAttributeManager;
import com.moengage.inapp.InAppController;
import com.moengage.location.GeoManager;
import com.moengage.push.PushHandler;
import com.moengage.push.PushManager;
import com.moengage.push.TokenHandler;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.json.JSONObject;

import static com.moe.pushlibrary.MoEHelper.isAppInForeground;

/**
 * @author Umang Chamaria
 */
public class MoEDispatcher implements OnTaskCompleteListener {

  private static final String TAG = "MoEDispatcher";

  private Context context;

  private ConfigurationProvider configProvider;

  private TaskProcessor taskProcessor;

  private static MoEDispatcher instance;

  //flag to delete data after interaction data is sent.
  private boolean shouldClearData = false;

  /**
   * List of all the running tasks
   */
  private HashMap<String, Boolean> runningTaskList;

  private boolean shouldTrackUniqueId = false;

  private JSONObject uniqueIdAttribute = null;

  private ScheduledExecutorService mScheduler;

  private OnLogoutCompleteListener logoutCompleteListener;

  private MoEAttributeManager attributeManager = null;

  private DeviceAddManager deviceAddManager = null;

  private MoECoreEvaluator coreEvaluator = null;

  private ReportsBatchHelper batchHelper = null;

  /**
   * MoEDispatcher constructor. There should be only one instance of
   * MoEDispatcher per task instance of an APP
   *
   * @param context Application Context
   */
  private MoEDispatcher(Context context) {
    if (context != null) {
      this.context = context;
      configProvider = ConfigurationProvider.getInstance(this.context);
      taskProcessor = TaskProcessor.getInstance();
      runningTaskList = new HashMap<>();
      //checkAndAddDevice();
      taskProcessor.setOnTaskCompleteListener(this);
      attributeManager = new MoEAttributeManager(context);
    } else {
      Logger.e("MoEDispatcher  : context is null");
    }
  }

  public static MoEDispatcher getInstance(Context context) {
    if (instance == null) {
      synchronized (MoEDispatcher.class) {
        if (instance == null) instance = new MoEDispatcher(context);
      }
    }
    return instance;
  }

  /**
   * Forcefully try to show in-apps
   *
   * @param force if true tries to show in-app
   */
  public void checkForInAppMessages(boolean force) {
    Logger.v("MoEDispatcher: showInAppIfPossible: Check in app messages");
    if (force) {
      InAppController.getInstance().showInAppIfEligible(context);
    }
  }

  /**
   * Lifecycle callback of an activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   * @param intent intent calling that activity
   */
  public void onStart(Activity activity, Intent intent) {
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    if (null == activity) {
      Logger.e("MoEDispatcher:onStart activity instance is null");
      return;
    }
    if (null == intent) {
      intent = activity.getIntent();
    }
    context = activity.getApplicationContext();
      Logger.v("MoEDispatcher:onStart ----");
      MoEHelperUtils.dumpIntentExtras(intent);
    String currentActivityName = activity.getClass().getName();

    addTaskToQueue(new ActivityStartTask(activity));

    Context context = activity.getApplicationContext();
    int currentOrientation = context.getResources().getConfiguration().orientation;
    //SYNC INAPPS
    String savedActivityName = InAppController.getInstance().getActivityName();
    int savedOrientation = InAppController.getInstance().getActivityOrientation();
    if (savedActivityName != null && savedOrientation != -1) {
      if (savedActivityName.equals(currentActivityName) && savedOrientation != currentOrientation) {
        InAppController.getInstance().showInAppOnConfigurationChange(this.context);
      } else {
        //sync for in-apps and save current orientation and name
        saveCurrentActivityDetails(currentActivityName, currentOrientation);
        syncInAppsAndGeo();
      }
    } else {
      //sync for in-apps and save current orientation and name
      saveCurrentActivityDetails(currentActivityName, currentOrientation);
      syncInAppsAndGeo();
    }
    saveCurrentActivityDetails(currentActivityName, currentOrientation);

    if (MoEHelper.getActivityCounter() == 1) {
      pushTokenFallBack();
    }
    MoEUtils.updateTestDeviceState(this.context);
  }

  private void pushTokenFallBack() {
    PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      pushHandler.setPushRegistrationFallback(context);
    }
  }

  private void syncInAppsAndGeo() {
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    Logger.v("MoEDispatcher: Fetch or query in app message");
    // sync in-app
    InAppController.getInstance().showInAppIfEligible(context);
    //sync geo fences
    GeoManager.getInstance().updateFenceAndLocation(context);
  }

  /**
   * Logs impression for notification click
   *
   * @param gtime notification clicked time
   */
  public void trackNotificationClicked(long gtime) {
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    addTaskToQueue(new NotificationClickedTask(context, gtime));
  }

  /**
   * Saves a user attribute
   *
   * @param userJson user attribute in json format
   */
  public void setUserAttribute(JSONObject userJson) {
    attributeManager.setUserAttribute(userJson);
  }

  public void setCustomUserAttribute(JSONObject userJson) {
    attributeManager.setCustomUserAttribute(userJson);
  }

  /**
   * Tells the sdk whether it is an app update or install when the 1st app with MoEngage goes live.
   *
   * @param existing true if its an update else false
   */
  public void setExistingUser(boolean existing) {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
      boolean installRegistered =
          ConfigurationProvider.getInstance(context).isInstallRegistered();
      int currentVersion = configProvider.getAppVersion();
      if (existing) {
        int prevVersion = configProvider.getStoredAppVersion();
        if (currentVersion == prevVersion) {
          //already stored
          return;
        }
        configProvider.storeAppVersion(currentVersion);
        PayloadBuilder eventObj =
            new PayloadBuilder().putAttrInt(MoEHelperConstants.FROM_VERSION, prevVersion)
                .putAttrInt(MoEHelperConstants.TO_VERSION, currentVersion)
                .putAttrDate(MoEHelperConstants.TIME_OF_UPDATE, new Date());
        MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_UPD, eventObj);
        Logger.v("MoEDispatcher:setExistingUser:tracking update");
      } else if (!installRegistered) {
        configProvider.storeAppVersion(currentVersion);
        PayloadBuilder builder = new PayloadBuilder();
        builder.putAttrInt(MoEHelperConstants.VERSION, currentVersion)
            .putAttrInt(MoEConstants.GENERIC_PARAM_V2_KEY_LIBVERSION,
                MoEHelperConstants.LIB_VERSION)
            .putAttrLong(MoEHelperConstants.TIME_OF_INSTALL, System.currentTimeMillis())
            .putAttrString(MoEConstants.GENERIC_PARAM_V2_KEY_OS,
                MoEConstants.GENERIC_PARAM_V2_VALUE_OS);
        MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_INSTALL, builder);
        Logger.v("MoEDispatcher:setExistingUser:tracking install");
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: setExistingUser: ", e);
    }
  }

  /**
   * Cancel gcm registration fallback.
   */
  void cancelRegistrationFallback() {
    MoEUtils.setRegistrationScheduled(context, false);
    MoEUtils.saveCurrentExponentialCounter(context, 1);
  }

  public void onResume(Activity activity, boolean isRestoring) {
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    if (!isRestoring) {
      showDialogAfterPushClick(activity);
    }
  }

  /**
   * Lifecycle callback of activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   * @param configChange true if orientation change, else false
   */
  public void onStop(Activity activity, boolean configChange) {
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    if (null == activity) return;
    if (configChange) return;
    addTaskToQueue(new ActivityStopTask(context, activity.getClass().getName()));
  }

  /**
   * Logs out the user. Clears the data and tries to re-register for push in case push
   * registration is handled by MoEngage.
   */
  @RestrictTo(Scope.LIBRARY)
  @WorkerThread public void handleLogout(boolean isForcedLogout) {
    Logger.i("Started logout process");
    if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    trackLogoutEvent(isForcedLogout);
    addTaskToQueueBeginning(new SendInteractionDataTask(context));
    shouldClearData = true;
  }

  private void trackLogoutEvent(boolean isForcedLogout) {
    try {
      PayloadBuilder eventAttributes = new PayloadBuilder();
      if (isForcedLogout) eventAttributes.putAttrString("type", "forced");
      eventAttributes.setNonInteractive();
      Event event =
          new Event(MoEConstants.LOGOUT_EVENT, eventAttributes.build());
      MoEDAO.getInstance(context).addEvent(event);
    } catch (Exception e) {
      Logger.f("MoEDispatcher: trackLogoutEvent(): ", e);
    }
  }


  @WorkerThread private void clearDataOnLogout() {
    //truncate all tables
    context.getContentResolver()
        .delete(MoEDataContract.DatapointEntity.getContentUri(context), null, null);
    context.getContentResolver()
        .delete(MoEDataContract.MessageEntity.getContentUri(context), null, null);
    context.getContentResolver()
        .delete(MoEDataContract.InAppMessageEntity.getContentUri(context), null, null);
    context.getContentResolver()
        .delete(MoEDataContract.UserAttributeEntity.getContentUri(context), null, null);
    context.getContentResolver().delete(MoEDataContract.CampaignListEntity.getContentUri
        (context), null, null);
    context.getContentResolver().delete(MoEDataContract.BatchDataEntity.getContentUri(context),
        null, null);
    context.getContentResolver().delete(MoEDataContract.DTEntity.getContentUri(context), null,
        null);
    MoEDAO.getInstance(context).clearAttributeCache();
    //remove shared preferences associated with a user
    configProvider.removeUserConfigurationOnLogout();

    AnalyticsHelper.getInstance(context).onLogout(context);
    notifyLogoutCompleteListener();
    PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      //pushHandler.deleteToken(context, null);
      pushHandler.registerForPushToken(context);
    }
    deviceAddManager.registerDevice(context);
    shouldClearData = false;
    Logger.i("Completed logout process");
  }

  /**
   * Sends all the tracked events to the server.
   */
  public void sendInteractionData() {
    startTask(new SendInteractionDataTask(context));
  }

  public void sendInteractionData(OnJobComplete jobComplete, JobParameters parameters){
    startTask(new SendInteractionDataTask(context, jobComplete, parameters));
  }

  /**
   * Equivalent to the lifecycle callback of {@link Fragment}. Required to show in-apps in
   * fragment.
   *
   * @param activity Parent Activity of the fragment
   * @param fragmentName Name of the calling fragment
   */
  public void onFragmentStart(Activity activity, String fragmentName) {
    syncInAppsAndGeo();
  }

  /**
   * Updates the local DB that the specified message has been clicked<br>
   * <b>Note : Don't call on UI Thread</b>.
   *
   * @param id The id associated with the inbox message that was clicked
   */
  @WorkerThread public void setInboxMessageClicked(long id) {
    MoEDAO.getInstance(context).setMessageClicked(id);
  }

  /**
   * Handles the app update event, registers,and tries to get new GCM registration ID<br>
   * <b>Note : Don't call on UI Thread</b>
   */
  @WorkerThread void handleAppUpdateEvent() {
    //Logging an update event
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
      int prevVersion = configProvider.getStoredAppVersion();
      PayloadBuilder eventObj = new PayloadBuilder();
      eventObj.putAttrInt(MoEHelperConstants.FROM_VERSION, prevVersion);
      eventObj.putAttrInt(MoEHelperConstants.TO_VERSION, configProvider.getAppVersion());
      //update event
      Logger.i("Adding an update event");
      MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_UPD, eventObj);
      if (!isAppInForeground()) {
        sendInteractionData();
      }
    } catch (Exception e) {
      Logger.f("Adding update event", e);
    }
  }

  /**
   * Get a all inbox messages<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return A populated cursor with the inbox messages or null
   */
  @Nullable @WorkerThread public Cursor getAllMessages() {
    return MoEDAO.getInstance(context).getMessages(context);
  }

  /**
   * Returns the number of unread messages. Unread messages are counted based on clicks<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return The Count of unread messages
   */
  @WorkerThread public int getUnreadMessageCount() {
    return MoEDAO.getInstance(context).getUnreadMessageCount();
  }

  /**
   * Initializes the MoEngage SDK
   *
   * @param senderId GCM Project's Sender ID
   * @param appId Application Id from the MoEngage Dashboard
   */

  public void initialize(final String senderId, final String appId) {
    if (!TextUtils.isEmpty(appId)) {
      configProvider.saveAppDetails(senderId, appId);
      //force registration for push
      dispatchPushTask(TokenHandler.REQ_REGISTRATION);
    } else {
      Logger.e("MoEDispatcher: initialize : AppId is null");
    }
  }

  public void checkAndShowLinkedInApp(String campaignId) {
    InAppController.getInstance().fetchLinkedInApp(context, campaignId);
  }

  /**
   * Show a dialog after push was clicked
   *
   * @param activity The instance of the activity where the dialog needs to be created
   */
  private void showDialogAfterPushClick(Activity activity) {
    if (null == activity) return;
    try {
      Intent intent = activity.getIntent();
      if (intent != null) {
        Bundle extras = intent.getExtras();
        if (extras != null && extras.containsKey(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG)) {
          //remove the extra
          intent.removeExtra(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG);
          if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_COUPON_CODE)) {
            MoEUtils.showCouponDialog(extras.getString(MoEHelperConstants.GCM_EXTRA_CONTENT),
                extras.getString(MoEHelperConstants.GCM_EXTRA_COUPON_CODE), activity);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_CONTENT);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_COUPON_CODE);
          } else {
            MoEUtils.showNormalDialogWithOk(extras.getString(MoEHelperConstants.GCM_EXTRA_CONTENT),
                activity);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_CONTENT);
          }
        }
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: showDialogAfterPushClick : ", e);
    }
  }

  /**
   * (non-JavaDoc)
   * Save the activity name and activity orientation.
   * Details required for showing notifications on orientation change
   *
   * @param currActivityName activity name to be saved
   * @param currentOrientation activity orientation name
   */
  private void saveCurrentActivityDetails(String currActivityName, int currentOrientation) {
    InAppController.getInstance().setActivityName(currActivityName);
    InAppController.getInstance().setActivityOrientation(currentOrientation);
  }

  /**
   * Checks whether the task is synchronous or not and adds to the processing queue accordingly.
   * If task is synchronous and is in runningTaskList do not add to processing queue else add
   * in processing queue
   *
   * @param task task to be add to the processor
   */
  public void addTaskToQueue(ITask task) {
    Logger.v("Trying to add " + task.getTaskTag() + " to the queue");
    if (canAddTaskToQueue(task)){
      Logger.v(task.getTaskTag() + " added to queue");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      taskProcessor.addTask(task);
    }else {
      Logger.v(TAG + " addTaskToQueue() : Task is already queued. Cannot add it to queue.");
    }
  }

  /**
   * Checks whether the task is synchronous or not and adds to the processing queue accordingly.
   * If task is synchronous and is in runningTaskList do not add to processing queue else add
   * to front of processing queue
   *
   * @param task task to be add to the processor
   */
  public void addTaskToQueueBeginning(ITask task) {
    Logger.v("Trying to add " + task.getTaskTag() + " to the queue");
    if (canAddTaskToQueue(task)){
      Logger.v(task.getTaskTag() + " added to beginning of queue");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      taskProcessor.addTaskToFront(task);
    }else {
      Logger.v(TAG + " addTaskToQueueBeginning() : Task is already queued. Cannot add it to queue.");
    }
  }

  public void startTask(ITask task){
    Logger.v(TAG + " startTask() : Try to start task");
    if (canAddTaskToQueue(task)) {
      Logger.v(task.getTaskTag() + " Starting task");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      taskProcessor.startTask(task);
    } else {
      Logger.v(
          TAG + " startTask() : Cannot start task. Task is already in progress or queued.");
    }
  }

  @Override public void onTaskComplete(String tag, TaskResult taskResult) {
    // removing task from running list
    Logger.v("Task completed : " + tag);
    if (runningTaskList.containsKey(tag)) {
      runningTaskList.remove(tag);
    }
    switch (tag) {
      case SDKTask.TAG_SEND_INTERACTION_DATA:
        if (shouldClearData) {
          clearDataOnLogout();
          if (shouldTrackUniqueId) trackChangedUniqueId();
        }
        break;
      case SDKTask.TAG_TRACK_ATTRIBUTE:
        if (!taskResult.isSuccess()) {
          shouldTrackUniqueId = true;
          uniqueIdAttribute = (JSONObject) taskResult.getPayload();
        }
        break;
      case SDKTask.TAG_DEVICE_ADD:
        deviceAddManager.processTaskResult(context, taskResult);
        break;
      case SDKTask.TAG_SYNC_CONFIG_API:
        onConfigApiSyncComplete(taskResult);
        break;
    }
  }

  private void dispatchPushTask(String extra) {
    PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      pushHandler.offLoadToWorker(context, extra);
    }
  }

  public void logPushFailureEvent(Context context, String e) {
    // intentionally not removed. Can cause crashes when main sdk and add on not updated in tandem.
  }

  private void syncConfigIfRequired() {
    if (configProvider.getLastConfigSyncTime() + 3600000 < MoEUtils.currentTime()) {
      startTask(new ConfigApiNetworkTask(context));
    }
  }

  public void trackDeviceLocale() {
    try {
      if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY", Locale.getDefault().getCountry());;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_DISPLAY", Locale.getDefault().getDisplayCountry());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE", Locale.getDefault().getLanguage());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_DISPLAY",
          Locale.getDefault().getDisplayLanguage());
      trackDeviceAndUserAttribute("LOCALE_DISPLAY", Locale.getDefault().getDisplayName());
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_ ISO3", Locale.getDefault().getISO3Country());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_ISO3", Locale.getDefault().getISO3Language());
    } catch (Exception e) {
      Logger.f("MoEDispatcher : trackDeviceLocale", e);
    }
  }

  private void trackDeviceAndUserAttribute(String attrName, String attrValue) {
    try {
      JSONObject attribute = new JSONObject();
      attribute.put(attrName, attrValue);
      setUserAttribute(attribute);
      //setDeviceAttribute(attribute);
    } catch (Exception e) {
      Logger.f("MoEDispatcher: trackDeviceAndUserAttribute() ", e);
    }
  }

  public void logoutUser(final boolean isForcedLogout) {
    try {
      Bundle extras = new Bundle();
      extras.putBoolean(MoEConstants.SERVICE_LOGOUT_TYPE, isForcedLogout);
      addTaskToQueue(new MoEWorkerTask(context, MoEConstants.SERVICE_TYPE_LOGOUT, extras));
    } catch(Exception e){
      Logger.f("MoEDispatcher: logoutUser() ", e);
    }
  }

  private void trackChangedUniqueId() {
    if (uniqueIdAttribute != null) {
      setUserAttribute(uniqueIdAttribute);
      uniqueIdAttribute = null;
      shouldTrackUniqueId = false;
    }
  }

  public void setAlias(JSONObject aliasJSON){
   if (!ConfigurationCache.getInstance().getRemoteConfiguration().isAppEnabled()) return;
    addTaskToQueue(new SetAliasTask(context, aliasJSON));
  }

  public void setDeviceAttribute(JSONObject deviceAttribute){
    addTaskToQueue(new SetDeviceAttributeTask(context, deviceAttribute));
  }

  private void schedulePeriodicFlushIfRequired() {
    try {
      if (ConfigurationCache.getInstance().getRemoteConfiguration().isPeriodicFlushEnabled() && MoEHelper.getInstance(context)
          .getPeriodicSyncState()) {
        Runnable syncRunnable = new Runnable() {
          @Override public void run() {
            Logger.v("MoEDispatcher: schedulePeriodicFlushIfRequired() inside runnable, will sync "
                + "now");
            sendInteractionData();
          }
        };
        long timeDelay = ConfigurationCache.getInstance().getRemoteConfiguration().getPeriodicFlushTime();
        if (MoEHelper.getInstance(context).getFlushInterval() > timeDelay){
          timeDelay = MoEHelper.getInstance(context).getFlushInterval();
        }
        Logger.v("MoEDispatcher: schedulePeriodicFlushIfRequired() scheduling periodic sync");
        mScheduler = Executors.newScheduledThreadPool(1);
        mScheduler.scheduleWithFixedDelay(syncRunnable, timeDelay, timeDelay, TimeUnit.SECONDS);
      }
    } catch (Exception e) {
      Logger.e("MoEDispatcher: schedulePeriodicFlushIfRequired() ", e);
    }
  }

  private void shutDownPeriodicFlush() {
    try {
      if (ConfigurationCache.getInstance().getRemoteConfiguration().isPeriodicFlushEnabled() && MoEHelper.getInstance(context)
          .getPeriodicSyncState() && mScheduler != null) {
        Logger.v("MoEDispatcher: shutDownPeriodicFlush() shutting down periodic flush");
        mScheduler.shutdownNow();
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: shutDownPeriodicFlush() ", e);
    }
  }

  public void onAppOpen() {
    try {
      syncConfigIfRequired();

      InAppController.getInstance().syncInAppIfRequired(context);
      PushAmpManager.getInstance().forceServerSync(context, true);

      PushHandler pushHandler = PushManager.getInstance().getPushHandler();
      if (pushHandler != null){
        pushHandler.offLoadToWorker(context, TokenHandler.REG_ON_APP_OPEN);
      }
      schedulePeriodicFlushIfRequired();
      MoEDTManager.getInstance().forceSyncDeviceTriggers(context);
      updateFeatureConfigForOptOutIfRequired();
      MiPushManager.getInstance().initMiPush(MoEHelper.getInstance(context).getApplication());
    } catch (Exception e) {
      Logger.f("MoEDispatcher: onAppOpen() ", e);
    }
  }

  void updateFeatureConfigForOptOutIfRequired(){
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    if (provider.isDataTrackingOptedOut()){
      // gaid optOut
      provider.optOutOfAdIdCollection(true);
      //androidId optOut
      provider.optOutOfAndroidIdCollection(true);
      //location optOut
      provider.optOutOfTrackLocation(true);
      //geo-fence optOut
      provider.optOutOfSetGeoFence(true);
      //device attributes optOut
      provider.optOutOfDeviceAttributesCollection(true);
      //setting location services state
      provider.setLocationServicesState(false);
    }
    if (provider.isPushNotificationOptedOut()) {
      provider.clearPushToken();
    }
  }

  public void setOnLogoutCompleteListener(OnLogoutCompleteListener listener) {
    logoutCompleteListener = listener;
  }

  public void removeOnLogoutCompleteListener() {
    logoutCompleteListener = null;
  }

  private void notifyLogoutCompleteListener() {
    Logger.v(TAG + " notifyLogoutCompleteListener() : Notifying listeners");
    if (logoutCompleteListener != null){
      logoutCompleteListener.logoutComplete();
    }
  }

  void onAppClose() {
    Logger.v(TAG + " onAppClose(): Application going to background.");
    // notify when app goes to background
    notifyOnAppBackground();
    // retry device add if token and app id is available
    getDeviceAddManager().retryDeviceRegistrationIfRequired(context);
    // shutdown periodic flush
    shutDownPeriodicFlush();
    // save in-app check failure counters
    InAppController.getInstance().writeInAppCheckFailureCounterToStorage(context);
    //schedule device triggers
    MoEDTManager.getInstance().scheduleBackgroundSync(context);
    // save sent screens
    configProvider.storeSentScreenList();
    //schedule message pull
    PushAmpManager.getInstance().scheduleServerSync(context);

    // track exit intent
    trackAppExit();
    // update last active time
    AnalyticsHelper.getInstance(context).onAppClose(context);
  }

  private void notifyOnAppBackground(){
    OnAppBackgroundListener listener = MoEHelper.getInstance(context).getOnAppBackgroundListener();
    if (listener != null){
      listener.goingToBackground();
    } else {
      Logger.v(TAG + " execute() : on app background listener not set cannot provide callback");
    }
  }

  private void trackAppExit(){
    MoEHelper.getInstance(context).trackEvent(MoEConstants.MOE_APP_EXIT_EVENT, new PayloadBuilder());
  }

  public MoEAttributeManager getAttributeManager() {
    return attributeManager;
  }

  public DeviceAddManager getDeviceAddManager() {
    if (deviceAddManager == null) {
      deviceAddManager = new DeviceAddManager();
    }
    return deviceAddManager;
  }

  public MoECoreEvaluator getCoreEvaluator() {
    if (coreEvaluator == null) {
      coreEvaluator = new MoECoreEvaluator();
    }
    return coreEvaluator;
  }

  public ReportsBatchHelper getBatchHelper(){
    if (batchHelper == null){
      batchHelper = new ReportsBatchHelper();
    }
    return batchHelper;
  }

  private boolean canAddTaskToQueue(ITask task) {
    if (!task.isSynchronous()) return true;
    return !runningTaskList.containsKey(task.getTaskTag());
  }

  private void onConfigApiSyncComplete(TaskResult taskResult) {
    if (taskResult == null || !taskResult.isSuccess()) return;
    if (MoEUtils.canEnableMiPush(ConfigurationCache.getInstance().getRemoteConfiguration())){
      ConfigurationCache.getInstance().setPushService(MoEConstants.PUSH_SERVICE_MI_PUSH);
      MiPushManager.getInstance().initMiPush(MoEHelper.getInstance(context).getApplication());
    }else {
      configProvider.saveMiPushToken("");
      configProvider.setMiTokenServerSentState(false);
      ConfigurationCache.getInstance().setPushService(MoEConstants.PUSH_SERVICE_FCM);
    }
  }
}
