package com.moengage.core.executor;

import android.support.annotation.NonNull;
import android.text.TextUtils;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;

/**
 * @author Umang Chamaria on 11/16/15.
 */
public class TaskProcessor {

  private BlockingDeque<ITask> taskQueue;

  private ExecutorService executorService = Executors.newCachedThreadPool();

  private ArrayList<WeakReference<OnTaskCompleteListener>> taskCompleteListeners;

  private ITask active;

  private final Object lock = new Object();

  private static TaskProcessor instance = null;

  private TaskProcessor() {
    taskQueue = new LinkedBlockingDeque<>();
    taskCompleteListeners = new ArrayList<>();
  }

  public static TaskProcessor getInstance(){
    if (instance == null){
      synchronized (TaskProcessor.class){
        if (instance == null) instance = new TaskProcessor();
      }
    }
    return instance;
  }

  public void addTask(ITask task) {
    if (task != null) {
      taskQueue.add(task);
      startExecution();
    }
  }

  public void addTaskToFront(ITask task) {
    if (task != null) {
      taskQueue.addFirst(task);
      startExecution();
    }
  }

  public void startExecution() {
    if (active == null) {
      scheduleNext();
    }
  }

  private void scheduleNext() {
    if ((active = taskQueue.poll()) != null) {
      executorService.submit(new Runnable() {
        @Override
        public void run() {
          executeTask(active);
          scheduleNext();
        }
      });
    }
  }

  public void startTask(final ITask task){
    executorService.submit(new Runnable() {
      @Override public void run() {
        executeTask(task);
      }
    });
  }
  
  private void executeTask(ITask task) {
    TaskResult result = task.execute();
    String action = task.getTaskTag();
    if (!TextUtils.isEmpty(action))
      notifyListener(action, result);

    task.onPostExecute(result);
  }

  public void removeOnTaskCompleteListener(@NonNull OnTaskCompleteListener listener) {
    if (taskCompleteListeners != null && listener != null) {
      int index = taskCompleteListeners.indexOf(listener);
      if (index != -1) taskCompleteListeners.remove(index);
    }
  }

  public void setOnTaskCompleteListener(OnTaskCompleteListener listener) {
    taskCompleteListeners.add(new WeakReference<OnTaskCompleteListener>(listener));
  }

  public void notifyListener(String tag, TaskResult result) {
    synchronized (lock) {
      if (taskCompleteListeners != null) {
        for (WeakReference<OnTaskCompleteListener> taskCompleteListener : taskCompleteListeners){
          if (taskCompleteListener.get() != null){
            taskCompleteListener.get().onTaskComplete(tag, result);
          }
        }
      }
    }
  }
}