package com.moengage.inapp;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import java.util.HashMap;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class InAppController {

  public enum NETWORK_CALL_TYPE {

    AUTO_TRIGGER_EVENT,
    SYNC_IN_APPS,
    SINGLE_FETCH
  }

  private InAppHandler inAppHandler;

  /**
   * orientation of the activity in which in-app is being shown
   */
  private int mActivityOrientation = -1;
  /**
   * Name of activity in which in-app was being shown
   */
  private String mActivityName = null;
  private static InAppController _INSTANCE;

  private InAppController() {
    //check for PushHandler on class path
    loadInAppHandler();
  }

  private void loadInAppHandler() {
    try {
      Class handler = Class.forName("com.moengage.inapp.InAppHandlerImpl");
      inAppHandler = (InAppHandler) handler.newInstance();
      Logger.v("InAppController:loadInAppHandler InApp Module present");
    } catch (Exception e) {
      Logger.e("InAppController : loadInAppHandler : InApp Module not present " + e.getMessage());
    }
  }

  @Deprecated
  public InAppHandler getInAppHandler() {
    return inAppHandler;
  }

  @Nullable
  private InAppHandler getInAppHandler(Context context){
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    if (provider.isInAppOptedOut() || !provider.isInAppEnabled()) return null;
    return inAppHandler;
  }

  public static InAppController getInstance() {
    if (_INSTANCE == null) {
      _INSTANCE = new InAppController();
    }
    return _INSTANCE;
  }

  public String getActivityName() {
    return mActivityName;
  }

  public void setActivityName(String activityName) {
    mActivityName = activityName;
  }

  public int getActivityOrientation() {
    return mActivityOrientation;
  }

  public void setActivityOrientation(int activityOrientation) {
    mActivityOrientation = activityOrientation;
  }

  public void trackAPIFailure(String apiType){
    InAppHandler inAppHandler = getInAppHandler();
    if (inAppHandler != null){
      inAppHandler.logInAppAPIFailure(apiType);
    }
  }

  public static String SYNC_API_FAILURE = "sync_api_fail";
  public static String SMART_API_FAILURE = "smart_api_fail";
  public static String SINGLE_API_FAILURE = "single_fetch_api_failure";

  public void writeInAppCheckFailureCounterToStorage(Context context){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.writeInAppCheckFailureCounterToStorage(context);
    }
  }

  public void registerInAppManager(Activity activity){
    InAppHandler inAppHandler = getInAppHandler(activity.getApplicationContext());
    if (inAppHandler != null){
      inAppHandler.registerInAppManager(activity);
    }
  }

  public void unregisterInAppManager(Activity activity){
    InAppHandler inAppHandler = getInAppHandler(activity.getApplicationContext());
    if (inAppHandler != null){
      inAppHandler.unregisterInAppManager(activity);
    }
  }

  public void showInAppIfEligible(Context context){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.showInAppIfEligible(context);
    }
  }

  public void showInAppOnConfigurationChange(Context context){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.showInAppOnConfigurationChange(context);
    }
  }

  public void fetchLinkedInApp(Context context, String campaignId){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null) {
      inAppHandler.fetchLinkedInApp(context, campaignId);
    }
  }

  public void registerAutoTriggerEvent(Context context, Event event){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null) {
      inAppHandler.registerAutoTriggerEvent(context, event);
    }
  }

  public void parseAndSaveInApps(Context context, JSONObject jsonResponse){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.parseAndSaveInApps(jsonResponse, context);
    }
  }

  public void tryShowAutoTriggerInApp(Context context, JSONObject jsonResponse){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.tryShowAutoTriggerInApp(context, jsonResponse);
    }
  }

  public void showLinkedInApp(Context context, JSONObject jsonResponse, HashMap<String, String>
      paramsMap){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.showLinkedInApp(context, jsonResponse, paramsMap);
    }
  }

  public void showTestInAppErrorDialog(Context context, String errorString){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.showTestInAppErrorDialog(errorString);
    }
  }

  public void syncInAppIfRequired(Context context){
    InAppHandler inAppHandler = getInAppHandler(context);
    if (inAppHandler != null){
      inAppHandler.syncInAppsIfRequired(context);
    }
  }

  public interface InAppHandler {

    /**
     * (non-Javadoc)
     * Shows in-app on Configuration change(a.k.a rotation )
     *
     * @param context Application Context
     */
    void showInAppOnConfigurationChange(Context context);

    /**
     * (non-Javadoc)
     * Show InApp if any eligible campaign exists.
     *
     * @param context Application Context
     */
    void showInAppIfEligible(Context context);

    /**
     * (non-Javadoc)
     * Try to show in-app for a smart event
     *
     * @param context Application Context
     * @param responseObject json response for the smart trigger
     */
    @WorkerThread void tryShowAutoTriggerInApp(Context context, JSONObject responseObject);

    /**
     * (non-Javadoc)
     * Parses the response for in-app api calls.
     *
     * @param responseObject API response object
     * @param context Application Context
     */
    void parseAndSaveInApps(JSONObject responseObject, Context context);

    /**
     * (non-Javadoc)
     * Registers a smart event and tries to show in-app for it
     *
     * @param context Application Context
     * @param event event for which there is a trigger
     */
    void registerAutoTriggerEvent(Context context, Event event);

    /**
     * (non-Javadoc)
     * Register to show in-apps in the activity.<br>
     * Should be called from {@link Activity#onStart()}
     *
     * @param currentActivity Activity instance in which in-apps needs to be shown.
     */
    void registerInAppManager(Activity currentActivity);

    /**
     * (non-Javadoc)
     * Unregister to show in-apps in the activity.<br>
     * Should be called from {@link Activity#onStop()}
     *
     * @param currentActivity Activity instance in which in-apps needs to be shown.
     */
    void unregisterInAppManager(Activity currentActivity);

    /**
     * Show linked in-app
     * @param campaignId Linked in-app campaignId
     * @param context Application Context
     */
    void fetchLinkedInApp(Context context, String campaignId);

    /**
     * Show linked/test inapp
     * @param context Application Context
     * @param jsonObject InApp JSON
     * @param paramsMap Campaign Id map
     */
    void showLinkedInApp(Context context, JSONObject jsonObject, HashMap<String, String> paramsMap);

    /**
     * Show error dialog if test in-app fails
     * @param errorMessage : error message
     */
    void showTestInAppErrorDialog(String errorMessage);

    /**
     * Store the in-app check failure counter as an event
     * @param context Application Context
     */
    void writeInAppCheckFailureCounterToStorage(Context context);

    /**
     * Log common in-app API eror
     * @param apiName API which failed
     */
    void logInAppAPIFailure(String apiName);

    /**
     * Syncs In-Apps with the server, if required.
     *
     * @param context Application Context
     */
    void syncInAppsIfRequired(Context context);
  }
}
