package com.moengage.core;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobParameters;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.moe.pushlibrary.models.BatchData;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.reports.ReportsBatchHelper;
import java.util.ArrayList;
import java.util.HashMap;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * (non-JavaDoc)
 * Sends all the events to the server once the application is closed.
 * Service constant MSG_SEND_INTERACTION_DATA
 *
 * @author Umang Chamaria
 */
class SendInteractionDataTask extends SDKTask {
  private OnJobComplete jobCompleteCallback;
  private JobParameters jobParameters;

  private final String TAG = "SendInteractionDataTask";

  SendInteractionDataTask(Context context, @Nullable OnJobComplete jobCompleteCallback,
      @Nullable JobParameters parameters) {
    super(context);
    this.jobCompleteCallback = jobCompleteCallback;
    this.jobParameters = parameters;
  }

  @Override public TaskResult execute() {
    try {
      if (!ConfigurationProvider.getInstance(mContext).isAppEnabled()) return null;
      Logger.v(TAG + " executing task");
      String appId  = MoEUtils.getAppId(mContext);
      if (TextUtils.isEmpty(appId)) {
        Logger.e(TAG + " execute: App-id not present cannot make report add call.");
        return null;
      }
      sendInteractionData(appId);
      schedulePeriodicRetryIfRequired();
      releaseJobLockIfRequired();
      Logger.v(TAG + " : completed task");
      return null;
    } catch (Exception e) {
      Logger.f(TAG + " : execute() ", e);
    }
    return null;
  }

  private void releaseJobLockIfRequired() {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
        && jobCompleteCallback != null
        && jobParameters != null) {
      Logger.v(TAG + " releaseJobLockIfRequired() : Trying to release job lock.");
      jobCompleteCallback.jobCompleted(jobParameters, false);
    }
  }

  @Override public String getTaskTag() {
    return TAG_SEND_INTERACTION_DATA;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private void sendInteractionData(String appId) {
    boolean result = false;
    ArrayList<BatchData> batchedData = null;
    String endpoint =
        MoEConstants.API_ENDPOINT_REPORT_ADD + "/" + appId;
    if (shouldSendDataToTestServer()) {
      endpoint = MoEConstants.API_ENDPOINT_INTEGRATION_VERIFICATION_REPORT_ADD;
    }
    ReportsBatchHelper batchHelper = new ReportsBatchHelper();
    for (; ; ) {
      batchedData = MoEDAO.getInstance(mContext).getBatchedData(100);
      Logger.d(
          "SendInteractionDataTask : sendInteractionData:Fetching interaction data in batches");
      if (batchedData == null || batchedData.isEmpty()) {
        Logger.d("SendInteractionDataTask : sendInteractionData: Found Nothing to send");
        return;
      }
      for (BatchData data : batchedData) {
        BatchData batch = batchHelper.updateBatchIfRequired(mContext, data);
        try {
          HashMap<String, String> headers = new HashMap<>();
          headers.put(MoEConstants.REQUEST_HEADER_REQUEST_ID, getRequestID(batch.batchData));
          result = APIManager.sendInteractionReport(mContext, cleanBody(batch.batchData), endpoint
              , headers);
        } catch (Exception e) {
          Logger.f("SendInteractionDataTask : API failed", e);
          result = false;
        }
        if (result) {
          Logger.d("SendInteractionDataTask : Batch sent successfully deleting batch");
          MoEDAO.getInstance(mContext).deleteBatch(batch);
        } else {
          int retryCount = ConfigurationProvider.getInstance(mContext).getImmediateRetryCount();
          switch (retryCount) {
            case 0:
              scheduleImmediateRetry(1);
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(++retryCount);
              break;
            case 1:
              scheduleImmediateRetry(3);
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(++retryCount);
              break;
            default:
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(0);
          }
          break;
        }
      }
      if (!result) break;
      batchedData.clear();
    }
  }

  /**
   * Checks whether device is registered as a test device or not
   *
   * @return true if registered as test device, else false
   */
  private boolean shouldSendDataToTestServer() {
    boolean isDeviceRegisteredForVerification =
        ConfigurationProvider.getInstance(mContext).isDeviceRegisteredForVerification();
    long registrationTime =
        ConfigurationProvider.getInstance(mContext).getVerificationRegistrationTime();
    return isDeviceRegisteredForVerification && ((registrationTime + 3600 * 1000)
        > MoEUtils.currentTime());
  }

  /**
   * Scheduling a periodic retry to send
   */
  private void scheduleRetryDataSyncAlarm() {
    Logger.v("Scheduling data sync retry");
    Intent alarmIntent = new Intent(mContext, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(mContext, 88888, alarmIntent, PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
    alarmManager.setInexactRepeating(AlarmManager.RTC_WAKEUP,
        MoEUtils.currentTime() + ConfigurationProvider.getInstance(mContext).getDataSyncRetryTime(),
        ConfigurationProvider.getInstance(mContext).getDataSyncRetryTime(), pendingIntent);
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP) private void scheduleRetryDataSyncJob() {
    Logger.v("Scheduling retry data sync job");
    ComponentName serviceComponent = new ComponentName(mContext, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(77777, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(MoEUtils.currentTime()
        + ConfigurationProvider.getInstance(mContext).getDataSyncRetryTime() * 2);
    builder.setMinimumLatency(ConfigurationProvider.getInstance(mContext).getDataSyncRetryTime());
    JobScheduler jobScheduler =
        (JobScheduler) mContext.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    jobScheduler.schedule(builder.build());
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP) private void scheduleImmediateRetrySyncJob(int minutes) {
    Logger.v("Scheduling immediate retry data sync job");
    ComponentName serviceComponent = new ComponentName(mContext, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(666666, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(MoEUtils.currentTime() + minutes * 2 * 60 * 1000);
    builder.setMinimumLatency(minutes * 60 * 1000);
    JobScheduler jobScheduler =
        (JobScheduler) mContext.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  private void scheduleImmediateRetryAlarm(int minutes) {
    Intent alarmIntent = new Intent(mContext, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(mContext, 55555, alarmIntent, PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP, MoEUtils.currentTime() + (minutes * 60 * 1000),
          pendingIntent);
    }
  }

  private void scheduleImmediateRetry(int minutes) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      scheduleImmediateRetrySyncJob(minutes);
    } else {
      scheduleImmediateRetryAlarm(minutes);
    }
  }

  @Nullable private String getRequestID(JSONObject batchData) {
    try {
      return batchData.getString(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    } catch (JSONException e) {
      Logger.e("SendInteractionDataTask: getRequestID(): Exception ", e);
    }
    return null;
  }

  private JSONObject cleanBody(JSONObject batchData) {
    batchData.remove(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    return batchData;
  }

  private void schedulePeriodicRetryIfRequired() {
    if (!ConfigurationProvider.getInstance(mContext).isBackgroundSyncEnabled()) return;
    Logger.v(TAG + " schedulePeriodicRetryIfRequired() : Will schedule background retry.");
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      scheduleRetryDataSyncJob();
    } else {
      scheduleRetryDataSyncAlarm();
    }
  }
}
