package com.moengage.core;

import android.content.Context;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import java.util.List;

/**
 * Logs activity start event
 * Service constant MSG_ACTIVITY_START
 * @author Umang Chamaria
 */
class ActivityStartTask extends SDKTask {

  private String mCurrentActivityName;
  private boolean mNeedToCheckForGAIDChange;

  ActivityStartTask(Context context, String currentActivityName,
      boolean needToCheckForGAIDChange) {
    super(context);
    mContext = context;
    mCurrentActivityName = currentActivityName;
    mNeedToCheckForGAIDChange = needToCheckForGAIDChange;
  }

  @Override public TaskResult execute() {
    Logger.v("ActivityStartTask : started execution");
    if (!TextUtils.isEmpty(mCurrentActivityName)) {
      Logger.v("ActivityLifecycleStart : " + mCurrentActivityName + " started");
      if (MoEHelper.getActivityCounter() == 1 && mNeedToCheckForGAIDChange){
        MoEUtils.trackActivityStates(MoEConstants.EVENT_ACTION_ACTIVITY_START, mCurrentActivityName,
            mContext);
      }else if (!isActivityTracked()){
        MoEUtils.trackActivityStates(MoEConstants.EVENT_ACTION_ACTIVITY_START, mCurrentActivityName,
            mContext);
        ConfigurationProvider.getInstance(mContext).addScreenToSentList(mCurrentActivityName);
      }
    }
    if (mNeedToCheckForGAIDChange) {
      MoEDAO.getInstance(mContext).removeExpiredData();
      checkGoogleAdvertisementIDAndUpdate();
    } else {
      Logger.v("ActivityStartTask : No Need to check GAID");
    }
    mTaskResult.setIsSuccess(true);
    mTaskResult.setPayload(mNeedToCheckForGAIDChange);
    Logger.v("ActivityStartTask : completed execution");
    return mTaskResult;
  }

  @Override public String getTaskTag() {
    return TAG_ACTIVITY_START;
  }

  @Override public boolean isSynchronous() {
    return false;
  }

  private void checkGoogleAdvertisementIDAndUpdate() {
    mNeedToCheckForGAIDChange = false;
    ConfigurationProvider provider = ConfigurationProvider.getInstance(mContext);
    if (provider.isAdIdCollectionProhibitted()) {
      Logger.f("ActivityStartTask : Opted out of GAID Collection");
      return;
    }

    String storedGAID = provider.getStoredGAID();
    int storedISLAT = provider.getStoredISLAT();
    AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(mContext);
    if (null != adInfo) {
      //check if the google advertisement id has changed or not
      if (!TextUtils.isEmpty(adInfo.getId()) && (TextUtils.isEmpty(storedGAID) || (!adInfo.getId()
          .equals(storedGAID)))) {
        MoEUtils.setUserAttributeInternal(mContext, MoEConstants.ATTR_MOE_GAID, adInfo.getId());
        provider.storeGAID(adInfo.getId());
      } else {
        mNeedToCheckForGAIDChange = true;
      }
      //check if the adTracking opt out has changed or not
      if (adInfo.isLimitAdTrackingEnabled() != storedISLAT) {
        MoEUtils.setUserAttributeInternal(mContext, MoEConstants.ATTR_IS_LAT,
            Integer.toString(adInfo.isLimitAdTrackingEnabled()));
        provider.storeISLAT(adInfo.isLimitAdTrackingEnabled());
      } else {
        mNeedToCheckForGAIDChange = true;
      }
    } else {
      mNeedToCheckForGAIDChange = true;
    }
  }

  private boolean isActivityTracked(){
    try {
      List<String> trackedScreen = ConfigurationProvider.getInstance(mContext).getSentScreenList();
      return trackedScreen != null && trackedScreen.contains(mCurrentActivityName);
    }catch (Exception e){
      Logger.f("ActivityStartTask: isActivityTracked : ", e);
    }
    return false;
  }
}