package com.moengage.core;

import android.content.Context;
import android.net.Uri;
import android.os.Build;
import android.os.Build.VERSION;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.WindowManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.rest.RequestBuilder;
import com.moengage.core.rest.RequestBuilder.RequestType;
import com.moengage.core.rest.Response;
import com.moengage.core.rest.RestClient;
import com.moengage.core.utils.JsonBuilder;
import com.moengage.core.utils.RestUtils;
import java.util.TimeZone;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Makes a device add call on fresh installation
 * Service constant MSG_DEVICE_ADD
 *
 * @author Umang Chamaria
 */
public class DeviceAddTask extends SDKTask {

  private static final String TAG = "DeviceAddTask";
  private String requestTime;
  private String batchId;
  private ConfigurationProvider provider;
  private Bundle taskResult;
  private JSONObject devicePreference;

  DeviceAddTask(Context context, JSONObject devicePreference) {
    super(context);
    provider = ConfigurationProvider.getInstance(mContext);
    taskResult = new Bundle();
    this.devicePreference = devicePreference;
  }

  @Override public TaskResult execute() {
    Logger.v(TAG + " execution started");
    try {
      String appId = MoEUtils.getAppId(mContext);
      if (TextUtils.isEmpty(appId)) {
        Logger.e(TAG + " execute: Cannot make device add call, app id not present.");
        mTaskResult.setIsSuccess(false);
        return mTaskResult;
      }
      Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder(mContext)
          .appendEncodedPath(MoEConstants.API_ENDPOINT_DEVICE_ADD)
          .appendPath(appId);
      RequestBuilder requestBuilder = RestUtils.getBaseRequestBuilder(uriBuilder.build(),
          RequestType.POST, appId)
          .addBody(getBody())
          .addHeader(MoEConstants.REQUEST_HEADER_REQUEST_ID, getRequestId());
      RestClient restClient = new RestClient(requestBuilder.build());
      Response response = restClient.executeRequest();
      if (response != null && response.getResponseCode() == 200){
        mTaskResult.setIsSuccess(true);
      }
    } catch (Exception e) {
      Logger.f(TAG + " execute() : ", e);
    }
    Logger.v(TAG + " execution completed");
    mTaskResult.setPayload(taskResult);
    return mTaskResult;
  }

  @Override public String getTaskTag() {
    return TAG_DEVICE_ADD;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private JSONObject getSDKMeta() throws JSONException {
    JSONObject metaJSON = new JSONObject();
    batchId = MoEUtils.getBatchId();
    requestTime = MoEUtils.getCurrentISOTime();
    metaJSON.put(MoEConstants.ATTR_BATCH_ID, batchId)
        .put(MoEConstants.REQUEST_ATTR_REQUEST_TIME, requestTime)
        .put(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE, devicePreference);
    return metaJSON;
  }

  private JSONObject getQueryParams() throws JSONException {
    JsonBuilder builder = RestUtils.getDefaultParams(mContext);

    builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());

    String fcmToken = provider.getGCMToken();
    if (!TextUtils.isEmpty(fcmToken) && !provider.isPushNotificationOptedOut()){
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GCM_ID, fcmToken);
      taskResult.putBoolean(DeviceAddManager.CONSTANT_FCM_TOKEN_PRESENT, true);
    }

    String miPushToken = provider.getMiPushToken();
    if (!TextUtils.isEmpty(miPushToken) && !provider.isPushNotificationOptedOut()){
      builder.putString(MoEConstants.PARAM_V2_KEY_MI_PUSH_TOKEN, miPushToken);
      taskResult.putBoolean(DeviceAddManager.CONSTANT_MI_TOKEN_PRESENT, true);
    }

    if (!provider.isDataTrackingOptedOut()) {
      String androidId = MoEUtils.getAndroidID(mContext);
      if (!TextUtils.isEmpty(androidId)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, androidId);
      }
      if (!provider.isAdIdCollectionProhibitted()) {
        String gaid = provider.getStoredGAID();
        if (TextUtils.isEmpty(gaid)) {
          AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(mContext);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!TextUtils.isEmpty(gaid)) {
          builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION,
          String.valueOf(Build.VERSION.SDK_INT));
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL);
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
          provider.getAppVersionName());

      String nwType = MoEUtils.getNetworkType(mContext);
      if (!TextUtils.isEmpty(nwType)) {
        builder.putString(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
      }
    }

    return builder.build();
  }

  private JSONObject getBody() throws JSONException {
    JsonBuilder requestBody =
        getDeviceInfo().putJsonObject(MoEConstants.ATTR_SDK_META, getSDKMeta())
            .putJsonObject(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, getQueryParams());
    return requestBody.build();
  }

  private JsonBuilder getDeviceInfo() throws JSONException {
    JsonBuilder deviceInfo = new JsonBuilder();
    if (provider.isDeviceAttributesCollectionProhibited()
        || provider.isDataTrackingOptedOut()) {
      return deviceInfo;
    }
    deviceInfo.putString("OS_VERSION", VERSION.RELEASE)
        .putInt("OS_API_LEVEL", VERSION.SDK_INT)
        .putString("DEVICE", Build.DEVICE)
        .putString("MODEL", Build.MODEL)
        .putString("PRODUCT", Build.PRODUCT)
        .putString("MANUFACTURER", Build.MANUFACTURER);
    String deviceId = MoEUtils.getAndroidID(mContext);
    if (!TextUtils.isEmpty(deviceId)) {
      deviceInfo.putString("DEVICE_ID", deviceId);
    }
    String operator = MoEUtils.getOperatorName(mContext);
    if (!TextUtils.isEmpty(operator)) {
      deviceInfo.putString("CARRIER", operator);
    }
    WindowManager wm = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
    DisplayMetrics outMetrics = new DisplayMetrics();
    if (wm != null) {
      wm.getDefaultDisplay().getMetrics(outMetrics);
      deviceInfo.putInt("DENSITYDPI", outMetrics.densityDpi)
          .putInt("WIDTH", outMetrics.widthPixels)
          .putInt("HEIGHT", outMetrics.heightPixels);
    }
    if (!provider.isAdIdCollectionProhibitted()) {
      AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(mContext);
      if (null != adInfo) {
        deviceInfo.putString(MoEConstants.ATTR_MOE_GAID, adInfo.getId())
            .putInt(MoEConstants.ATTR_IS_LAT, adInfo.isLimitAdTrackingEnabled());
      }
    }
    return deviceInfo;
  }

  private String getRequestId() {
    return MoEUtils.getSha1ForString(
        batchId + requestTime + provider.getCurrentUserId());
  }
}
