package com.moengage.core;

import android.content.Context;
import android.os.Bundle;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.utils.JsonBuilder;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class DeviceAddManager {

  private static final String TAG = "DeviceAddManager";

  public static final String CONSTANT_FCM_TOKEN_PRESENT = "is_fcm_token_present";
  public static final String CONSTANT_MI_TOKEN_PRESENT = "is_gcm_token_present";

  private boolean isDeviceAddInProgress;
  private boolean isFcmRegistrationPending;
  private boolean isMiRegistrationPending;
  private boolean isGdprRequestPending;

  public void registerGdprOptOut(Context context) {
    if (isDeviceAddInProgress){
      Logger.v(TAG + " registerGdprOptOut() : Device Add is in progress, will send gdpr opt-out "
          + "after current request completes.");
      isGdprRequestPending = true;
    }else {
      Logger.v(TAG + " registerGdprOptOut() : Initiating request to send GDPR opt out.");
      initiateDeviceAdd(context);
    }
  }

  public void registerFcmToken(Context context) {
    if (isDeviceAddInProgress) {
      Logger.v(TAG
          + " registerFcmToken() : Device add in progress will send FCM token to server "
          + "after current request completes.");
      isFcmRegistrationPending = true;
    } else {
      Logger.v(TAG + " registerFcmToken() : Initiating request for sending FCM token to server.");
      initiateDeviceAdd(context);
    }
  }

  public void registerMiToken(Context context) {
    ConfigurationProvider.getInstance(context).setMiTokenServerSentState(false);
    if (isDeviceAddInProgress) {
      Logger.v(TAG
          + " registerMiToken() : Device add in progress will send Mi Push token to "
          + "server after current request completes.");
      isMiRegistrationPending = true;
    } else {
      Logger.v(TAG + " registerMiToken() : Initiating request for sending Mi Push token to server");
      initiateDeviceAdd(context);
    }
  }

  void retryDeviceRegistrationIfRequired(Context context) {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    if (!provider.isDeviceRegistered()) {
      Logger.v(TAG + " retryDeviceRegistrationIfRequired() : Device not registered yet. Will try "
          + "to register device.");
      initiateDeviceAdd(context);
    }
  }

  private void initiateDeviceAdd(Context context) {
    if (!isDeviceAddInProgress) {
      isDeviceAddInProgress = true;
      ConfigurationProvider.getInstance(context).setDeviceRegistrationState(false);
      MoEDispatcher.getInstance(context).addTaskToQueue(new DeviceAddTask(context,
          getPreferencesJson(context)));
    } else {
      Logger.v(TAG
          + " initiateDeviceAdd() : Device Add is already in progress cannot schedule "
          + "another one");
    }
  }

  void processTaskResult(Context context, TaskResult taskResult) {
    try {
      isDeviceAddInProgress = false;
      Logger.v(TAG + " processTaskResult() : Device Add completed, processing response.");
      if (taskResult == null) return;
      ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
      provider.setDeviceRegistrationState(taskResult.isSuccess());
      if (!taskResult.isSuccess()) {
        Logger.v(TAG + " processTaskResult() : Device add failed recently");
        return;
      }
      Bundle pushParams = (Bundle) taskResult.getPayload();
      if (pushParams == null) return;

      boolean hasSentMiToken = pushParams.getBoolean(CONSTANT_MI_TOKEN_PRESENT, false);
      boolean hasSentFcmToken = pushParams.getBoolean(CONSTANT_FCM_TOKEN_PRESENT, false);

      if (hasSentFcmToken) isFcmRegistrationPending = false;
      if (hasSentMiToken) isMiRegistrationPending = false;

      provider.setFcmTokenServerSendState(hasSentFcmToken);
      provider.setMiTokenServerSentState(hasSentMiToken);

      if (isMiRegistrationPending) {
        isMiRegistrationPending = false;
        registerMiToken(context);
      }

      if (isFcmRegistrationPending) {
        isFcmRegistrationPending = false;
        registerFcmToken(context);
      }

      if (isGdprRequestPending) {
        isGdprRequestPending = false;
        registerGdprOptOut(context);
      }
    } catch (Exception e) {
      Logger.e( TAG + " processTaskResult() : Exception ", e);
    }
  }

  private JSONObject getPreferencesJson(Context context){
    JsonBuilder preferences = new JsonBuilder();
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    try {
      preferences.putBoolean(MoEConstants.REQUEST_ATTR_PUSH_PREFERENCE,
          !provider.isPushNotificationOptedOut());
      preferences.putBoolean(MoEConstants.REQUEST_ATTR_IN_APP_PREFERENCE,
          !provider.isInAppOptedOut());
      preferences.putBoolean(MoEConstants.REQUEST_ATTR_DATA_TRACKING_PREFERENCE,
          !provider.isDataTrackingOptedOut());
    } catch (Exception e) {
      Logger.e( TAG + " getPreferencesJson() : Exception: ", e);
    }
    return preferences.build();
  }
}
