package com.moengage.core.userattributes;

import android.content.Context;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.MoEAttribute;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class TrackAttributeTask extends SDKTask {
  private static final String TAG = "TrackAttributeTask";

  private JSONObject attributeJson;
  private boolean isCustomAttribute;

  TrackAttributeTask(Context context, JSONObject attributeJson, boolean isCustomAttribute) {
    super(context);
    this.attributeJson = attributeJson;
    this.isCustomAttribute = isCustomAttribute;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " executing Task : ");
      if (isCustomAttribute) {
        Logger.v(TAG + " execute() : No need to cache custom attributes, will track attribute.");
        writeUserAttributeToStorage(attributeJson);
        mTaskResult.setIsSuccess(true);
        return mTaskResult;
      }
      MoEAttribute trackedAttribute = MoEUtils.convertJsonToAttributeObject(attributeJson);
      if (trackedAttribute == null) {
        Logger.v(TAG
            + " execute() : Conversion from Json to object failed cannot decide whether to track "
            + "attribute or not, will track attribute.");
        writeUserAttributeToStorage(attributeJson);
        mTaskResult.setIsSuccess(true);
        return mTaskResult;
      }
      Logger.v(TAG
          + " execute() : Will try to send attribute to server. Attribute: "
          + trackedAttribute.toString());
      MoEAttribute savedAttribute =
          MoEDAO.getInstance(mContext).getAttributeByName(trackedAttribute.getName());
      //check for unique id change
      String currentUniqueId = MoEUtils.getUserAttributeUniqueId(mContext);
      if (trackedAttribute.getName().equals(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID)
          && currentUniqueId != null
          && !trackedAttribute.getValue().equals(currentUniqueId)) {
        Logger.v(TAG
            + " execute(): User Attribute Unique Id has changed will trigger force "
            + "logout.");
        MoEDispatcher.getInstance(mContext).handleLogout(true);
        return createTaskResult(attributeJson, false);
      }
      if (savedAttribute == null) {
        Logger.v(TAG
            + " execute() : This attribute hasn't been cached yet, will track this "
            + "attribute");
        writeUserAttributeToStorage(attributeJson);
        cacheAttribute(trackedAttribute);
        mTaskResult.setIsSuccess(true);
        return mTaskResult;
      }

      Logger.v(TAG + " execute(): Saved user attribute: " + savedAttribute.toString());

      if (shouldTrackAttribute(trackedAttribute, savedAttribute,
          ConfigurationProvider.getInstance(mContext).getUserAttributeCachingTime())) {
        writeUserAttributeToStorage(attributeJson);
        cacheAttribute(trackedAttribute);
      } else {
        Logger.v(TAG + " execute() : Attribute Already tracked. Will not be sent to server.");
      }
      Logger.v(TAG + " execute() : completed execution");
    } catch (Exception e) {
      Logger.f(TAG + " execute() : Exception ", e);
    }
    mTaskResult.setIsSuccess(true);
    return mTaskResult;
  }

  @Override public String getTaskTag() {
    return TAG_TRACK_ATTRIBUTE;
  }

  @Override public boolean isSynchronous() {
    return false;
  }

  private void writeUserAttributeToStorage(JSONObject attributeJson) {
    Event event = new Event(
        MoEHelperUtils.getDatapointJSON(MoEConstants.EVENT_ACTION_USER_ATTRIBUTE, attributeJson));
    MoEDispatcher.getInstance(mContext).writeDataPointToStorage(event);
  }

  private boolean shouldTrackAttribute(MoEAttribute trackedAttribute, MoEAttribute savedAttribute,
      long sendDelay) {
    if (!trackedAttribute.getName().equals(savedAttribute.getName())) return true;
    if (!trackedAttribute.getValue().equals(savedAttribute.getValue())) return true;
    if (!trackedAttribute.getDataType().equals(savedAttribute.getDataType())) return true;
    return savedAttribute.getLastTrackedTime() + sendDelay < trackedAttribute.getLastTrackedTime();
  }

  private void cacheAttribute(MoEAttribute attribute) {
    Logger.v(TAG + " cacheAttribute(): Will cache attribute: " + attribute.toString());
    if (attribute == null) {
      Logger.e(TAG + " cacheAttribute() : Attribute object is null cannot cache it.");
      return;
    }
    if (attribute.getName().equals(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID)){
      Logger.v(TAG + " cacheAttribute(): Attribute to cache is USER_ATTRIBUTE_UNIQUE_ID will copy"
          + " it to shared preference as well");
      MoEDAO.getInstance(mContext).saveUserAttributeUniqueId(attribute);
      return;
    }
    MoEDAO.getInstance(mContext).addOrUpdateAttributeToCache(attribute);
  }
}
