package com.moengage.core;

import android.app.Application;
import android.content.Context;
import android.support.annotation.ColorRes;
import android.support.annotation.DrawableRes;
import android.support.annotation.IntegerRes;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import com.delight.pushlibrary.R;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.push.PushManager;
import java.util.ArrayList;
import java.util.List;

/**
 * Initialise MoEngage SDK using this class.
 *
 * Please follow the guidelines mentioned in <a
 * href="https://docs.moengage.com/docs/sdk-installation"
 * >MoEngage Integration Docs</a> for integrating the
 * MoEngage SDK.<br>
 *
 * This initializer should be called in in the `onCreate()` of the {@link Application} class.
 * Below is a sample of how the SDK should be initialised.
 * <pre>
 *   <code>
 *     MoEngage moEngage =
 * new MoEngage.Builder(this, "DAO6UGZ73D9RTK8B5W96TPYN")
 * .setSenderId("615448685370")
 * .setNotificationSmallIcon(R.drawable.icon)
 * .setNotificationLargeIcon(R.drawable.ic_launcher)
 * .setLogLevel(Logger.VERBOSE)
 * .setNotificationType(R.integer.notification_type_multiple)
 * .build();
 * MoEngage.initialise(moEngage);
 *
 *   </code>
 * </pre>
 *
 * <b>Note: The above initialisation example does not contain all the methods. It only includes a
 * few commonly used methods. Please refer to the documentation of {@link MoEngage.Builder}
 * for all methods and add them according to your needs.</b>
 *
 * @author Umang Chamaria
 * @since 9.0.00
 */

public class MoEngage {

  public enum DATA_REGION {
    /**
     * Enum to redirect data to Indian Cluster
     */
    REGION_INDIA(1001), /**
     * Enum to redirect data to European Cluster
     */
    REGION_EU(1002), /**
     * Enum to redirect data to USA Cluster. This is the default region need not be mentioned
     * explicitly
     */
    REGION_DEFAULT(1003);

    private final int region;

    DATA_REGION(int region) {
      this.region = region;
    }

    public int getRegion() {
      return region;
    }
  }

  private static final String TAG = "MoEngage";

  private MoEngage.Builder mBuilder;

  private MoEngage(MoEngage.Builder builder) {
    mBuilder = builder;
  }

  /**
   * Initialise MoEngage SDK with {@link MoEngage} object returned by {@link MoEngage.Builder}
   *
   * @param moEngage object of {@link MoEngage}
   * @since 9.0.00
   */
  public static void initialise(@NonNull MoEngage moEngage) {
    if (moEngage == null) {
      Logger.e(TAG + " Object instance is null cannot initialise");
      return;
    }
    MoEngage.Builder builder = moEngage.mBuilder;
    if (builder == null || builder.context == null || builder.application == null) {
      Logger.e(TAG + "Builder/Context/Application is null. Cannot initialise SDK.");
      return;
    }
    Context context = builder.context;
    ConfigurationProvider.shouldReadManifestConfig = false;
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    //setting log level
    Logger.setLogLevel(builder.logLevel);
    //setting log state for signed build
    if (builder.logStatus) Logger.setLogStatus(builder.logStatus);
    Logger.enableDebugLog(context);
    //setting app id
    if (!builder.isSegmentIntegration) {
      if (!TextUtils.isEmpty(builder.appId)) {
        provider.setAppId(builder.appId);
      } else {
        Logger.e(TAG + "init() : " + "App-id not passed. Cannot use MoEngage Platform");
        return;
      }
    }else {
      Logger.v( TAG + " initialise() : Segment integration enabled will not use app id");
    }
    // set large icon
    if (builder.largeIcon != -1) {
      provider.setLargeIcon(builder.largeIcon);
    } else {
      Logger.e(TAG + "init() : " + "Large icon not set");
    }
    //set small icon
    if (builder.smallIcon != -1) {
      provider.setSmallIcon(builder.smallIcon);
    } else {
      Logger.e(TAG + "init() : " + "Small icon not set cannot show notification");
    }
    //sender id
    if (!TextUtils.isEmpty(builder.senderId)) {
      provider.setSenderID(builder.senderId);
    }
    // notification color
    if (builder.color != -1) {
      provider.setNotificationColor(builder.color);
    } else {
      provider.setColorFallback();
    }
    //notification tone
    if (!TextUtils.isEmpty(builder.tone)) {
      String tone = builder.tone;
      if (builder.tone.contains(".")) {
        tone = builder.tone.substring(0, builder.tone.lastIndexOf("."));
      }
      provider.setNotificationTone(tone);
    }
    //notification type
    provider.setNotificationType(context.getResources().getInteger(builder.notificationType));
    // inAppOptOut activity
    ArrayList<String> activityNameListInApp = new ArrayList<>();
    if (builder.inAppOptOutList != null) {
      try {
        for (Class clazz : builder.inAppOptOutList) {
          activityNameListInApp.add(clazz.getName());
        }
      } catch (Exception e) {
        Logger.f(TAG + "init() : " + "Activity Opt out ", e);
      }
    }
    activityNameListInApp.add("com.moengage.pushbase.activities.PushTracker");
    activityNameListInApp.add("com.moengage.pushbase.activities.SnoozeTracker");
    provider.setInAppOptOutActivityList(activityNameListInApp);
    //activity tracking opt-out
    if (builder.activityTrackingOptOutList != null) {
      try {
        ArrayList<String> activityNameList =
            new ArrayList<>(builder.activityTrackingOptOutList.size());
        for (Class clazz : builder.activityTrackingOptOutList) {
          activityNameList.add(clazz.getName());
        }
        provider.setTrackingOptedOutActivities(activityNameList);
      } catch (Exception e) {
        Logger.f(TAG + "init() : " + "Activity Opt out ", e);
      }
    }
    //back-stack opt out
    PushManager.getInstance().optoutBackStackBuilder(builder.backStackBuilderOptOut);
    //nav bar opt out
    provider.setNavBarOptOut(builder.navBarOptOut);
    // moEngageExtrasOptOut
    PushManager.getInstance().optOutMoEngageExtras(builder.moEngageExtrasOptOut);
    // gaid optOut
    provider.optOutOfAdIdCollection(builder.gaidOptOut);
    //androidId optOut
    provider.optOutOfAndroidIdCollection(builder.androidIdOptOut);
    //imei optOut
    provider.optOutOfIMEICollection(builder.imeiOptOut);
    //location optOut
    provider.optOutOfTrackLocation(builder.locationOptOut);
    //geo-fence optOut
    provider.optOutOfSetGeoFence(builder.geofenceOptOut);
    //operator name optOut
    provider.optOutOfOperatorNameCollection(builder.carrierNameOptOut);
    //device attributes optOut
    provider.optOutOfDeviceAttributesCollection(builder.deviceAttributeOptOut);
    // push registration optOpt
    provider.setPushRegistrationState(builder.isPushRegistrationEnabled);
    //set redirection
    provider.setDataRegion(builder.redirectionRegion.getRegion());
    //setting location services state
    provider.setLocationServicesState(builder.locationServices);
    // enable Baidu push
    if (builder.enableBaiduPush && !TextUtils.isEmpty(builder.baiduKey)) {
      PushManager.getInstance().enableBaiduPush();
      provider.setBaiduApiKey(builder.baiduKey);
    }
    //auto integrate
    if (builder.application != null && !builder.isSegmentIntegration) {
      MoEHelper.getInstance(context).autoIntegrate(builder.application);
    }else {
      Logger.v(TAG + " initialise() : Segment integration is enabled. Will not register for "
          + "lifecycle callbacks.");
    }
    //flush interval
    if (builder.flushInterval != -1) {
      MoEHelper.getInstance(context).setFlushInterval(builder.flushInterval);
    }
    //disable periodic flush
    MoEHelper.getInstance(context).setPeriodicFlushState(builder.isPeriodicFlushEnabled);
    // notification large icon optOut
    provider.optOutOfNotificationLargeIcon(builder.optOutNotificationLargeIcon);
    // background sync state
    provider.setBackgroundSyncState(builder.isBackgroundSyncEnabled);
    // dt background sync state
    provider.setDTBackgroundSyncState(builder.isRealTimeTriggerBackgroundSyncEnabled);
    //enabling instant app
    provider.setInstantAppState(builder.isInstantApp);
    //printing config
    try {
      StringBuilder stringBuilder = new StringBuilder("SDK Init Config: Details -> \n");
      stringBuilder.append("\n App id: ")
          .append(builder.appId)
          .append("\n sender id: ")
          .append(builder.senderId)
          .append("\n large icon: ")
          .append(builder.largeIcon)
          .append("\n small icon: ")
          .append(builder.smallIcon)
          .append("\n notification color: ")
          .append(builder.color)
          .append("\n notification tone: ")
          .append(builder.tone)
          .append("\n in-app out list");
      if (builder.inAppOptOutList != null) {
        stringBuilder.append(builder.inAppOptOutList.toString());
      }
      stringBuilder.append("\n activity tracking opt-out: ");
      if (builder.activityTrackingOptOutList != null) {
        stringBuilder.append(builder.activityTrackingOptOutList.toString());
      }
      stringBuilder.append("\n notification type: ")
          .append(builder.notificationType)
          .append("\n backStackBuilderOptOut: ")
          .append(builder.backStackBuilderOptOut)
          .append("\n navBarOptOut: ")
          .append(builder.navBarOptOut)
          .append("\n moEngageExtrasOptOut: ")
          .append(builder.moEngageExtrasOptOut)
          .append("\n gaidOptOut: ")
          .append(builder.gaidOptOut)
          .append("\n androidIdOptOut: ")
          .append(builder.androidIdOptOut)
          .append("\n imeiOptOut: ")
          .append(builder.imeiOptOut)
          .append("\n locationOptOut: ")
          .append(builder.locationOptOut)
          .append("\n geofenceOptOut: ")
          .append(builder.geofenceOptOut)
          .append("\n carrierNameOptOut: ")
          .append(builder.carrierNameOptOut)
          .append("\n isPushRegistrationEnabled: ")
          .append(builder.isPushRegistrationEnabled)
          .append("\n redirectionRegion: ")
          .append(builder.redirectionRegion.getRegion())
          .append("\n flushInterval: ")
          .append(builder.flushInterval)
          .append("\n isPeriodicFlushEnabled: ")
          .append(builder.isPeriodicFlushEnabled)
          .append("\n enableBaiduPush: ")
          .append(builder.enableBaiduPush)
          .append("\n baiduKey: ")
          .append(builder.baiduKey)
          .append("\n logLevel: ")
          .append(builder.logLevel)
          .append("\n logStatus: ")
          .append(builder.logStatus)
          .append("\n locationServices: ")
          .append(builder.locationServices)
          .append("\n optOutNotificationLargeIcon: ")
          .append(builder.optOutNotificationLargeIcon)
          .append("\n isBackgroundSyncEnabled: ")
          .append(builder.isBackgroundSyncEnabled)
          .append("\n isRealTimeTriggerBackgroundSyncEnabled: ")
          .append(builder.isRealTimeTriggerBackgroundSyncEnabled)
          .append("\n isSegmentIntegration: ")
          .append(builder.isSegmentIntegration)
          .append("\n isInstantAppEnabled: ")
          .append(builder.isInstantApp);

      Logger.d(TAG + " initialise(): Config: " + stringBuilder.toString());
    } catch (Exception e) {
      Logger.e(TAG + " initialise() : ", e);
    }
  }

  /**
   * Optionally opt-out of data tracking. When data tracking is opted no event or user
   * attribute is tracked on MoEngage Platform.
   *
   * @param context Application context.
   * @param shouldOptOutDataTracking true if you don't want to track user data, else false.
   * @since 9.1.00
   */
  public static void optOutDataTracking(Context context, boolean shouldOptOutDataTracking){
    Logger.d(TAG + " optOutDataTracking() : Opt Out Called with value: " + shouldOptOutDataTracking);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    boolean savedState = provider.isDataTrackingOptedOut();
    provider.optOutOfDataTracking(shouldOptOutDataTracking);
    if (savedState != shouldOptOutDataTracking){
      MoEDispatcher.getInstance(context).addTaskToQueueBeginning(new DeviceAddTask(context));
    }
    if (shouldOptOutDataTracking){
      // gaid optOut
      provider.optOutOfAdIdCollection(true);
      //androidId optOut
      provider.optOutOfAndroidIdCollection(true);
      //imei optOut
      provider.optOutOfIMEICollection(true);
      //location optOut
      provider.optOutOfTrackLocation(true);
      //geo-fence optOut
      provider.optOutOfSetGeoFence(true);
      //device attributes optOut
      provider.optOutOfDeviceAttributesCollection(true);
      //setting location services state
      provider.setLocationServicesState(false);
      //clean data
      MoEDispatcher.getInstance(context).addTaskToQueue(new DataTrackingOptOutTask(context));
    }
  }

  /**
   * Optionally opt-out of push campaigns. No push campaigns will be shown once this is
   * opted out.
   *
   * @param context Application context.
   * @param shouldOptOutPushNotification true if you don't want users to receive push
   * notification, else false.
   * @since 9.1.00
   */
  public static void optOutPushNotification(Context context, boolean
      shouldOptOutPushNotification){
    Logger.d(TAG + "optOutPushNotification: Opt Out Called with value: " + shouldOptOutPushNotification);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    boolean savedState = provider.isPushNotificationOptedOut();
    provider.optOutOfPushNotification(shouldOptOutPushNotification);
    if (shouldOptOutPushNotification){
      provider.setGCMToken("");
    }
    if (savedState != shouldOptOutPushNotification){
      MoEDispatcher.getInstance(context).addTaskToQueueBeginning(new DeviceAddTask(context));
    }
  }

  /**
   * Optionally opt-out of in-app campaigns. No in-app campaigns will be shown once this is
   * opted out.
   *
   * @param context Application context.
   * @param shouldOptOutInApp true if you don't want users to receive in-app
   * notification, else false.
   * @since 9.1.00
   */
  public static void optOutInAppNotification(Context context, boolean
      shouldOptOutInApp) {
    Logger.d(TAG + " optOutInAppNotification() : Opt out called with value: " + shouldOptOutInApp);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    boolean savedState = provider.isInAppOptedOut();
    provider.optOutOfInAppNotification(shouldOptOutInApp);
    if (savedState != shouldOptOutInApp){
      MoEDispatcher.getInstance(context).addTaskToQueueBeginning(new DeviceAddTask(context));
    }
  }

  /**
   * Initialise MoEngage SDK using this class.
   *
   * Please follow the guidelines mentioned in <a
   * href="https://docs.moengage.com/docs/sdk-installation"
   * >MoEngage Integration Docs</a> for integrating the
   * MoEngage SDK.<br>
   *
   * This initializer should be called in in the `onCreate()` of the {@link Application} class.
   * Below is a sample of how the SDK should be initialised.
   * <pre>
   *   <code>
   *     MoEngage moEngage =
   * new MoEngage.Builder(this, "DAO6UGZ73D9RTK8B5W96TPYN")
   * .setSenderId("615448685370")
   * .setNotificationSmallIcon(R.drawable.icon)
   * .setNotificationLargeIcon(R.drawable.ic_launcher)
   * .setLogLevel(Logger.VERBOSE)
   * .setNotificationType(R.integer.notification_type_multiple)
   * .build();
   * MoEngage.initialise(moEngage);
   *
   *   </code>
   * </pre>
   *
   * @author Umang Chamaria
   * @since 9.0.00
   */
  public static class Builder {

    private static final String TAG = "MoEngage.Builder";

    private Context context;
    private String appId;
    private int largeIcon = -1;
    private int smallIcon = -1;
    private String senderId;
    private int color = -1;
    private String tone;
    private int notificationType = R.integer.notification_type_single;
    private Application application;
    private List<Class> inAppOptOutList;
    private List<Class> activityTrackingOptOutList;
    private boolean backStackBuilderOptOut;
    private boolean navBarOptOut;
    private boolean moEngageExtrasOptOut;
    private boolean gaidOptOut;
    private boolean androidIdOptOut;
    private boolean imeiOptOut = true;
    private boolean locationOptOut;
    private boolean geofenceOptOut;
    private boolean carrierNameOptOut;
    private boolean deviceAttributeOptOut;
    private boolean isPushRegistrationEnabled = true;
    private DATA_REGION redirectionRegion = DATA_REGION.REGION_DEFAULT;
    private long flushInterval = -1;
    private boolean isPeriodicFlushEnabled = true;
    private boolean enableBaiduPush;
    private String baiduKey;
    private int logLevel = Logger.ERROR;
    private boolean logStatus;
    private boolean locationServices;
    private boolean optOutNotificationLargeIcon;
    private boolean isBackgroundSyncEnabled = true;
    private boolean isRealTimeTriggerBackgroundSyncEnabled = true;
    private boolean isSegmentIntegration = false;
    private boolean isInstantApp = false;

    /**
     * Builder to initialise MoEngage SDK.
     * App Id can we found on the
     * <a href="https://app.moengage.com/v3/#/settings/app/general">Settings Page</a> of MoEngage
     * Dashboard
     *
     * @param application Instance of {@link Application}
     * @param appId App ID from MoEngage Dashboard
     * @since 9.0.00
     */
    public Builder(@NonNull Application application, @NonNull String appId) {
      this.application = application;
      this.context = application.getApplicationContext();
      this.appId = appId;
    }

    /**
     * Set the large icon to be shown in the notification.
     *
     * @param largeIcon resource link to the large icon.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationLargeIcon(@DrawableRes int largeIcon) {
      this.largeIcon = largeIcon;
      return this;
    }

    /**
     * Set the small icon to be shown in the notification.<br>
     * <b>This icon is a mandate for showing notifications</b><br>
     * Make sure the small icon follows the notification icon guidelines.
     * Guideline: Notification small icon should be flat, pictured face on, and must be white on
     * a transparent background. For more details refer to the
     * <a href="https://developer.android.com/guide/practices/ui_guidelines/icon_design_status_bar.html#style11">official docs</a>
     * for more details.
     *
     * @param smallIcon resource link to the small icon.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationSmallIcon(@DrawableRes int smallIcon) {
      this.smallIcon = smallIcon;
      return this;
    }

    /**
     * Set the sender-id/project id for the GCM/FCM project.<br>
     * <b>Sender id is a mandate when you are using GCM or Instant apps and MoEngage SDK is
     * registering for push.</b>
     *
     * @param senderId sender id/project id for your GCM/FCM project.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setSenderId(@NonNull String senderId) {
      this.senderId = senderId;
      return this;
    }

    /**
     * Set the color for notification. Currently color is given more precedence over large icon.
     *
     * @param color color resource id
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationColor(@ColorRes int color) {
      this.color = color;
      return this;
    }

    /**
     * Set the file name for notification tone.<br>
     * <b>Note:</b> This will work only below Android Oreo Devices.
     * <b>File should be present in the raw folder. Do not pass the file extension, only file
     * name should be passed.</b>
     *
     * @param tone file name for the notification tone.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationTone(String tone) {
      this.tone = tone;
      return this;
    }

    /**
     * By default the SDK shows only one notification at a given point in time. In case you want to
     * show multiple notification you need to set the type as multiple.
     *
     * @param notificationType Possible values: <br>
     * R.integer.notification_type_single - for single notification.<br>
     * R.integer.notification_type_multiple - for multiple notification.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationType(@IntegerRes int notificationType) {
      this.notificationType = notificationType;
      return this;
    }

    /**
     * By default the MoEngage SDK will try to show in-app messages on all the activities. If you
     * want block in-apps on any specific screen(s) use this API to pass a list of screens on
     * which in-apps should not be shown.
     *
     * @param inAppOptOutClassList list of {@link Class} on which in-app should not be shown.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setInAppOptOut(List<Class> inAppOptOutClassList) {
      inAppOptOutList = inAppOptOutClassList;
      return this;
    }

    /**
     * By default the MoEngage SDK would track screen-name for all the activities. In case you
     * don't want to track any specific screen(s) use this API to pass the list of screens to the
     * SDK which should not be tracked.
     *
     * @param trackingOptOut list of {@link Class} which should not be tracked.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setTrackingOptOut(List<Class> trackingOptOut) {
      activityTrackingOptOutList = trackingOptOut;
      return this;
    }

    /**
     * By default MoEngage SDK synthesis the back-stack from the manifest when redirecting user on
     * push click. In case you don't want back-stack synthesis use this opt-out
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutBackStackBuilder() {
      backStackBuilderOptOut = true;
      return this;
    }

    /**
     * By default the MoEngage SDK hides the top navigation bar when showing in-apps. If your
     * application has navigation drawer and fragments this creates a problem. In case your app
     * has a Navigation Drawer + Fragment combination use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     */
    public Builder optOutNavBar() {
      navBarOptOut = true;
      return this;
    }

    /**
     * By default MoEngage SDK adds the entire payload in the URL params. In case you want only
     * the key value pairs passed during campaign creation use this opt-out.<br>
     * Note: There is a backend config change required for this. Please contact our support
     * team with your App Name when using this.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutMoEngageExtras() {
      moEngageExtrasOptOut = true;
      return this;
    }

    /**
     * By default MoEngage SDK collects Google Advertising Identifier(GAID). In case you want to
     * restrict this GAID collection use this this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutGAIDCollection() {
      gaidOptOut = true;
      return this;
    }

    /**
     * By Default MoEngage SDK collects Android id. In case you want to restrict Android id
     * collection use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutAndroidIdCollection() {
      androidIdOptOut = true;
      return this;
    }

    /**
     * MoEngage SDK collects device IMEI if user permission is granted. In case you want to
     * restrict collection of IMEI use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optInIMEICollection() {
      imeiOptOut = false;
      return this;
    }

    /**
     * By default MoEngage SDK tracks user location. In case you want to restrict collection of
     * user location use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutLocationTracking() {
      locationOptOut = true;
      return this;
    }

    /**
     * By default the MoEngage SDK tries to set geo-fence based on the geo-fence campaigns created
     * on MoEngage Dashboard. In case you want don't want to set geo-fence for use this opt-out<br>
     * Note: If this opt-out is used no geo-fence campaign created on MoEngage Dashboard will
     * be delivered.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutGeoFence() {
      geofenceOptOut = true;
      return this;
    }

    /**
     * MoEngage SDK tries to collect the operator name if permission is granted.
     * In case you want to restrict collection of operator name use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutCarrierNameCollection() {
      carrierNameOptOut = true;
      return this;
    }

    /**
     * MoEngage SDK collects device attributes like height, width,screen density. In case you
     * want to restrict device attribute collection use this op-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutDeviceAttributeCollection() {
      deviceAttributeOptOut = true;
      return this;
    }

    /**
     * By default the MoEngage SDK registers for push token. In case you don't want the SDK to
     * register for push use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutTokenRegistration() {
      isPushRegistrationEnabled = false;
      return this;
    }

    /**
     * Redirect data to different regions if required.<br/>
     *
     * @param regionConstant one of the possible values based on the region you want to redirect
     * to.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder redirectDataToRegion(DATA_REGION regionConstant) {
      redirectionRegion = regionConstant;
      return this;
    }

    /**
     * Set the interval(in seconds) at which the SDK flushes out the data to the server.<br>
     * Note: Value will only be accepted if it is greater than the SDK defined flush interval.
     *
     * @param interval time interval to flush data. Value should be in seconds.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setFlushInterval(long interval) {
      flushInterval = interval;
      return this;
    }

    /**
     * By default MoEngage SDK sync's data every few minutes when the app is in foreground. In
     * case you want to disable periodic flush use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutPeriodicFlush() {
      isPeriodicFlushEnabled = false;
      return this;
    }

    /**
     * Sets log level for MoEngage SDK's logs. Default level is INFO
     *
     * @param logLevel log Level {@link Logger}
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setLogLevel(int logLevel) {
      this.logLevel = logLevel;
      return this;
    }

    /**
     * Enables/Disables MoEngage logs.<br><b>Note : This API should be used only if logs are
     * required in production/signed builds.</b>
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder enableLogsForSignedBuild() {
      this.logStatus = true;
      return this;
    }

    /**
     * Enable baidu push service. Refer to the documentation for additional configuration.
     * Documentation <a href="https://docs.moengage.com/docs/configuring-baidu">https://docs
     * .moengage.com/docs/configuring-baidu</a>
     *
     * @param baiduPushKey API key for client side registration.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder enableBaiduPush(@NonNull String baiduPushKey) {
      enableBaiduPush = true;
      this.baiduKey = baiduPushKey;
      return this;
    }

    /**
     * Enable location services for the SDK. By default SDK does not track location or set
     * geo-fence.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder enableLocationServices() {
      locationServices = true;
      return this;
    }

    /**
     * Optionally remove notification large icon
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutNotificationLargeIcon() {
      optOutNotificationLargeIcon = true;
      return this;
    }

    /**
     * Builds the configuration to initialise MoEngage SDK.
     * Call {@link MoEngage#initialise(MoEngage)} with the returned object to
     *
     * @return instance of {@link MoEngage}
     * @since 9.0.00
     */
    public MoEngage build() {
      return new MoEngage(this);
    }

    /**
     * MoEngage SDK by default syncs data in the background at a regular interval.
     * In case you don't want the SDK to sync data in background use this opt-out.<br/>
     * <b>Note:<b/> This can lead some un-intended delay in data appearing on MoEngage Dashboard.
     *
     * @return instance of {@link MoEngage}
     * @since 9.1.05
     */
    public Builder optOutBackgroundSync(){
      isBackgroundSyncEnabled = false;
      return this;
    }

    /**
     * MoEngage SDK by default syncs Real Time Trigger campaigns in the background at a regular
     * interval. In case you don't want the SDK to sync campaign in the background use this
     * opt-out.<br/>
     * <b>Note:</b> This can lead to reduced delivery of Real Time Trigger Campaigns.
     *
     * @return instance of {@link MoEngage}
     * @since 9.1.05
     */
    public Builder optOutRealTimeTriggerBackgroundSync(){
      isRealTimeTriggerBackgroundSyncEnabled = false;
      return this;
    }

    /**
     * This API it to notify MoEngage SDK that the app uses MoEngage SDK along with Segment's
     * SDK as a bundle.<br>
     * <b>Note:</b> Do not use this API if you are not using MoEngage via Segment as a bundled
     * SDK. Calling this API without using Segment bundle will result in malfunction of the SDK.
     *
     * @return instance of {@link MoEngage}
     * @since 9.1.06
     */
    public Builder enableSegmentIntegration(){
      isSegmentIntegration = true;
      return this;
    }

    /**
     * Enable Instant App features, namely using the instant app token registration.
     *
     * @return instance of {@link MoEngage}
     * @since 9.3.01
     */
    public Builder enableInstantApp(){
      isInstantApp = true;
      return this;
    }
  }
}
