package com.moengage.core;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.inapp.InAppController;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import org.json.JSONObject;

/**
 * Central point for all event related processing, i.e. tracking, blacklisting, trigger-events
 * @author Umang Chamaria
 */

public class MoEEventManager{

  private static final String TAG = "MoEEventManager";

  private static MoEEventManager _INSTANCE = null;

  private List<String> mBlackListedEvents = null;

  private ConfigurationProvider mConfigProvider = null;

  private Context mContext;

  private int mEventCounter = 0;

  private List<String> mTriggerEvents;

  private List<String> mFlushEvents;

  private LinkedHashSet<String> mGDPRWhiteList;

  private MoEEventManager(Context context){
    mContext = context;
    mConfigProvider = ConfigurationProvider.getInstance(context);
    getBlackListedEvents();
    getTriggerEvents();
    addDefaultFlushEvents();
    addDefaultGDPRWhiteList();
    getFlushEvents();
    getGDPRWhiteList();
  }

  public static MoEEventManager getInstance(Context context){
    if (_INSTANCE == null){
      _INSTANCE = new MoEEventManager(context);
    }
    return _INSTANCE;
  }

  private boolean isEventBlackListed(String eventName) {
    return !mBlackListedEvents.isEmpty() && mBlackListedEvents.contains(eventName);
  }

  void getBlackListedEvents() {
    try {
      mBlackListedEvents = new ArrayList<>();
      String blackListEventsString = mConfigProvider.getBlackListEvents();
      if (!TextUtils.isEmpty(blackListEventsString)) {
        String[] blackListEventArrary = blackListEventsString.split(MoEConstants.EVENT_SEPARATOR);
        if (blackListEventArrary != null && blackListEventArrary.length > 0) {
          for (String blackEvent : blackListEventArrary) {
            mBlackListedEvents.add(blackEvent);
          }
        }
      }
    } catch (Exception e) {
      Logger.e("MoEEventManager: getBlackListedEvents() ", e);
    }
  }

  int getEventCounter() {
    return mEventCounter;
  }

  void incrementEventCounter() {
    mEventCounter++;
  }

  void setEventCounter(int counter) {
    mEventCounter = counter;
  }

  /**
   * Tracks an event and if the event is a smart trigger event makes a call to the backend.
   *
   * @param action event name
   * @param attrs event attributes
   */
  public void trackEvent(final String action, JSONObject attrs) {
    try {

      if (!mConfigProvider.isAppEnabled()) return;
      if (mConfigProvider.isDataTrackingOptedOut()) {
        if (!isGDPRWhiteListEvent(action)) {
          Logger.e( TAG + " trackEvent() : Data tracking is opted out and this is not a GDPR "
              + "whitelist event cannot track. Event: " + action);
          return;
        }
      }
      if (isEventBlackListed(action)) {
        Logger.e("MoEEventManager: Event Blacklisted : " + action);
        return;
      }
      Event event = new Event(action.trim(), attrs);
      if (action.equals(MoEHelperConstants.EVENT_APP_INSTALL)) {
        MoEUtils.setInstallRegistered(mContext);
      }
      if (isSmartTriggerEvent(action)) {
        // This is a smart action
        Logger.v("MoEEventManager:acting on auto trigger");
        InAppController.getInstance().registerAutoTriggerEvent(mContext, event);
      }
      //check and show device triggers if possible
      MoEDTManager.getInstance().showTriggerIfPossible(mContext, action, attrs);
      //write data-point to storage
      MoEDispatcher.getInstance(mContext).writeDataPointToStorage(event);
      //check if flush event and flush
      flushIfRequired(event);
    } catch (Exception e) {
      Logger.f("MoEEventManager: trackEvent() ", e);
    }
  }

  /**
   * Checks whether a performed action is a smart trigger event or not
   * @param action The action which needs to be checked
   * @return true if the action is a smart trigger event, false otherwise
   */
  private boolean isSmartTriggerEvent(final String action) {
    return (mTriggerEvents != null && mTriggerEvents.contains(action)) || MoEHelperConstants
        .EVENT_APP_INSTALL.equals(action);
  }

  /**
   * Reads the trigger events list from storage and cache it.
   */
  public void getTriggerEvents(){
    try{
      String list = mConfigProvider.getSmartTriggerList();
      if (list == null){
        Logger.v("MoEEventManager:No smart triggers found");
        return;
      }
      String triggers[] = list.split(MoEConstants.EVENT_SEPARATOR);
      mTriggerEvents = new ArrayList<>(triggers.length);
      for (String event : triggers){
        mTriggerEvents.add(event);
      }
    } catch(Exception e){
      Logger.e("MoEEventManager: getTriggerEvents()", e);
    }
  }

  void getFlushEvents(){
    try {
      String events = mConfigProvider.getFlushEvents();
      if (TextUtils.isEmpty(events)){
        Logger.v("MoEEventManager: getFlushEvents() No flush events");
        return;
      }
      String[] flushEvents = events.split(MoEConstants.EVENT_SEPARATOR);
      if (mFlushEvents == null){
        mFlushEvents = new ArrayList<>();
      }
      for (String event: flushEvents){
        mFlushEvents.add(event);
      }
    }catch (Exception e){
      Logger.e("MoEEventManager: getFlushEvents()");
    }
  }

  private boolean isFlushEvent(String eventName){
    return mFlushEvents != null && mFlushEvents.contains(eventName);
  }

  private void flushIfRequired(@NonNull Event event){
    if (event.eventName != null && isFlushEvent(event.eventName)){
      Logger.v(TAG + " flushIfRequired() flush event : " + event.eventName );
      MoEHelper.getInstance(mContext).syncInteractionDataNow();
    }
  }
  private void addDefaultFlushEvents() {
    if (mFlushEvents == null){
      mFlushEvents = new ArrayList<>();
    }
    mFlushEvents.add(MoEHelperConstants.NOTIFICATION_RECEIVED_MOE);
  }

  private void addDefaultGDPRWhiteList(){
    if (mGDPRWhiteList == null){
      mGDPRWhiteList =  new LinkedHashSet<>();
    }
    mGDPRWhiteList.add(MoEHelperConstants.NOTIFICATION_RECEIVED_MOE);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_NOTIFICATION_CLICKED);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_NOTIFICATION_CLEARED);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_IN_APP_SHOWN);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_IN_APP_CLICKED);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_IN_APP_AUTO_DISMISS);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_IN_APP_CLOSE_CLICKED);
    mGDPRWhiteList.add(MoEHelperConstants.IN_APP_CHECK_FAILURE_EVENT);
    mGDPRWhiteList.add(MoEHelperConstants.EVENT_ACTION_WEB_ACTIVITY_CLICK);
    mGDPRWhiteList.add(MoEConstants.EVENT_ACTION_COUPON_CODE_COPY);
    mGDPRWhiteList.add(MoEHelperConstants.NOTIFICATION_OFFLINE_MOE);
    mGDPRWhiteList.add(MoEHelperConstants.DT_CAMPAIGN_SCHEDULED);
    mGDPRWhiteList.add(MoEConstants.EVENT_ACTION_ACTIVITY_START);
    mGDPRWhiteList.add(MoEHelperConstants.APP_RATED_EVENT);
    mGDPRWhiteList.add(MoEHelperConstants.TOKEN_EVENT);
    mGDPRWhiteList.add(MoEConstants.MOE_APP_EXIT_EVENT);
  }

  void getGDPRWhiteList() {
    if (mGDPRWhiteList == null){
      mGDPRWhiteList =  new LinkedHashSet<>();
    }
    try {
      String events = mConfigProvider.getGDPRWhiteListEvents();
      if (TextUtils.isEmpty(events)){
        Logger.v(TAG + " getGDPRWhiteList() No flush events");
        return;
      }
      String[] gdprWhiteListEvents = events.split(MoEConstants.EVENT_SEPARATOR);
      for (String event: gdprWhiteListEvents){
        mGDPRWhiteList.add(event);
      }
    }catch (Exception e){
      Logger.e(TAG + "getGDPRWhiteList()", e);
    }
  }

  private boolean isGDPRWhiteListEvent(String eventName){
    return mGDPRWhiteList != null && mGDPRWhiteList.contains(eventName);
  }

}
