package com.moengage.core;

import android.app.Activity;
import android.app.Fragment;
import android.app.job.JobParameters;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.os.Bundle;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.executor.ITask;
import com.moengage.core.executor.OnTaskCompleteListener;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskProcessor;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.listeners.OnLogoutCompleteListener;
import com.moengage.inapp.InAppController;
import com.moengage.location.GeoManager;
import com.moengage.push.MoEMessagingManager;
import com.moengage.push.PushManager;
import com.moengage.push.PushManager.PushHandler;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.json.JSONObject;

import static com.moe.pushlibrary.MoEHelper.isAppInForeground;

/**
 * @author Umang Chamaria
 */
public class MoEDispatcher implements OnTaskCompleteListener {

  private static final String TAG = "MoEDispatcher";

  private Context mContext;

  private ConfigurationProvider configProvider;

  private TaskProcessor mTaskProcessor;

  private static MoEDispatcher _INSTANCE;

  //flag to delete data after interaction data is sent.
  private boolean shouldClearData = false;

  /**
   * List of all the running tasks
   */
  private HashMap<String, Boolean> runningTaskList;

  /**
   * Flag to check whether checking for GAID change is required.
   */
  private boolean mNeedToCheckForGAIDChange = true;

  private boolean shouldTrackUniqueId = false;

  private JSONObject uniqueIdAttribute = null;

  private ScheduledExecutorService mScheduler;

  private OnLogoutCompleteListener logoutCompleteListener;

  /**
   * MoEDispatcher constructor. There should be only one instance of
   * MoEDispatcher per task instance of an APP
   *
   * @param context Application Context
   */
  private MoEDispatcher(Context context) {
    if (context != null) {
      mContext = context;
      configProvider = ConfigurationProvider.getInstance(mContext);
      mTaskProcessor = TaskProcessor.getInstance();
      runningTaskList = new HashMap<>();
      //checkAndAddDevice();
      mTaskProcessor.setOnTaskCompleteListener(this);
    } else {
      Logger.e("MoEDispatcher  : context is null");
    }
  }

  public static MoEDispatcher getInstance(Context context) {
    if (_INSTANCE == null) {
      _INSTANCE = new MoEDispatcher(context);
    }
    return _INSTANCE;
  }

  /**
   * Forcefully try to show in-apps
   *
   * @param force if true tries to show in-app
   */
  public void checkForInAppMessages(boolean force) {
    if (!configProvider.isInAppEnabled() || !configProvider.isAppEnabled()) return;
    Logger.v("MoEDispatcher: showInAppIfPossible: Check in app messages");
    if (force) {
      InAppController.getInstance().showInAppIfPossible(mContext);
    }
  }

  /**
   * Lifecycle callback of an activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   * @param intent intent calling that activity
   */
  public void onStart(Activity activity, Intent intent) {
    if (!configProvider.isAppEnabled()) return;
    if (null == activity) {
      Logger.e("MoEDispatcher:onStart activity instance is null");
      return;
    }
    if (null == intent) {
      intent = activity.getIntent();
    }
    mContext = activity.getApplicationContext();
      Logger.v("MoEDispatcher:onStart ----");
      MoEHelperUtils.dumpIntentExtras(intent);
    String currentActivityName = activity.getClass().getName();
    if (!isActivityOptedOut(activity)) {
      addTaskToQueue(
          new ActivityStartTask(mContext, currentActivityName, mNeedToCheckForGAIDChange));
    }
    Context context = activity.getApplicationContext();
    int currentOrientation = context.getResources().getConfiguration().orientation;
    //SYNC INAPPS
    String savedActivityName = InAppController.getInstance().getActivityName();
    int savedOrientation = InAppController.getInstance().getActivityOrientation();
    if (savedActivityName != null && savedOrientation != -1) {
      if (savedActivityName.equals(currentActivityName) && savedOrientation != currentOrientation) {
        InAppController.getInstance().showInAppOnConfigurationChange(mContext);
      } else {
        //sync for in-apps and save current orientation and name
        saveCurrentActivityDetails(currentActivityName, currentOrientation);
        syncInAppsAndGeo();
      }
    } else {
      //sync for in-apps and save current orientation and name
      saveCurrentActivityDetails(currentActivityName, currentOrientation);
      syncInAppsAndGeo();
    }
    saveCurrentActivityDetails(currentActivityName, currentOrientation);

    //THIS is JUST A FAIL OVER IF SOMEONE IS HANDLING RE-DIRECTIONS ON THEIR OWN
    if (intent != null) {
      Bundle extras = intent.getExtras();
      if (extras != null) {
        //REMOVE navigation source
        extras.remove(MoEHelperConstants.NAVIGATION_PROVIDER_KEY);
        extras.remove(MoEHelperConstants.NAVIGATION_SOURCE_KEY);
        //NOT required any more
        //TODO this can be removed - present for legacy issues where some clients handled
        // everything on their own
        PushManager.PushHandler pushHandler = PushManager.getInstance().getPushHandler();
        if (pushHandler != null) {
          pushHandler.logNotificationClicked(context, intent);
        }
      }
    }

    if (MoEHelper.getActivityCounter() == 1) {
      pushTokenFallBack();
    }
    MoEUtils.updateTestDeviceState(mContext);
  }

  private void pushTokenFallBack() {
    PushManager.PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      pushHandler.setPushRegistrationFallback(mContext);
    }
  }

  private void syncInAppsAndGeo() {
    if (!configProvider.isAppEnabled()) return;
    Logger.v("MoEDispatcher: Fetch or query in app message");
    // sync in-app
    InAppController.getInstance().syncOrShowInApps(mContext);
    //sync geo fences
    GeoManager.getInstance().updateFenceAndLocation(mContext);
  }

  /**
   * Logs impression for notification click
   *
   * @param gtime notification clicked time
   */
  public void trackNotificationClicked(long gtime) {
    if (!configProvider.isAppEnabled()) return;
    addTaskToQueue(new NotificationClickedTask(mContext, gtime));
  }

  /**
   * Saves a user attribute
   *
   * @param userJson user attribute in json format
   */
  public void setUserAttribute(JSONObject userJson) {
    addTaskToQueueBeginning(new SetUserAttributeTask(mContext, userJson, true));
  }

  public void setCustomUserAttribute(JSONObject userJson) {
    if (!configProvider.isAppEnabled()) return;
    addTaskToQueueBeginning(new SetUserAttributeTask(mContext, userJson, false));
  }

  /**
   * Tells the sdk whether it is an app update or install when the 1st app with MoEngage goes live.
   *
   * @param existing true if its an update else false
   * @param context Application Context
   */
  public void setExistingUser(boolean existing, Context context) {
    try {
      if (!configProvider.isAppEnabled()) return;
      boolean installRegistered = MoEUtils.isInstallRegistered(context);
      int currentVersion = configProvider.getAppVersion();
      if (existing) {
        int prevVersion = configProvider.getStoredAppVersion();
        if (currentVersion == prevVersion) {
          //already stored
          return;
        }
        configProvider.storeAppVersion(currentVersion);
        JSONObject eventObj =
            new PayloadBuilder().putAttrInt(MoEHelperConstants.FROM_VERSION, prevVersion)
                .putAttrInt(MoEHelperConstants.TO_VERSION, currentVersion)
                .putAttrDate(MoEHelperConstants.TIME_OF_UPDATE, new Date())
                .build();
        MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_UPD, eventObj);
        Logger.v("MoEDispatcher:setExistingUser:tracking update");
      } else if (!installRegistered) {
        configProvider.storeAppVersion(currentVersion);
        PayloadBuilder builder = new PayloadBuilder();
        String referrer = MoEHelperUtils.getInstallReferrer(context);
        if (!TextUtils.isEmpty(referrer)) {
          builder.putAttrString(MoEHelperConstants.PREF_KEY_INSTALL_REFERRER, referrer);
        }
        builder.putAttrInt(MoEHelperConstants.VERSION, currentVersion)
            .putAttrInt(MoEConstants.GENERIC_PARAM_V2_KEY_LIBVERSION,
                MoEHelperConstants.LIB_VERSION)
            .putAttrLong(MoEHelperConstants.TIME_OF_INSTALL, System.currentTimeMillis())
            .putAttrString(MoEConstants.GENERIC_PARAM_V2_KEY_OS,
                MoEConstants.GENERIC_PARAM_V2_VALUE_OS);
        MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_INSTALL, builder.build());
        Logger.v("MoEDispatcher:setExistingUser:tracking install");
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: setExistingUser: ", e);
    }
    //IF EXISTING USER - ADD USER UPDATED APP
    //ELSE NEW USER onboard
  }

  /**
   * Cancel gcm registration fallback.
   */
  void cancelRegistrationFallback() {
    MoEUtils.setRegistrationScheduled(mContext, false);
    MoEUtils.saveCurrentExponentialCounter(mContext, 1);
  }

  /**
   * (non-JavaDoc)
   * Saves the event in the database
   *
   * @param event event to be saved
   */
   void writeDataPointToStorage(Event event) {
    if (!configProvider.isAppEnabled()) return;
    addTaskToQueue(new TrackEventTask(mContext, event));
  }

  public void onResume(Activity activity, boolean isRestoring) {
    if (!configProvider.isAppEnabled()) return;
    if (!isRestoring) {
      showDialogAfterPushClick(activity);
    }
  }

  /**
   * Lifecycle callback of activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   * @param configChange true if orientation change, else false
   */
  public void onStop(Activity activity, boolean configChange) {
    if (!configProvider.isAppEnabled()) return;
    if (null == activity) return;
    if (configChange) return;
    addTaskToQueue(new ActivityStopTask(mContext, activity.getClass().getName()));
  }

  /**
   * Logs out the user. Clears the data and tries to re-register for push in case push
   * registration is handled by MoEngage.
   */
  @WorkerThread void handleLogout(boolean isForcedLogout) {
    Logger.i("Started logout process");
    if (!configProvider.isAppEnabled()) return;
    trackLogoutEvent(isForcedLogout);
    //send batched events to server
    sendInteractionData();
    shouldClearData = true;
  }

  private void trackLogoutEvent(boolean isForcedLogout) {
    try {
      JSONObject eventAttributes = new JSONObject();
      if (isForcedLogout) eventAttributes.put("type", "forced");
      Event event =
          new Event(MoEHelperUtils.getDatapointJSON(MoEConstants.LOGOUT_EVENT, eventAttributes));
      MoEDAO.getInstance(mContext).addEvent(event, mContext);
    } catch (Exception e) {
      Logger.f("MoEDispatcher: trackLogoutEvent(): ", e);
    }
  }

  @WorkerThread private void clearDataOnLogout() {
    //truncate all tables
    mContext.getContentResolver()
        .delete(MoEDataContract.DatapointEntity.getContentUri(mContext), null, null);
    mContext.getContentResolver()
        .delete(MoEDataContract.MessageEntity.getContentUri(mContext), null, null);
    mContext.getContentResolver()
        .delete(MoEDataContract.InAppMessageEntity.getContentUri(mContext), null, null);
    mContext.getContentResolver()
        .delete(MoEDataContract.UserAttributeEntity.getContentUri(mContext), null, null);
    mContext.getContentResolver().delete(MoEDataContract.CampaignListEntity.getContentUri
        (mContext), null, null);
    mContext.getContentResolver().delete(MoEDataContract.BatchDataEntity.getContentUri(mContext),
        null, null);
    mContext.getContentResolver().delete(MoEDataContract.DTEntity.getContentUri(mContext), null,
        null);
    //remove shared preferences associated with a user
    configProvider.removeUserConfigurationOnLogout();
    //call device add and user add
    configProvider.setDeviceRegistered(false);
    notifyLogoutCompleteListener();
    PushManager.PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      //pushHandler.deleteToken(mContext, null);
      pushHandler.registerForPushToken(mContext);
    }
    shouldClearData = false;
    Logger.i("Completed logout process");
  }

  /**
   * Sends all the tracked events to the server.
   */
  public void sendInteractionData() {
    if (!configProvider.isAppEnabled()) return;
    addTaskToQueue(new CreatingDataBatchTask(mContext));
  }

  public void sendInteractionData(OnJobComplete jobComplete, JobParameters parameters){
    if (!configProvider.isAppEnabled()) return;
    addTaskToQueueBeginning(new CreatingDataBatchTask(mContext, jobComplete, parameters));
  }

  /**
   * Equivalent to the lifecycle callback of {@link Fragment}. Required to show in-apps in
   * fragment.
   *
   * @param activity Parent Activity of the fragment
   * @param fragmentName Name of the calling fragment
   */
  public void onFragmentStart(Activity activity, String fragmentName) {
    syncInAppsAndGeo();
  }

  /**
   * Updates the local DB that the specified message has been clicked<br>
   * <b>Note : Don't call on UI Thread</b>.
   *
   * @param id The id associated with the inbox message that was clicked
   */
  @WorkerThread public void setInboxMessageClicked(long id) {
    MoEDAO.getInstance(mContext).setMessageClicked(id);
  }

  /**
   * Handles the app update event, registers,and tries to get new GCM registration ID<br>
   * <b>Note : Don't call on UI Thread</b>
   */
  @WorkerThread void handleAppUpdateEvent() {
    //Logging an update event
    try {
      if (!configProvider.isAppEnabled()) return;
      int prevVersion = configProvider.getStoredAppVersion();
      JSONObject eventObj = new JSONObject();
      eventObj.put(MoEHelperConstants.FROM_VERSION, prevVersion);
      eventObj.put(MoEHelperConstants.TO_VERSION, configProvider.getAppVersion());
      Event event =
          new Event(MoEHelperUtils.getDatapointJSON(MoEHelperConstants.EVENT_APP_UPD, eventObj));
      //update token
      MoEDAO.getInstance(mContext).addEvent(event, mContext);
      if (!isAppInForeground()) {
        sendInteractionData();
      }
      Logger.i("Adding an update event");
    } catch (Exception e) {
      Logger.f("Adding update event", e);
    }
      dispatchPushTask(PushManager.REQ_REGISTRATION);
  }

  /**
   * Get a all inbox messages<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return A populated cursor with the inbox messages or null
   */
  @WorkerThread public Cursor getAllMessages() {
    return MoEDAO.getInstance(mContext).getMessages(mContext);
  }

  /**
   * Returns the number of unread messages. Unread messages are counted based on clicks<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return The Count of unread messages
   */
  @WorkerThread public int getUnreadMessageCount() {
    return MoEDAO.getInstance(mContext).getUnreadMessageCount();
  }

  /**
   * Initializes the MoEngage SDK
   *
   * @param senderId GCM Project's Sender ID
   * @param appId Application Id from the MoEngage Dashboard
   */

  public void initialize(final String senderId, final String appId) {
    if (!TextUtils.isEmpty(appId)) {
      configProvider.saveAppDetails(senderId, appId);
        //force registration for push
      dispatchPushTask(PushManager.REQ_REGISTRATION);
    } else {
      Logger.e("MoEDispatcher: initialize : AppId is null");
    }
  }

  public void checkAndShowLinkedInApp(String campaignId) {
    InAppController.getInstance().fetchLinkedInApp(mContext, campaignId);
  }

  /**
   * Show a dialog after push was clicked
   *
   * @param activity The instance of the activity where the dialog needs to be created
   */
  private void showDialogAfterPushClick(Activity activity) {
    if (null == activity) return;
    try {
      Intent intent = activity.getIntent();
      if (intent != null) {
        Bundle extras = intent.getExtras();
        if (extras != null && extras.containsKey(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG)) {
          //remove the extra
          intent.removeExtra(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG);
          if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_COUPON_CODE)) {
            MoEUtils.showCouponDialog(extras.getString(MoEHelperConstants.GCM_EXTRA_CONTENT),
                extras.getString(MoEHelperConstants.GCM_EXTRA_COUPON_CODE), activity);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_CONTENT);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_COUPON_CODE);
          } else {
            MoEUtils.showNormalDialogWithOk(extras.getString(MoEHelperConstants.GCM_EXTRA_CONTENT),
                activity);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_CONTENT);
          }
        }
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: showDialogAfterPushClick : ", e);
    }
  }


  /**
   * (non-JavaDoc)
   * Save the activity name and activity orientation.
   * Details required for showing notifications on orientation change
   *
   * @param currActivityName activity name to be saved
   * @param currentOrientation activity orientation name
   */
  private void saveCurrentActivityDetails(String currActivityName, int currentOrientation) {
    InAppController.getInstance().setActivityName(currActivityName);
    InAppController.getInstance().setActivityOrientation(currentOrientation);
  }

  /**
   * Checks whether the task is synchronous or not and adds to the processing queue accordingly.
   * If task is synchronous and is in runningTaskList do not add to processing queue else add
   * in processing queue
   *
   * @param task task to be add to the processor
   */
  public void addTaskToQueue(ITask task) {
    Logger.v("Trying to add " + task.getTaskTag() + " to the queue");
    if (task.isSynchronous()) {
      if (!runningTaskList.containsKey(task.getTaskTag())) {
        Logger.v(task.getTaskTag() + " added to queue");
        runningTaskList.put(task.getTaskTag(), task.isSynchronous());
        mTaskProcessor.addTask(task);
      }
    } else {
      Logger.v(task.getTaskTag() + " added to queue");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      mTaskProcessor.addTask(task);
    }
  }

  /**
   * Checks whether the task is synchronous or not and adds to the processing queue accordingly.
   * If task is synchronous and is in runningTaskList do not add to processing queue else add
   * to front of processing queue
   *
   * @param task task to be add to the processor
   */
  public void addTaskToQueueBeginning(ITask task) {
    Logger.v("Trying to add " + task.getTaskTag() + " to the queue");
    if (task.isSynchronous()) {
      if (!runningTaskList.containsKey(task.getTaskTag())) {
        Logger.v(task.getTaskTag() + " added to beginning of queue");
        runningTaskList.put(task.getTaskTag(), task.isSynchronous());
        mTaskProcessor.addTaskToFront(task);
      }
    } else {
      Logger.v(task.getTaskTag() + " added to beginning of queue");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      mTaskProcessor.addTaskToFront(task);
    }
  }

  private final Object lock = new Object();

/*  private void checkAndAddDevice() {
    synchronized (lock) {
      try {
        if (!configProvider.isAppEnabled()) return;
        if (!configProvider.isPushRegistrationEnabled()) {
          if (!TextUtils.isEmpty(configProvider.getGCMToken())
              && !configProvider.isDeviceRegistered()) {
            addTaskToQueue(new DeviceAddTask(mContext));
          }
        } else if (!configProvider.isDeviceRegistered()) {
          MoEUtils.setRegistrationScheduled(mContext, true);
          dispatchPushTask(PushManager.REQ_REGISTRATION);
        }
      } catch (Exception e) {
        Logger.f("MoEDispatcher:checkAndAddDevice", e);
      }
    }
  }*/

  @Override public void onTaskComplete(String tag, TaskResult taskResult) {
    // removing task from running list
    Logger.v("Task completed : " + tag);
    if (runningTaskList.containsKey(tag)) {
      runningTaskList.remove(tag);
    }
    switch (tag) {
      case SDKTask.TAG_ACTIVITY_START:
        if (taskResult.isSuccess()) {
          mNeedToCheckForGAIDChange = (boolean) taskResult.getPayload();
        }
        break;
      case SDKTask.TAG_INAPP_NETWORK_TASK:
        if (InAppController.NETWORK_CALL_TYPE.SYNC_IN_APPS.equals(taskResult.getPayload())) {
          InAppController.getInstance().setInAppSynced(taskResult.isSuccess(), mContext);
        }
        break;
      case SDKTask.TAG_SEND_INTERACTION_DATA:
        if (shouldClearData) {
          clearDataOnLogout();
          if (shouldTrackUniqueId) trackChangedUniqueId();
        }
        break;
      case SDKTask.TAG_SET_USER_ATTRIBUTES:
        if (!taskResult.isSuccess()){
          shouldTrackUniqueId = true;
          uniqueIdAttribute = (JSONObject) taskResult.getPayload();
        }
        break;
      case SDKTask.TAG_DEVICE_ADD:
        MoEDTManager.getInstance().forceSyncDeviceTriggers(mContext);
        break;

    }
  }

  void resetStates() {
    mNeedToCheckForGAIDChange = true;
  }

  private List<String> optedOutActivities;

  private boolean isActivityOptedOut(Activity activity) {
    try {
      if (optedOutActivities == null) {
        optedOutActivities = configProvider.getTrackingOptedOutActivities();
      }
      if (optedOutActivities != null && optedOutActivities.contains(
          activity.getClass().getName())) {
        return true;
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher#isActivityOptedOut Exception Occurred" + e);
    }
    return false;
  }

  private void dispatchPushTask(String extra) {
    PushManager.PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      pushHandler.offLoadToWorker(mContext, extra);
    }
  }

  public void logPushFailureEvent(Context context, String e) {
    // intentionally not removed. Can cause crashes when main sdk and add on not updated in tandem.
  }


  private void syncConfigIfRequired() {
    if (configProvider.getLastConfigSyncTime() + 3600000 < System.currentTimeMillis()) {
      addTaskToQueueBeginning(new SyncConfigAPITask(mContext));
    }
  }

  public void trackDeviceLocale() {
    try {
      if (!configProvider.isAppEnabled()) return;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY", Locale.getDefault().getCountry());;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_DISPLAY", Locale.getDefault().getDisplayCountry());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE", Locale.getDefault().getLanguage());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_DISPLAY", Locale.getDefault().getDisplayLanguage());
      trackDeviceAndUserAttribute("LOCALE_DISPLAY", Locale.getDefault().getDisplayName());
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_ ISO3", Locale.getDefault().getISO3Country());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_ISO3", Locale.getDefault().getISO3Language());
    } catch (Exception e) {
      Logger.f("MoEDispatcher : trackDeviceLocale", e);
    }
  }

  private void trackDeviceAndUserAttribute(String attrName, String attrValue){
    try{
      JSONObject attribute = new JSONObject();
      attribute.put(attrName, attrValue);
      setUserAttribute(attribute);
      //setDeviceAttribute(attribute);
    } catch(Exception e){
      Logger.f("MoEDispatcher: trackDeviceAndUserAttribute() ", e);
    }
  }

  public void logoutUser(final boolean isForcedLogout) {
    try{
      Bundle extras = new Bundle();
      extras.putBoolean(MoEConstants.SERVICE_LOGOUT_TYPE, isForcedLogout);
      addTaskToQueue(new MoEWorkerTask(mContext, MoEConstants.SERVICE_TYPE_LOGOUT, extras));
    } catch(Exception e){
      Logger.f("MoEDispatcher: logoutUser() ", e);
    }
  }

  private void trackChangedUniqueId(){
    if (uniqueIdAttribute != null){
      setUserAttribute(uniqueIdAttribute);
      uniqueIdAttribute = null;
      shouldTrackUniqueId = false;
    }
  }

  public void setAlias(JSONObject aliasJSON){
   if (!configProvider.isAppEnabled()) return;
    addTaskToQueue(new SetAliasTask(mContext, aliasJSON));
  }

  void setDeviceAttribute(JSONObject deviceAttribute){
    addTaskToQueue(new SetDeviceAttributeTask(mContext, deviceAttribute));
  }

   private void schedulePeriodicFlushIfRequired() {
    try {
      if (configProvider.isPeriodicFlushEnabled() && MoEHelper.getInstance(mContext)
          .getPeriodicSyncState()) {
        Runnable syncRunnable = new Runnable() {
          @Override public void run() {
            Logger.v("MoEDispatcher: schedulePeriodicFlushIfRequired() inside runnable, will sync "
                + "now");
            sendInteractionData();
          }
        };
        long timeDelay = configProvider.getPeriodicFlushTime();
        if (MoEHelper.getInstance(mContext).getFlushInterval() > timeDelay){
          timeDelay = MoEHelper.getInstance(mContext).getFlushInterval();
        }
        Logger.v("MoEDispatcher: schedulePeriodicFlushIfRequired() scheduling periodic sync");
        mScheduler = Executors.newScheduledThreadPool(1);
        mScheduler.scheduleWithFixedDelay(syncRunnable, timeDelay, timeDelay, TimeUnit.SECONDS);
      }
    } catch (Exception e) {
      Logger.e("MoEDispatcher: schedulePeriodicFlushIfRequired() ", e);
    }
  }

  void shutDownPeriodicFlush() {
    try {
      if (configProvider.isPeriodicFlushEnabled() && MoEHelper.getInstance(mContext)
          .getPeriodicSyncState() && mScheduler != null) {
        Logger.v("MoEDispatcher: shutDownPeriodicFlush() shutting down periodic flush");
        mScheduler.shutdownNow();
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: shutDownPeriodicFlush() ", e);
    }
  }

  public void onAppOpen(){
    try {
      syncConfigIfRequired();
      MoEMessagingManager.MessagingHandler messagingHandler = MoEMessagingManager.getInstance()
          .getMessagingHandler(mContext);
      if (messagingHandler != null){
        messagingHandler.forceMessageSync(mContext, true);
      }
      PushHandler pushHandler = PushManager.getInstance().getPushHandler();
      if (pushHandler != null){
        pushHandler.offLoadToWorker(mContext, PushManager.REG_ON_APP_OPEN);
      }
      schedulePeriodicFlushIfRequired();
      MoEDTManager.getInstance().forceSyncDeviceTriggers(mContext);
      updateFeatureConfigForOptOutIfRequired();
    } catch (Exception e) {
      Logger.f("MoEDispatcher: onAppOpen() ", e);
    }
  }

  void updateFeatureConfigForOptOutIfRequired(){
    ConfigurationProvider provider = ConfigurationProvider.getInstance(mContext);
    if (provider.isDataTrackingOptedOut()){
      // gaid optOut
      provider.optOutOfAdIdCollection(true);
      //androidId optOut
      provider.optOutOfAndroidIdCollection(true);
      //imei optOut
      provider.optOutOfIMEICollection(true);
      //location optOut
      provider.optOutOfTrackLocation(true);
      //geo-fence optOut
      provider.optOutOfSetGeoFence(true);
      //device attributes optOut
      provider.optOutOfDeviceAttributesCollection(true);
      //setting location services state
      provider.setLocationServicesState(false);
    }
    if (provider.isPushNotificationOptedOut()){
      provider.setGCMToken("");
    }
  }

  public void setOnLogoutCompleteListener(OnLogoutCompleteListener listener){
    logoutCompleteListener = listener;
  }

  public void removeOnLogoutCompleteListener(){
    logoutCompleteListener = null;
  }

  private void notifyLogoutCompleteListener() {
    Logger.v(TAG + " notifyLogoutCompleteListener() : ");
    if (logoutCompleteListener != null){
      logoutCompleteListener.logoutComplete();
    }
  }
}
