package com.moengage.core;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import com.moe.pushlibrary.models.BatchData;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import java.util.ArrayList;

/**
 * (non-JavaDoc)
 * Sends all the events to the server once the application is closed.
 * Service constant MSG_SEND_INTERACTION_DATA
 *
 * @author Umang Chamaria
 */
class SendInteractionDataTask extends SDKTask {
  SendInteractionDataTask(Context context) {
    super(context);
  }

  @Override public TaskResult execute() {
    try {
      if (!ConfigurationProvider.getInstance(mContext).isAppEnabled()) return null;
      Logger.v("SendInteractionDataTask : executing task");
      sendInteractionData();
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
        scheduleRetryDataSyncJob();
      }else {
        scheduleRetryDataSyncAlarm();
      }
      Logger.v("SendInteractionDataTask : completed task");
      return null;
    } catch (Exception e) {
      Logger.f("SendInteractionData: execute() ", e);
    }
    return null;
  }

  @Override public String getTaskTag() {
    return TAG_SEND_INTERACTION_DATA;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private void sendInteractionData() {
    boolean result = false;
    ArrayList<BatchData> batchedData = null;
    String endpoint =
        MoEConstants.API_ENDPOINT_STORE_REPORTS_V2 + "/" + MoEUtils.addDebugIfRequired(mContext,
            ConfigurationProvider.getInstance(mContext).getAppId());
    if (shouldSendDataToTestServer()) {
      endpoint = MoEConstants.API_ENDPOINT_INTEGRATION_VERIFICATION_REPORT_ADD;
    }
    for (; ; ) {
      batchedData = MoEDAO.getInstance(mContext).getBatchedData(100);
      Logger.d(
          "SendInteractionDataTask : sendInteractionData:Fetching interaction data in batches");
      if (batchedData == null || batchedData.isEmpty()) {
        Logger.d("SendInteractionDataTask : sendInteractionData: Found Nothing to send");
        return;
      }
      for (BatchData data: batchedData){
        try {
          result = APIManager.sendInteractionReport(mContext, data.batchData, endpoint);
        }catch (Exception e) {
          Logger.f("SendInteractionDataTask : API failed", e);
          result = false;
        }
        if (result) {
          Logger.d("SendInteractionDataTask : Batch sent successfully deleting batch");
          MoEDAO.getInstance(mContext).deleteBatch(data);
        } else {
          int retryCount = ConfigurationProvider.getInstance(mContext).getImmediateRetryCount();
          switch(retryCount){
            case 0:
              scheduleImmediateRetry(1);
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(++retryCount);
              break;
            case 1:
              scheduleImmediateRetry(3);
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(++retryCount);
              break;
            default:
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(0);
          }
          break;
        }
      }
      if (!result) break;
      batchedData.clear();
    }
  }

  /**
   * Checks whether device is registered as a test device or not
   * @return true if registered as test device, else false
   */
  private boolean shouldSendDataToTestServer() {
    boolean isDeviceRegisteredForVerification =
        ConfigurationProvider.getInstance(mContext).isDeviceRegisteredForVerification();
    long registrationTime =
        ConfigurationProvider.getInstance(mContext).getVerificationRegistrationTime();
    return isDeviceRegisteredForVerification && ((registrationTime + 3600 * 1000)
        > System.currentTimeMillis());
  }

  /**
   * Scheduling a periodic retry to send
   */
  private void scheduleRetryDataSyncAlarm(){
    Logger.v("Scheduling data sync retry");
    Intent alarmIntent = new Intent(mContext, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(mContext, 88888, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
    alarmManager.setInexactRepeating(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + ConfigurationProvider
        .getInstance(mContext).getRetrySyncTime(), ConfigurationProvider.getInstance
        (mContext).getRetrySyncTime(), pendingIntent);
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  private void scheduleRetryDataSyncJob(){
    Logger.v("Scheduling retry data sync job");
    ComponentName serviceComponent = new ComponentName(mContext, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(77777, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(System.currentTimeMillis() + ConfigurationProvider.getInstance
        (mContext).getRetrySyncTime() * 2);
    builder.setMinimumLatency( ConfigurationProvider.getInstance
        (mContext).getRetrySyncTime());
    JobScheduler jobScheduler =
        (JobScheduler) mContext.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    jobScheduler.schedule(builder.build());
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  private void scheduleImmediateRetrySyncJob(int minutes){
    Logger.v("Scheduling immediate retry data sync job");
    ComponentName serviceComponent = new ComponentName(mContext, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(666666, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(System.currentTimeMillis() + minutes * 2 * 60 * 1000);
    builder.setMinimumLatency(minutes * 60 * 1000);
    JobScheduler jobScheduler =
        (JobScheduler) mContext.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  private void scheduleImmediateRetryAlarm(int minutes){
    Intent alarmIntent = new Intent(mContext, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(mContext, 55555, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + (minutes * 60 * 1000),
          pendingIntent);
    }
  }

  private void scheduleImmediateRetry(int minutes){
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
      scheduleImmediateRetrySyncJob(minutes);
    }else {
      scheduleImmediateRetryAlarm(minutes);
    }
  }
}
