/* ************************************************************************
 * 
 * MOENGAGE CONFIDENTIAL
 * __________________
 * 
 *  [2014] - [2015] MoEngage Inc. 
 *  All Rights Reserved.
 * 
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.core;

import android.content.Context;
import android.os.Build;
import android.text.TextUtils;
import com.moe.pushlibrary.exceptions.SDKNotInitializedException;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.push.PushManager;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import javax.net.ssl.HttpsURLConnection;

import static com.moengage.core.MoEUtils.addDebugIfRequired;

/**
 * @author MoEngage (abhishek@moengage.com)
 * @version 3.0
 * @since 1.0
 */
final class MoERestClient {


  public enum RequestMethod {
    GET, POST
  }

  public enum API_VERSION {
    V1, V2
  }

  private static final String ENCODING_CHARSET_UTF8 = "UTF-8";

  private static final String URL_PARAM_SEPARATOR = "&";
  private static final String URL_PARAM_ASSIGNER = "=";
  private static final String URL_QUERY_PARAM_SEPARATOR = "?";
  private static String ANDROID_ID = null;
  private static boolean androidIDRetrieved = false;

  private HashMap<String, String> params;
  private String mStringBody = null;
  private byte[] mByteArray = null;
  private String url;

  private int responseCode;

  private String response;

  private String errorResponse;

  private String appID;

  public String getResponse() {
    return response;
  }

  int getResponseCode() {
    return responseCode;
  }

  /**
   * @throws SDKNotInitializedException
   */
  MoERestClient(String url, Context con, API_VERSION apiVersion)
      throws SDKNotInitializedException {
    this.url = url;
    params = new HashMap<>();
    if (!androidIDRetrieved) {
      androidIDRetrieved = true;
      ANDROID_ID = MoEUtils.getAndroidID(con);
    }
    appID = MoEUtils.addDebugIfRequired(con, ConfigurationProvider.getInstance(con).getAppId());
    if (API_VERSION.V1 == apiVersion) {
      initializeRestClientV1(con);
    } else {
      initializeRestClientV2(con);
    }
  }

  private void initializeRestClientV1(Context con) throws SDKNotInitializedException {
    params.put(
        MoEConstants.GENERIC_PARAM_V1_KEY_OS, MoEConstants.GENERIC_PARAM_V1_VALUE_ANDROID);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(con);
    String registrationId = provider.getGCMToken();
    String appId = provider.getAppId();

    String newUniqueID = provider.getCurrentUserId();
    appId = addDebugIfRequired(con, appId);

    if (!TextUtils.isEmpty(registrationId)) {
      addParam(MoEConstants.GENERIC_PARAM_V1_KEY_GCM_ID, registrationId);
    }
    if (!TextUtils.isEmpty(appId)) {
      addParam(MoEConstants.GENERIC_PARAM_V1_KEY_APP_ID, appId);
    } else {
      appId = provider.getAppId();
      if (TextUtils.isEmpty(appId)) {
        // check added for segment integration. In case of segment the APP-ID and sender id can
        // be passed from the segment dashboard and we get it back on initialize callback of
        // segment.
        throw new SDKNotInitializedException("APP ID has not been set");
      } else {
        addParam(MoEConstants.GENERIC_PARAM_V1_KEY_APP_ID, appId);
      }
    }
    if (!TextUtils.isEmpty(newUniqueID)) {
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_UUID, newUniqueID);
    }
    addParam(MoEConstants.GENERIC_PARAM_V1_KEY_APP_VERSION,
        Integer.toString(provider.getAppVersion()));
    addParam(MoEConstants.GENERIC_PARAM_V1_KEY_LIB_VERSION,
        Integer.toString(MoEHelperConstants.LIB_VERSION));
    addBaiduParamIfRequired();
    addSegmentParamIfRequired(con);
  }

  private void initializeRestClientV2(Context con) throws SDKNotInitializedException {
    //["os", "app_id", "os_ver", "sdk_ver", "model", "app_ver","device_ts","device_tz", "unique_id"]
    ConfigurationProvider provider = ConfigurationProvider.getInstance(con);
    String registrationId = provider.getGCMToken();
    String appId = provider.getAppId();
    String newUniqueID = provider.getCurrentUserId();
    String appVersion = Integer.toString(provider.getAppVersion());
    long millis = System.currentTimeMillis();
    appId = MoEUtils.addDebugIfRequired(con, appId);
    if (!TextUtils.isEmpty(registrationId) && !provider.isPushNotificationOptedOut()) {
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_GCM_ID, registrationId);
    }
    if (!TextUtils.isEmpty(appId)) {
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_APP_ID, appId);
    } else {
      appId = provider.getAppId();
      if (TextUtils.isEmpty(appId)) {
        throw new SDKNotInitializedException("APP ID has not been set");
      } else {
        addParam(MoEConstants.GENERIC_PARAM_V2_KEY_APP_ID, appId);
      }
    }
    if (!TextUtils.isEmpty(newUniqueID)) {
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_UUID, newUniqueID);
    }
    if (!TextUtils.isEmpty(appVersion)) {
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION, appVersion);
    }

    addParam(
        MoEConstants.GENERIC_PARAM_V2_KEY_OS, MoEConstants.GENERIC_PARAM_V2_VALUE_OS);
    addParam(MoEConstants.GENERIC_PARAM_V2_KEY_LIBVERSION,
        Integer.toString(MoEHelperConstants.LIB_VERSION));

    addParam(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE_OFFSET,
        String.valueOf(TimeZone.getDefault().getOffset(millis)));
    addParam(MoEConstants.GENERIC_PARAM_V2_KEY_TIMESTAMP, String.valueOf(millis));
    addParam(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());
    addPlatformIfRequired(provider);
    addBaiduParamIfRequired();
    addSegmentParamIfRequired(con);

    if (!provider.isDataTrackingOptedOut()){
      if (!TextUtils.isEmpty(ANDROID_ID)) {
        addParam(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, ANDROID_ID);
      }
      if (!provider.isAdIdCollectionProhibitted()) {
        String gaid = provider.getStoredGAID();
        if (TextUtils.isEmpty(gaid)) {
          AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(con);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!TextUtils.isEmpty(gaid)){
          addParam(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL);
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME, provider.getAppVersionName());

      String nwType = MoEUtils.getNetworkType(con);
      if (!TextUtils.isEmpty(nwType)) {
        addParam(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
      }
    }
  }

  /**
   * Add a request param
   * @param name Name of the get param
   * @param Value Value of the get param
   */
  private void addParam(String name, String Value) {
    params.put(name, Value);
  }

  void addParam(HashMap<String, String> paramsMap){
    for (Map.Entry<String, String> entry : paramsMap.entrySet()) {
      params.put(entry.getKey(), entry.getValue());
    }
  }

  /**
   * Add a request body
   * @param requestBody The request body which needs to be added to the HTTP request
   */
  void addBody(String requestBody) {
    mStringBody = requestBody;
  }

  /**
   * Convert input stream to String
   *
   * @param inputStream The input stream from the API response entity
   * @return String representation of the API response mStringBody
   */
  private String convertStreamToString(InputStream inputStream) {
    if (inputStream == null) return null;
    BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
    StringBuilder sb = new StringBuilder();

    String line;
    try {
      while ((line = reader.readLine()) != null) {
        sb.append(line);
      }
    } catch (IOException e) {
      Logger.f("MoERestClient:executeRequest: IOException", e);
    } catch (Exception e){
      Logger.f("MoERestClient:executeRequest: Exception", e);
    }finally {
      try {
        inputStream.close();
      } catch (IOException e) {
        Logger.f("MoERestClient:executeRequest: IOException", e);
      }catch (Exception e) {
        Logger.f("MoERestClient:executeRequest: Exception", e);
      }
    }
    return sb.toString();
  }



  private String getFinalURI(String URI) {
    StringBuilder builder = new StringBuilder((URI));
    if (!params.isEmpty()) {
      builder.append(URL_QUERY_PARAM_SEPARATOR);
      int size = params.size();
      Set<Map.Entry<String, String>> mapParams = params.entrySet();
      int index = 0;
      for (Map.Entry<String, String> param : mapParams) {
        try {
          builder.append(param.getKey());
          builder.append(URL_PARAM_ASSIGNER);
          builder.append(URLEncoder.encode(param.getValue(), ENCODING_CHARSET_UTF8));
          if (index <= size - 2) {
            builder.append(URL_PARAM_SEPARATOR);
          }
          index++;
        } catch (Exception e) {
          Logger.f("MoERestClient: getFinalURI ", e);
        }
      }
    }
    return builder.toString();
  }

  private static final String SCHEME_HTTPS = "https://";

  public void execute(RequestMethod method) throws IOException {
    URL finalURL = new URL(getFinalURI(url));
    Logger.d("MoERestClient: executing API: " + finalURL.toString());
    InputStream in;
    HttpURLConnection urlConnection = null;
    if (url.startsWith(SCHEME_HTTPS)){
      urlConnection = (HttpsURLConnection) urlConnection;
      urlConnection = (HttpsURLConnection) finalURL.openConnection();
    }else {
      urlConnection = (HttpURLConnection) finalURL.openConnection();
    }
    if (method == RequestMethod.POST) {
      addBody(urlConnection);
    }else{
      urlConnection.setRequestMethod("GET");
      urlConnection.addRequestProperty("X-MOENGAGE-APP-KEY", appID);
    }
    responseCode = urlConnection.getResponseCode();
    Logger.d("MoERestClient: ResponseCode: " + responseCode);
    if (200 != responseCode) {
      errorResponse = convertStreamToString(urlConnection.getErrorStream());
        Logger.f("MoERestClient: Response: API Failed: " + url + " response code :" +
            responseCode + "reason : " + errorResponse);
        if (!TextUtils.isEmpty(errorResponse))
          Logger.f("MoERestClient: with reason: " + errorResponse);
      return;
    }
    in = urlConnection.getInputStream();
    response = convertStreamToString(in);
    urlConnection.disconnect();
    if (!TextUtils.isEmpty(response)) Logger.d("MoERestClient: Response: " + response);
  }

  private void addBody(HttpURLConnection connection) throws IOException {

    connection.setDoOutput(true);
    connection.setRequestProperty("Accept-Charset", "UTF-8");
    connection.setRequestProperty("Content-type", "application/json");
    connection.addRequestProperty("X-MOENGAGE-APP-KEY", appID);

    // Write to the connection
    OutputStream output = connection.getOutputStream();
    if (null != mStringBody) {
      Logger.d("MoERestClient: addBody: string: "+ mStringBody);
      output.write(mStringBody.getBytes("UTF-8"));
    }else if( null != mByteArray){
      Logger.d("MoERestClient: addBody: bytes: "+ mByteArray.toString());
      output.write(mByteArray);
    }
    output.close();
  }

  private void addPlatformIfRequired(ConfigurationProvider provider){
    String platform = provider.getUnityVersion();
    if (!TextUtils.isEmpty(platform)){
      addParam("unity_ver", platform);
    }
  }
  private void addBaiduParamIfRequired(){
    if (PushManager.getInstance().isIsBaiduEnabled()){
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_PUSH_SERVER, MoEConstants
          .GENERIC_PARAM_V2_VALUE_PUSH_SERVER_BAIDU);
    }else {
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_PUSH_SERVER, MoEConstants
          .GENERIC_PARAM_V2_VALUE_PUSH_SERVER_ANDROID);
    }
  }

  private void addSegmentParamIfRequired(Context context){
    if (ConfigurationProvider.getInstance(context).isSegmentEnabled()){
      addParam(MoEConstants.GENERIC_PARAM_V2_KEY_INTEGRATION_TYPE, "segment");
    }
  }
}
