/* ************************************************************************
 * 
 * MOENGAGE CONFIDENTIAL
 * __________________
 * 
 *  [2014] - [2015] MoEngage Inc. 
 *  All Rights Reserved.
 * 
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.core;

import android.content.ContentProviderOperation;
import android.content.ContentValues;
import android.content.Context;
import android.content.OperationApplicationException;
import android.database.Cursor;
import android.net.Uri;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.models.BatchData;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.models.UserAttribute;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.BatchDataEntity;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DatapointEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppMessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.UserAttributeEntity;
import java.util.ArrayList;

/**
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.3
 * @since 5.0
 */
public final class MoEDAO {

  public static MoEDAO getInstance(Context context) {
    if (null == _INSTANCE) {
      _INSTANCE = new MoEDAO(context);
    }
    return _INSTANCE;
  }

  private static MoEDAO _INSTANCE = null;

  private Uri MESSAGES_CONTENT_URI = null;
  private Uri INAPP_CONTENT_URI = null;
  private Uri DATAPOINTS_CONTENT_URI = null;
  private Uri USER_ATTRIBUTES_URI = null;
  private Uri CAMPAIGN_LIST_URI = null;
  private Uri BATCHED_DATA_URI = null;

  private String AUTHORITY = null;
  private Context mContext;

  private MoEDAO(Context context) {
    MESSAGES_CONTENT_URI = MessageEntity.getContentUri(context);
    INAPP_CONTENT_URI = InAppMessageEntity.getContentUri(context);
    DATAPOINTS_CONTENT_URI = DatapointEntity.getContentUri(context);
    USER_ATTRIBUTES_URI = UserAttributeEntity.getContentUri(context);
    CAMPAIGN_LIST_URI = CampaignListEntity.getContentUri(context);
    BATCHED_DATA_URI = BatchDataEntity.getContentUri(context);
    AUTHORITY = MoEDataContract.getAuthority(context);
    mContext = context;
  }

  int getUnreadMessageCount() {
    Cursor cur = mContext.getContentResolver()
        .query(MESSAGES_CONTENT_URI, MessageEntity.PROJECTION, MessageEntity.MSG_CLICKED + " = ?",
            new String[] { "0" }, MessageEntity.DEFAULT_SORT_ORDER);
    int unReadCount = 0;
    if (null != cur) {
      unReadCount = cur.getCount();
      closeCursor(cur);
    }
    Logger.v("Getting Unread PromotionalMessage Count: count=" + unReadCount);

    return unReadCount;
  }

  void addEvent(Event event, Context context) {
    if (null == event) {
      Logger.v("Null event passed, skipping it");
      return;
    }
    Logger.v("Event : " + event.details);
    ContentValues values = new ContentValues();
    values.put(DatapointEntity.GTIME, event.gtime);
    values.put(DatapointEntity.DETAILS, event.details);
    Uri newRecord = context.getContentResolver().insert(DATAPOINTS_CONTENT_URI, values);
    if (null != newRecord) {
      Logger.v("New Event added with Uri: " + newRecord.toString());
    } else {
      Logger.v("Unable to add event");
    }
  }

  /**
   * Gets the events from datapoints table, if present.
   * @param batchSize maximum number of events to be returned
   * @return user events if present else null
   */
  ArrayList<Event> getInteractionData(int batchSize) {
    Uri CONTENT_URI = DATAPOINTS_CONTENT_URI.buildUpon()
        .appendQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT, String.valueOf(batchSize))
        .build();
    final Cursor cur = mContext.getContentResolver()
        .query(CONTENT_URI, DatapointEntity.PROJECTION, null, null, DatapointEntity.GTIME + " ASC");
    if (null == cur || cur.getCount() == 0) {
      Logger.v("Empty cursor");
      closeCursor(cur);
      return null;
    }

    ArrayList<Event> eventList = new ArrayList<Event>();
    while (cur.moveToNext()) {
      eventList.add(new Event(cur.getInt(DatapointEntity.COLUMN_INDEX_ID),
          cur.getString(DatapointEntity.COLUMN_INDEX_DETAILS)));
    }
    closeCursor(cur);
    return eventList;
  }

  /**
   * Deletes passed events from the datapoints table
   * @param events events to be deleted
   * @param context application context
   */
  void deleteInteractionData(ArrayList<Event> events, Context context) {
    ArrayList<ContentProviderOperation> operations = new ArrayList<ContentProviderOperation>();
    ContentProviderOperation operation;
    for (Event item : events) {
      operation = ContentProviderOperation.newDelete(DATAPOINTS_CONTENT_URI)
          .withSelection(DatapointEntity._ID + " = ?", new String[] { String.valueOf(item._id) })
          .build();
      operations.add(operation);
    }

    try {
      context.getContentResolver().applyBatch(AUTHORITY, operations);
    } catch (RemoteException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    } catch (OperationApplicationException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    }catch (Exception e){
      Logger.f("MoEDAO: deleteInteractionData", e);
    }
  }

  Cursor getMessages(Context context) {
    return context.getContentResolver()
        .query(MESSAGES_CONTENT_URI, MessageEntity.PROJECTION, null, null,
            MessageEntity.DEFAULT_SORT_ORDER);
  }

  boolean setMessageClicked(final long id) {
    Uri updateRec = MESSAGES_CONTENT_URI.buildUpon().appendPath(String.valueOf(id)).build();
    ContentValues values = new ContentValues();
    values.put(MessageEntity.MSG_CLICKED, 1);
    int rowCount = mContext.getContentResolver().update(updateRec, values, null, null);
    mContext.getContentResolver().notifyChange(updateRec, null);
    return rowCount > 0;
  }

  void removeExpiredData() {
    String currTime = Long.toString(System.currentTimeMillis());
    int rows = mContext.getContentResolver()
        .delete(INAPP_CONTENT_URI, InAppMessageEntity.MSG_TTL + " < ?" + " AND " +
                InAppMessageEntity.MSG_STATUS + " = ?",
            new String[] { Long.toString(System.currentTimeMillis() / 1000), "expired" });
    Logger.v("MoEDAO:removeExpiredData: Number of IN APP records deleted: " + rows);
    rows = mContext.getContentResolver()
        .delete(MESSAGES_CONTENT_URI, MessageEntity.MSG_TTL + " < ?", new String[] { currTime });
    Logger.v("MoEDAO:removeExpiredData: Number of PromotionalMessage records deleted: " + rows);
    rows = mContext.getContentResolver()
        .delete(CAMPAIGN_LIST_URI, CampaignListEntity.CAMPAIGN_ID_TTL + " < ?", new String[] { currTime });
    Logger.v("MoEDAO:removeExpiredData: Number of CampaignList records deleted: " + rows);
    mContext.getContentResolver().notifyChange(INAPP_CONTENT_URI, null);
    mContext.getContentResolver().notifyChange(MESSAGES_CONTENT_URI, null);
  }

  boolean setMessageClickedByTime(final long gtime) {
    ContentValues values = new ContentValues();
    values.put(MessageEntity.MSG_CLICKED, 1);
    int rowCount = mContext.getContentResolver()
        .update(MESSAGES_CONTENT_URI, values, MessageEntity.GTIME + " = ? ",
            new String[] { String.valueOf(gtime) });
    mContext.getContentResolver().notifyChange(MESSAGES_CONTENT_URI, null);
    return rowCount > 0;
  }

  void addOrUpdateUserAttribute(@NonNull UserAttribute userAttribute) {
    if (userAttribute == null) return;
    Logger.v("User Attribute -->"
        + userAttribute.userAttributeName
        + ":"
        + userAttribute.userAttributeValue);
    ContentValues contentValues = new ContentValues();
    contentValues.put(UserAttributeEntity.ATTRIBUTE_NAME, userAttribute.userAttributeName);
    contentValues.put(UserAttributeEntity.ATTRIBUTE_VALUE, userAttribute.userAttributeValue);
    Cursor cursor = null;
    try {
      //check if the user attribute already exists
      cursor = mContext.getContentResolver()
          .query(USER_ATTRIBUTES_URI, UserAttributeEntity.PROJECTION,
              UserAttributeEntity.ATTRIBUTE_NAME + "=?", new String[] {
                  userAttribute.userAttributeName
              }, null);
      if (cursor != null && cursor.moveToFirst()) {
        //update attribute value if exists
        updateUserAttribute(userAttribute, contentValues);
      } else {
        //add attribute value
        addUserAttribute(contentValues);
      }
    }catch (Exception e){
      Logger.e("MoEDAO: addOrUpdateUserAttribute()", e);
    }finally {
      if (cursor != null) {
        cursor.close();
      }
    }
  }

  private void addUserAttribute(ContentValues contentValues) {
    Uri newRecord = mContext.getContentResolver().insert(USER_ATTRIBUTES_URI, contentValues);
    if (null != newRecord) {
      Logger.v("New user attribute added with Uri: " + newRecord.toString());
    } else {
      Logger.v("Unable to user attribute");
    }
  }

  private void updateUserAttribute(@NonNull UserAttribute userAttribute,
      ContentValues contentValues) {
    int updateCount = mContext.getContentResolver()
        .update(USER_ATTRIBUTES_URI, contentValues, UserAttributeEntity.ATTRIBUTE_NAME + "=?",
            new String[] { userAttribute.userAttributeName });
    if (updateCount > 0) {
      Logger.v("New user attribute updated, count: " + updateCount);
    } else {
      Logger.v("Unable to user attribute");
    }
  }

  @Nullable UserAttribute getUserAttributesForKey(@NonNull String attributeName){
    if (TextUtils.isEmpty(attributeName)) return null;
    Cursor cursor = null;
    UserAttribute userAttribute = null;
    try{
      cursor = mContext.getContentResolver()
          .query(USER_ATTRIBUTES_URI, UserAttributeEntity.PROJECTION,
              UserAttributeEntity.ATTRIBUTE_NAME + "=?", new String[] {
                  attributeName
              }, null);
      if (cursor != null && cursor.moveToFirst()){
        userAttribute = new UserAttribute();
        userAttribute.userAttributeName =
            cursor.getString(UserAttributeEntity.COLUMN_INDEX_ATTRIBUTE_NAME);
        userAttribute.userAttributeValue =
            cursor.getString(UserAttributeEntity.COLUMN_INDEX_ATTRIBUTE_VALUE);
      }
    } finally{
      closeCursor(cursor);
    }
    return userAttribute;
  }

  /**
   * Gets batches of interaction data from batchdata table
   * @param batchSize max number of batches that should be returned
   * @return batched data if present, else null
   */
  @Nullable ArrayList<BatchData> getBatchedData(int batchSize) {
    Uri CONTENT_URI = BATCHED_DATA_URI.buildUpon()
        .appendQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT, String.valueOf(batchSize))
        .build();
    Cursor cursor = null;
    ArrayList<BatchData> batchList = null;
    try {
      cursor = mContext.getContentResolver()
          .query(CONTENT_URI, BatchDataEntity.PROJECTION, null, null, null);
      if (cursor == null || cursor.getCount() == 0) {
        Logger.v("Empty cursor");
        closeCursor(cursor);
        return null;
      }
      batchList = new ArrayList<>(cursor.getCount());
      if (cursor.moveToFirst()) {
        do {
          long _id = cursor.getLong(cursor.getColumnIndex(BatchDataEntity._ID));
          String data = cursor.getString(cursor.getColumnIndex(BatchDataEntity.BATCHED_DATA));
          batchList.add(new BatchData(_id, data));
        }while (cursor.moveToNext());
      }
    } catch (Exception e){
      Logger.f( "MoEDAO getBatchedData() :exception ", e);
    } finally{
      closeCursor(cursor);
    }
    return batchList;
  }

  /**
   * Writes batched interaction data to batchdata table
   * @param batch batched data
   */
  void writeBatch(@NonNull String batch){
    if (batch == null) return;
    ContentValues values = new ContentValues();
    values.put(BatchDataEntity.BATCHED_DATA, batch);
    Uri newBatch = mContext.getContentResolver().insert(BATCHED_DATA_URI, values);
    if (newBatch != null){
      Logger.v("MoEDAO: writeBatch() New batch added : uri " + newBatch.toString());
    }else {
      Logger.f("MoEDAO: writeBatch() unable to add batch");
    }
  }

  /**
   * Deletes passed batch from the batchdata table
   * @param batch batch to be deleted
   */
  void deleteBatch(BatchData batch) {
    ArrayList<ContentProviderOperation> operations = new ArrayList<>();
    ContentProviderOperation operation;
      operation = ContentProviderOperation.newDelete(BATCHED_DATA_URI)
          .withSelection(BatchDataEntity._ID + " = ?", new String[] { String.valueOf(batch._id) })
          .build();
      operations.add(operation);
    try {
      mContext.getContentResolver().applyBatch(AUTHORITY, operations);
    } catch (RemoteException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    } catch (OperationApplicationException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    }catch (Exception e){
      Logger.f("MoEDAO: deleteInteractionData", e);
    }
  }

  /**
   * Deletes all events from data-points table.
   */
  @WorkerThread
  void deleteAllEvents(){
    mContext.getContentResolver()
        .delete(MoEDataContract.DatapointEntity.getContentUri(mContext), null, null);
  }

  /**
   * Deletes all pending bathes from batches table.
   */
  @WorkerThread
  void deleteAllBatches(){
    mContext.getContentResolver().delete(MoEDataContract.BatchDataEntity.getContentUri(mContext),
        null, null);
  }

  private void closeCursor(Cursor cursor){
    if (cursor != null){
      cursor.close();
    }
  }
}

