package com.moengage.core;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.util.Log;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.executor.TaskProcessor;

/**
 * Sets the logging level for MoEngage SDK for DEBUG builds only.<br>
 * Log Levels :<br>
 * {@link Logger#VERBOSE} - Prints verbose, debug, warnings, errors and info logs<br>
 * {@link Logger#DEBUG} - Prints debug, warnings, errors and info logs<br>
 * {@link Logger#WARN} - Prints warning, error and info logs.<br>
 * {@link Logger#ERROR} - Prints error and info logs.<br>
 * {@link Logger#INFO} - Prints only info logs.<br>
 * {@link Logger#NO_LOGS} - No logs.<br>
 *
 * @author Umang Chamaria
 * @since 7.0.00
 */
public class Logger {

  /**
   * Priority constant for the println method; use Logger.v.
   */
  public static final int VERBOSE = 5;
  /**
   * Priority constant for the println method; use Logger.d.
   */
  public static final int DEBUG = 4;
  /**
   * Priority constant for the println method; use Logger.w.
   */
  public static final int WARN = 3;
  /**
   * Priority constant for the println method; use Logger.e.
   */
  public static final int ERROR = 2;
  /**
   * Priority constant for the println method; use Logger.i.
   */
  public static final int INFO = 1;

  /**
   * Constant for no logs
   */
  public static final int NO_LOGS = 0;

  private static int LOG_LEVEL = ERROR;

  private static Context mContext;

  //ERROR>WARN>INFO>DEBUG>V - LEVEL
  private Logger() {
  }

  /**
   * @param msg The message you would like logged.
   */
  public static void v(String msg) {
    if (isDebugEnabled() && LOG_LEVEL >= VERBOSE) Log.v(TAG, msg);
  }

  public static void v(String msg, Throwable tr) {
    if (isDebugEnabled() && LOG_LEVEL >= VERBOSE) Log.v(TAG, msg, tr);
  }

  /**
   * @param msg The message you would like logged.
   */
  public static void d(String msg) {
    if (isDebugEnabled() && LOG_LEVEL >= DEBUG) Log.d(TAG, msg);
  }

  /**
   * @param msg The message you would like logged.
   * @param tr An exception to log
   */
  public static void d(String msg, Throwable tr) {
    if (isDebugEnabled() && LOG_LEVEL >= DEBUG) Log.d(TAG, msg, tr);
  }

  /**
   * @param msg The message you would like logged.
   */
  public static void i(String msg) {
    if (isDebugEnabled() && LOG_LEVEL >= INFO) Log.i(TAG, msg);
  }

  /**
   * @param msg The message you would like logged.
   * @param tr An exception to log
   */
  public static void i(String msg, Throwable tr) {
    if (isDebugEnabled() && LOG_LEVEL >= INFO) Log.i(TAG, msg, tr);
  }

  /**
   * @param msg The message you would like logged.
   */
  public static void w(String msg) {
    if (isDebugEnabled() && LOG_LEVEL >= WARN) Log.w(TAG, msg);
  }

  /**
   * @param msg The message you would like logged.
   * @param tr An exception to log
   */
  public static void w(String msg, Throwable tr) {
    if (isDebugEnabled() && LOG_LEVEL >= WARN) Log.w(TAG, msg, tr);
  }

  /*
   * Send a {@link #WARN} log message and log the exception.
   * @param tag Used to identify the source of a log message.  It usually identifies
   *        the class or activity where the log call occurs.
   * @param tr An exception to log
   */
/*  public static int w(String tag, Throwable tr) {
    return println(LOG_ID_MAIN, WARN, tag, getStackTraceString(tr));
  }*/

  /**
   * @param msg The message you would like logged.
   */
  public static void e(String msg) {
    if (isDebugEnabled() && LOG_LEVEL >= ERROR) Log.e(TAG, msg);
  }

  /**
   * @param msg The message you would like logged.
   * @param tr An exception to log
   */
  public static void e(String msg, Throwable tr) {
    if (isDebugEnabled() && LOG_LEVEL >= ERROR) Log.e(TAG, msg, tr);
  }

  public static void f(String msg) {
    if (mContext != null) {
      TaskProcessor.getInstance().addTask(new SendLogEntriesTask(mContext, msg, null, "error"));
    }
    if (isDebugEnabled() && LOG_LEVEL >= ERROR) Log.e(TAG, msg);
  }

  public static void f(String msg, Throwable tr) {
    if (mContext != null) {
      TaskProcessor.getInstance().addTask(new SendLogEntriesTask(mContext, msg, tr, "error"));
    }
    if (isDebugEnabled() && LOG_LEVEL >= ERROR) Log.e(TAG, msg, tr);
  }

  public static void setLogLevel(int logLevel) {
    LOG_LEVEL = logLevel;
  }

  private static final String TAG;

  static {
    TAG = "MoEngage_v" + MoEHelperConstants.LIB_VERSION;
  }

  /**
   * Enables DEBUG logs. Should be set to true only in development and
   * testing. Should be set to false when in going live
   */
  private static boolean MOE_DEBUG_ENABLED = false;

  public static boolean isDebugEnabled() {
    return MOE_DEBUG_ENABLED;
  }

  public static void enableDebugLog(Context context) {
    if (null == context) return;
    try {
      boolean debugging =
          (0 != (context.getApplicationInfo().flags & ApplicationInfo.FLAG_DEBUGGABLE));
      setLogStatus(debugging || ConfigurationProvider.getInstance(context).isDebugLogEnabled());
      mContext = context;
    } catch (Exception e) {
      Logger.e("Logger : enableDebugLog", e);
    }
  }

  /**
   * Enables/Disables MoEngage logs.<br><b>Note : This API should be used only if logs are required
   * in
   * production/signed builds.</b>
   *
   * @param value true to enable logs, else false
   */
  public static void setLogStatus(boolean value) {
    MOE_DEBUG_ENABLED = value;
  }
}
