/* ************************************************************************
 *
 * MOENGAGE CONFIDENTIAL
 * __________________
 *
 *  [2014] - [2015] MoEngage Inc.
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.push;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.DeviceAddTask;
import com.moengage.core.Logger;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEngage;
import com.moengage.core.MoEngage.Builder;
import java.util.Map;

/**
 * This is a Manager class which has callbacks for handling GCM payload handling and token refresh
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3
 */
public class PushManager {

  public static final String REQ_REGISTRATION = "MOE_REG_REQ";
  public static final String REQ_DELETE_TOKEN = "MOE_DEL_TOK";
  public static final String REQ_REFRESH = "MOE_REG_REFRESH";
  private final String ATTR_PUSH_TOKEN = "push_token";
  private final String ATTR_REGISTRATION_BY = "registered_by";
  private final String TOKEN_EVENT = "TOKEN_EVENT";
  public static final String TOKEN_BY_MOE = "MoE";
  public static final String REG_ON_APP_OPEN = "REG_ON_APP_OPEN";
  private boolean isBaiduEnabled = false;
  public static final String SHOW_NOTIFICATION = "SHOW_NOTIFICATION";

  private PushManager() {
    //check for PushHandler on class path
    loadPushHandler();
  }

  private final Object lock = new Object();

  private void loadPushHandler() {
    try {
      //TODO think of adding play services check here
      if (!isBaiduEnabled) {
        try {
          Class handler = Class.forName("com.moengage.firebase.PushHandlerImpl");
          pushHandler = (PushHandler) handler.newInstance();
          Logger.v("PushManager:loadPushHandler FCM Enabled");
        } catch (Exception e) {
          Class handler = Class.forName("com.moengage.push.gcm.PushHandlerImpl");
          pushHandler = (PushHandler) handler.newInstance();
          Logger.v("PushManager:loadPushHandler GCM Enabled");
        }
      } else {
        Class handler = Class.forName("com.moengage.baidu.PushHandlerImpl");
        pushHandler = (PushHandler) handler.newInstance();
        Logger.v("PushManager:loadPushHandler Baidu Enabled");
      }
      //TODO ADD ADM
    } catch (Exception e) {
      Logger.e("PushManager : loadPushHandler : did not find supported module: " + e.getMessage());
    }
  }

  private static PushManager _INSTANCE = null;

  public static PushManager getInstance() {
    if (null == _INSTANCE) {
      _INSTANCE = new PushManager();
    }
    return _INSTANCE;
  }

  public void setMessageListener(Object messageListener) {
    if (pushHandler != null) {
      pushHandler.setMessageListener(messageListener);
    }
  }

  public void refreshToken(Context context, String token) {
    refreshTokenInternal(context, token, "App");
  }

  public void refreshTokenInternal(Context context, String token, String pushRegisteredBy) {
    if (TextUtils.isEmpty(token)) return;
    Logger.v("PushManager:refreshToken");
    synchronized (lock) {
      Logger.v("PushManager:refreshToken before ripping: = " + token);
      token = ripMultiplexingExtras(token);
      if (tokenListener != null) tokenListener.onTokenReceived(token);
      String oldRegId = ConfigurationProvider.getInstance(context).getGCMToken();
      boolean update = tokenRefreshRequired(context, token);
      if (update || !ConfigurationProvider.getInstance(context).isDeviceRegistered()) {
        ConfigurationProvider.getInstance(context).setGCMToken(token);
        MoEDispatcher.getInstance(context).addTaskToQueue(new DeviceAddTask(context));
        PayloadBuilder payloadBuilder = new PayloadBuilder();
        payloadBuilder.putAttrString(ATTR_PUSH_TOKEN, token);
        payloadBuilder.putAttrString(ATTR_REGISTRATION_BY, pushRegisteredBy);
        MoEHelper.getInstance(context).trackEvent(TOKEN_EVENT, payloadBuilder.build());
      }
      Logger.v("PushManager:refreshToken oldId: = "
          + oldRegId
          + " token = "
          + token
          + " --updating[true/false]: "
          + update);
    }
  }

  private static final String ID_PREFIX = "|ID|";

  private String ripMultiplexingExtras(String token) {
    return !TextUtils.isEmpty(token) && token.startsWith(ID_PREFIX) ? token.substring(7) : token;
  }

  public boolean tokenRefreshRequired(Context context, String newToken) {
    if (!TextUtils.isEmpty(newToken)) {
      String oldToken = ConfigurationProvider.getInstance(context).getGCMToken();
      return TextUtils.isEmpty(oldToken) || !newToken.equals(oldToken);
    } else {
      return false;
    }
  }

  public interface PushHandler {

    /**
     * (non-JavaDoc)
     *
     * @param context Instance of the application {@link Context}
     * @return GCM token
     * <b>Note : This method should not be called from the main thread</b>
     */
    @WorkerThread String registerForPushToken(Context context);

    /**
     * (non-JavaDoc)
     *
     * @param context Instance of the application {@link Context}
     * @param extras {@link Bundle} extras which is passed along with the GCM payload
     */
    void handlePushPayload(Context context, Bundle extras);

    /**
     * (non-JavaDoc)
     * No longer supported by the SDK need to remove
     * @param context Instance of the application {@link Context}
     * @param intent {@link Intent} extras which is passed along with the GCM payload
     */
    @Deprecated
    void handlePushPayload(Context context, Intent intent);

    /**
     * (non-JavaDoc)
     * Deletes the gcm token.<br> <b>Note : This method should not be called from the main thread.
     * <br>Should only be called whenever the user is logged out</b>
     *
     * @param context Instance of the application {@link Context}
     * @param senderId SenderId or ProjectId from Developer's Console
     */
    @WorkerThread void deleteToken(Context context, String senderId);

    /**
     * (non-JavaDoc)
     * Starts intent service to offload task from main thread
     *
     * @param context Application Context
     * @param extra parameter which decides the task to be performed
     */
    void offLoadToWorker(Context context, String extra);

    /**
     * (non-JavaDoc)
     * Log Notification click event. used in cases where people handle everything on there own
     *
     * @param context Application Context
     * @param intent Calling intent
     */
    void logNotificationClicked(Context context, Intent intent);

    /**
     * (non-JavaDoc)
     * Set Customized Push Message Listener
     *
     * @param messageListener Object of a class extending PushMessageListener
     */
    void setMessageListener(Object messageListener);

    /**
     * Set an alarm to register for push if no token is found on app open
     *
     * @param context Application context
     */
    void setPushRegistrationFallback(Context context);

    /**
     * Returns an instance of PushMessageListener
     *
     * @return instance of PushMessageListener, either default or custom
     */
    Object getMessageListener();

    /**
     * @param context Instance of the application context {@link Context}
     * @param pushPayload {@link Map} map which is passed along with the FCM payload
     */
    void handlePushPayload(Context context, Map<String, String> pushPayload);

    @Nullable @WorkerThread String getPushToken(Context context);

    /**
     *
     * @param context Instance of the application context {@link Context}
     * @param pushPayload String which is passed along with the Baidu payload
     */
    void handlePushPayload(Context context, String pushPayload);
  }

  public PushHandler getPushHandler() {
    return pushHandler;
  }

  private PushHandler pushHandler;
  private boolean backStackBuilderOptoutFlag = false;

  /**
   * This API is deprecated use {@link Builder#optOutBackStackBuilder()}
   */
  @Deprecated
  public final void optoutBackStackBuilder(Boolean value) {
    backStackBuilderOptoutFlag = value;
  }

  /*
   * Method to get status whether Client has opted out from BackStack creation or not
   *
   * @return value of backStackBuilderOptoutFlag
   */
  public final boolean isBackStackBuilderOptedOut(Context context) {
    return backStackBuilderOptoutFlag;
  }

  /**
   * This API is deprecated use {@link MoEngage.Builder#enableBaiduPush(String)}
   */
  @Deprecated public void enableBaiduPush() {
    isBaiduEnabled = true;
    loadPushHandler();
  }

  public boolean isIsBaiduEnabled() {
    return isBaiduEnabled;
  }

  /**
   * Listener to receive a callback on Push token change
   */
  public interface OnTokenReceivedListener {
    /**
     * Call back method when token is received
     *
     * @param token GCM token
     */
    void onTokenReceived(String token);
  }

  private OnTokenReceivedListener tokenListener;

  public void setTokenObserver(OnTokenReceivedListener tokenObserver) {
    tokenListener = tokenObserver;
  }

  private boolean optOutOfMoEngageExtras = false;

  /**
   * This API is deprecated use {@link Builder#optOutMoEngageExtras()}
   */
  @Deprecated
  public final void optOutMoEngageExtras(boolean value){
    optOutOfMoEngageExtras = value;
  }

  public final boolean isMoEngageExtrasOptedOut(){
    return optOutOfMoEngageExtras;
  }
}
