/* ************************************************************************
 * 
 * MOENGAGE CONFIDENTIAL
 * __________________
 * 
 *  [2014] - [2015] MoEngage Inc. 
 *  All Rights Reserved.
 * 
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moe.pushlibrary;

import android.app.Activity;
import android.app.Application;
import android.app.Fragment;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.location.Location;
import android.os.Build;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEDTManager;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEEventManager;
import com.moengage.inapp.InAppController;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.json.JSONObject;

/**
 * This is the helper class for MoEngage SDK which exposes public APIs for
 * integrating MoEngage SDK.
 *
 * Please follow the guidelines mentioned at<a
 * href="http://docs.moengage.com/en/latest/android.html"
 * >http://docs.moengage.com/en/latest/android.html</a> for integrating the
 * MoEngage SDK
 *
 * How to integrate with the MoEHelper ? The SDK requires the following changes
 * in the manifest file (Some of these are optional):
 *
 * <pre>
 * 	<code>
 *  &lt;permission android:name="{PACKAGE_NAME}.permission.C2D_MESSAGE"
 * android:protectionLevel="signature" /&gt;
 *  &lt;uses-permission android:name="{PACKAGE_NAME}.permission.C2D_MESSAGE" /&gt;
 *  &lt;uses-permission android:name="com.google.android.c2dm.permission.RECEIVE" /&gt;
 *  &lt;uses-permission android:name="android.permission.WAKE_LOCK"/&gt;
 *  &lt;application &gt;
 *  <b>&lt;!-- SDK configuration - MANDATORY FIELD--&gt;</b>
 *  &lt;!-- APP ID as mentioned in the MoEngage App Settings page --&gt;
 *  &lt;meta-data android:name="APP_ID" android:value="MAZW1111MMSXXX67C3T9KXXX" /&gt;
 *  &lt;!-- The12 digit sender ID as seen on the google cloud project console --&gt;
 *  &lt;meta-data android:name="SENDER_ID" android:value="123456789012" /&gt;
 *  &lt;!-- Optional to be used only by those who want to set the GCM registration token from the
 * app --&gt;
 *  &lt;meta-data android:name="SKIP_GCM_REGISTRATION" android:value="false" /&gt;
 *  &lt;!-- Optional to pass the GCM payload to the specified service after tracking an impression
 * --&gt;
 *  &lt;meta-data android:name="PASS_GCM_INTENT" android:value="com.example.YourService" /&gt;
 *  &lt;!-- The icon which will be used for posting notification --&gt;
 *  &lt;meta-data android:name="NOTIFICATION_ICON" android:value="R.drawable.ic_launcher" /&gt;
 *  &lt;!-- Optional The type of notification which needs to be posted --&gt;
 *  &lt;meta-data android:name="NOTIFICATION_TYPE" android:value="@integer/notification_type_single"
 * /&gt;
 *  &lt;!-- Optional The notification tone which will be used. Has to be present in the app raw
 * folder --&gt;
 *  &lt;meta-data android:name="NOTIFICATION_TONE" android:value="railtone" /&gt;
 *  &lt;!-- Receivers required--&gt;
 *  &lt;receiver android:name="com.moe.pushlibrary.InstallReceiver" /&gt;
 *  &lt;receiver android:name="com.moe.pushlibrary.PushGcmBroadcastReceiver"
 *         android:permission="com.google.android.c2dm.permission.SEND" &gt;
 *         &lt;intent-filter&gt;
 *             &lt;action android:name="com.google.android.c2dm.intent.RECEIVE" /&gt;
 *             &lt;action android:name="com.google.android.c2dm.intent.REGISTRATION" /&gt;
 *             &lt;category android:name="{PACKAGE_NAME}" /&gt;
 *         &lt;/intent-filter&gt;
 *  &lt;/receiver&gt;
 *
 *  &lt;receiver android:name="com.moe.pushlibrary.AppUpdateReceiver"&gt;
 *  &lt;intent-filter&gt;
 *         &lt;action android:name="android.intent.action.PACKAGE_REPLACED" /&gt;
 *         &lt;data android:path="{PACKAGE_NAME}"
 *             android:scheme="package" /&gt;
 *     &lt;/intent-filter&gt;
 *  &lt;/receiver&gt;
 *  &lt;!-- Services required--&gt;
 *  &lt;service android:name="com.moe.pushlibrary.MoEWorker" /&gt;
 *  &lt;service android:name="com.moe.pushlibrary.internal.MoEService" /&gt;
 *  <b>&lt;!-- This is optional and is required only when GEO Fencing is involved --&gt; </b>
 *  &lt;uses-permission android:name="android.permission.ACCESS_FINE_LOCATION"/&gt;
 *  &lt;service android:name="com.moe.pushlibrary.GeofenceIntentService" /&gt;
 *  </code>
 * </pre>
 *
 * The all activities which you want to track, show in app messages or launch
 * from notifications or show coupons dialog should have the following code.
 *
 * <pre>
 * <code>
 * 	public class YourActivity extends Activity{
 *
 * 		protected MoEHelper mHelper = null;//Declare a member variable
 *
 * 		&#64;Override
 * 		protected void onCreate(Bundle savedInstanceState) {
 * 			super.onCreate(savedInstanceState);
 * 			<b>mHelper = new MoEHelper(this);</b>//instantiate in onCreate
 *    }
 *
 * 		&#64;Override
 * 		protected void onResume() {
 * 			super.onResume();
 * 			<b>mHelper.onResume(this);</b>//onResume callback
 *    }
 *
 * 		&#64;Override
 * 		protected void onStop() {
 * 			super.onStop();
 * 			<b>mHelper.onStop(this);</b>//onStop callback
 *    }
 *
 * 		&#64;Override
 * 		protected void onPause() {
 * 			super.onPause();
 * 			<b>mHelper.onPause(this);</b>//onPause callback
 *    }
 *
 * 		&#64;Override
 * 		protected void onSaveInstanceState(Bundle outState) {
 * 			super.onSaveInstanceState(outState);
 * 			<b>mHelper.onSaveInstanceState(outState);</b>//onSaveInstance callback
 *    }
 *
 * 		&#64;Override
 * 		protected void onRestoreInstanceState(Bundle savedInstanceState) {
 * 			super.onRestoreInstanceState(savedInstanceState);
 * 			<b>mHelper.onRestoreInstanceState(savedInstanceState);</b>//onRestoreInstance callback
 *    }
 *  }
 * </code>
 * </pre>
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 5.0
 * @since 1.0
 */
public class MoEHelper {

  /**
   * The entire task is delegated to the controller
   */
  private MoEDispatcher mDispatcher = null;
  /**
   * Bundle extra which denotes that the activity was recreated for
   * configuration change and this should be accounted for when tracking
   */
  private String BUNDLE_EXTRA_RESTORING = "EXTRA_RESTORING";
  /**
   * Denotes that the activity is being restored from a previous state and it
   * should be tracked again for start
   */
  private boolean isActivityBeingRestored = false;
  /**
   * An instance of the application {@link Context}
   */
  private Context mContext;
  /**
   * Maintains the visible activity count
   */
  private static int activityCounter = 0;

  private static String TAG = "MoEHelper";

  public static int getActivityCounter(){
    return activityCounter;
  }

  /**
   * Increment the activity counter.
   */
  private static synchronized void incrementCounter() {
    activityCounter++;
  }

  /**
   * Decrement the activity counter.
   */
  private static synchronized void decrementCounter() {
    activityCounter--;
  }

  /**
   * Checks whether the app is in foreground or not.
   * If any screen is not being tracked then,
   * on that screen it would not return the correct value
   *
   * @return true if the app is in foreground else returns false
   */
  public static boolean isAppInForeground() {
    return activityCounter > 0;
  }

  /**
   * Instantiates an instance of {@link MoEHelper}.
   * Use {@link MoEHelper#getInstance(Context)}
   * @param con Reference to the Application {@link Context}
   */
  @Deprecated
  public MoEHelper(Context con) {
    Logger.enableDebugLog(con);
    mContext = con.getApplicationContext();
    if (null == mDispatcher) {
      mDispatcher = getDelegate();
    }
    _INSTANCE = this;
  }

  private boolean mStarted = false;
  private boolean mResumed = false;

  /*
   * This is for backward porting the exposed API to reduce the number of
   * object created.
   */
  private static MoEHelper _INSTANCE = null;

  /**
   * Returns a new Instance of {@link MoEHelper} for every call.
   *
   * @param con Application Context Reference. Should be a not NULL value
   * @return a new instance of MoEHelper
   */
  public static synchronized MoEHelper getInstance(Context con) {
    if (null == _INSTANCE) {
      _INSTANCE = new MoEHelper(con);
    }
    return _INSTANCE;
  }

  /**
   * Initializes the MoEngage SDK with the provided values.Follow Integration
   * steps mentioned at <a href=
   * "http://docs.moengage.com/en/latest/android.html#initializing-the-sdk-and-push-notifications"
   * > http://docs.moengage.com/en/latest/android.html#initializing-the-sdk-
   * and-push-notifications</a>
   *
   * @param sender_id The GCM Sender ID is the the twelve digit sender ID of your
   * Google API project
   * @param app_unique_id an application specific id which can be found in the &quot;App
   * Settings&quot; page of your MoEngage account.
   */
  public void initialize(String sender_id, String app_unique_id) {
    mDispatcher.initialize(sender_id, app_unique_id);
  }

	/* ********************** Activity call backs ************************** */

  /**
   * Integration point for on start of the {@link Activity} to which
   * {@link MoEHelper} is attached to. Should be called from
   * {@link Activity#onNewIntent}
   *
   * @param activity The current {@link Activity} to which this instance is
   * attached to
   * @param intent The new intent delivered to this activity
   */
  public void onNewIntent(Activity activity, Intent intent) {
    if(!isActivityBeingRestored){
      mDispatcher.onStart(activity, intent);
    }
    InAppController.InAppHandler inAppHandler = InAppController.getInstance().getInAppHandler();
    if (inAppHandler != null){
      inAppHandler.registerInAppManager(activity);
    }
  }

  /**
   * Integration point for on start of the {@link Activity} to which
   * {@link MoEHelper} is attached to. Should be called from
   * {@link Activity#onStart}
   *
   * @param activity The current {@link Activity} to which this instance is
   * attached to
   */
  public void onStart(@NonNull Activity activity) {
    if(isAutoIntegration)return;
    onStartInternal(activity);
  }

  void onStartInternal(@NonNull Activity activity){
    if (getActivityCounter() == 0){
      mDispatcher.onAppOpen();
    }
    incrementCounter();
    mStarted = true;
    mContext = activity.getApplicationContext();
    onNewIntent(activity, null);
  }

  /**
   * Integration point for on stop of the {@link Activity} to which
   * {@link MoEHelper} is attached to. Should be called from
   * {@link Activity#onStop}
   *
   * @param activity The current {@link Activity} to which this instance is
   * attached to
   */
  public void onStop(@NonNull Activity activity){
    if(isAutoIntegration)return;
    onStopInternal(activity);
  }

  void onStopInternal(@NonNull Activity activity) {
    Logger.v("Activity onStop called for " + activity.toString());
    boolean configChange = MoEHelperUtils.isChangingConfiguration(activity);
    decrementCounter();
    InAppController.InAppHandler inAppHandler = InAppController.getInstance().getInAppHandler();
    if (inAppHandler != null) {
      inAppHandler.unregisterInAppManager(activity);
    }
    mDispatcher.onStop(activity, configChange);
    String activityName = activity.getClass().getName();
    if (!mStarted) {
      Logger.e("MoEHelper: onStart callback not called: " + activityName);
    }
    if (!mResumed) {
      Logger.e("MoEHelper: onResume callback not called: " + activityName);
    }

  }

  /**
   * Integration point for on resume of the activity to which
   * {@link MoEHelper} is attached to. Should be called from
   * {@link Activity#onResume}
   *
   * @param activity The current {@link Activity} to which this instance is
   * attached to
   */
  public void onResume(@NonNull Activity activity) {
    if(isAutoIntegration)return;
    onResumeInternal(activity);
  }

  void onResumeInternal(Activity activity){
    if (null == mContext) {
      mContext = activity.getApplicationContext();
    }
    Logger.v("Activity onResume called for " + activity.toString());
    mResumed = true;
    mDispatcher.onResume(activity, isActivityBeingRestored);
    isActivityBeingRestored = false;
  }

	/* ********************** Fragment call backs ************************** */

  /**
   * Fragment callback method for {@link Fragment#onStart()}
   *
   * @param activity The activity instance in which this {@link Fragment} is
   * visible
   * @param fragmentName The name of the fragment which is currently being started
   */
  public void onFragmentStart(Activity activity, String fragmentName) {
    mDispatcher.onFragmentStart(activity, fragmentName);
  }

  /**
   * Fragment callback method for {@link Fragment#onStop()}
   *
   * @param activity The activity instance in which this {@link Fragment} is
   * visible
   * @param fragmentName The name of the fragment which is currently being stopped
   */
  public void onFragmentStop(Activity activity, String fragmentName) {
  }

	/* ********************** Handling config change ************************ */

  /**
   * Called to retrieve per-instance state from an activity before being
   * killed so that the activity is not tracked twice in MoEngage Platform.
   * Also helps to show the In App dialog even after the Configuration change
   *
   * @param outState Bundle in which to place your saved state. Same as your
   * {@link Activity#onSaveInstanceState}
   */
  public void onSaveInstanceState(Bundle outState) {
      Logger.v("MoEHelper:onSaveInstanceState-- saving state");
      if (null == outState) return;
      outState.putBoolean(BUNDLE_EXTRA_RESTORING, true);
   }

  /**
   * Called to restore the per instance state of the Activity so that it is
   * not tracked twice and also to restore the In App PromotionalMessage if it was being
   * shown before the configuration change occurred
   *
   * @param savedInstanceState the data most recently supplied in
   * {@link #onSaveInstanceState}.
   */
  public void onRestoreInstanceState(Bundle savedInstanceState) {
    Logger.v("MoEHelper:onRestoreInstanceState-- restoring state");
    if (null == savedInstanceState) return;
    if (savedInstanceState.containsKey(BUNDLE_EXTRA_RESTORING)) {
      isActivityBeingRestored = true;
      savedInstanceState.remove(BUNDLE_EXTRA_RESTORING);
    }
  }

  /**
   * Tracks the specified event
   *
   * @param action The action associated with the Event
   * @param attrs The attributes of associated with the Event
   * @return {@link MoEHelper} instance
   */
  public MoEHelper trackEvent(@NonNull final String action, JSONObject attrs) {
    if (TextUtils.isEmpty(action)) return this;
    MoEEventManager.getInstance(mContext).trackEvent(action.trim(), attrs);
    return this;
  }

  /**
   * Tracks the specified event
   *
   * @param action The action associated with the Event
   * @param attrs A {@link HashMap} of all the attributes which needs to be
   * logged.
   * @return {@link MoEHelper} instance
   */
  public MoEHelper trackEvent(@NonNull final String action, @NonNull HashMap<String, String> attrs) {
    if(TextUtils.isEmpty(action))return this;
    Set<String> keySet = attrs.keySet();
    if (keySet.isEmpty()) {
      return this;
    }
    JSONObject newAttrs = new JSONObject();
    for (String key : keySet) {
      try {
        newAttrs.put(key, attrs.get(key));
      } catch (Exception e) {
        Logger.f("MoEHelper:trackEvent", e);
      }
    }
    trackEvent(action.trim(), newAttrs);
    return this;
  }

  /**
   * Tracks the specified event
   *
   * @param action The action associated with the Event
   * @param attrs A {@link Map} of all the attributes which needs to be
   * logged.
   * @return {@link MoEHelper} instance
   */
  public MoEHelper trackEvent(@NonNull final String action, @NonNull Map<String, String> attrs) {
    Set<String> keySet = attrs.keySet();
    if (keySet.isEmpty()) {
      return this;
    }
    JSONObject newAttrs = new JSONObject();
    for (String key : keySet) {
      try {
        newAttrs.put(key, attrs.get(key));
      } catch (Exception e) {
        Logger.f("MoEHelper:trackEvent", e);
      }
    }
    trackEvent(action, newAttrs);
    return this;
  }

  /**
   * Tracks the specified event.
   *
   * @param action The action associated with the Event
   * @return {@link MoEHelper} instance
   */
  public MoEHelper trackEvent(@NonNull String action){
    if (TextUtils.isEmpty(action)) return this;
    MoEEventManager.getInstance(mContext).trackEvent(action.trim(), new JSONObject());
    return this;
  }

  /**
   * Set an user attribute. This can be used to segment users on MoEngage
   * platform.
   *
   * @param userAttribute The user attribute name which needs to be added. This field
   * cannot be null
   * @param attributeValue The value associated with the specified user attribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, @NonNull String attributeValue) {
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    try {
      if (attributeValue == null) {
        attributeValue = "";
      } else if (MoEHelperConstants.USER_ATTRIBUTE_USER_BDAY.equals(userAttribute)) {
        attributeValue = URLEncoder.encode(attributeValue, "UTF-8");
      }
    } catch (UnsupportedEncodingException e1) {
      Logger.f("MoEHelper:setUserAttribute", e1);
    }catch (Exception e){
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      mDispatcher.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, int attributeValue) {
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      mDispatcher.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, boolean
      attributeValue) {
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      mDispatcher.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, double attributeValue) {
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      mDispatcher.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, float attributeValue) {
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      mDispatcher.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, long attributeValue) {
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      mDispatcher.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.f("MoEHelper:setUserAttribute", e);
    }
    return this;
  }

  public MoEHelper setUserAttributeEpochTime(@NonNull String userAttribute, long attributeValue){
    if (userAttribute == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    mDispatcher.setCustomUserAttribute(new PayloadBuilder().putAttrDateEpoch(userAttribute,
        attributeValue).build());
    return this;
  }

  /**
   * Set the user attributes with the attributes specified in the HashMap
   *
   * @param attributeMap The attribute map which needs to be set for the user
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull HashMap<String, Object> attributeMap) {
    if (attributeMap.isEmpty()) {
      Logger.e("MoEHelper:User attribute map cannot be null or empty");
      return this;
    }
    Set<String> keySet = attributeMap.keySet();
    if ( keySet.isEmpty() ) {
      return this;
    }
    for (String key : keySet) {
      try {
        if(TextUtils.isEmpty(key))continue;
        Object attrValue = attributeMap.get(key);
        if( attrValue instanceof Date){
          setUserAttribute(key.trim(), (Date)attrValue);
          continue;
        }else if( attrValue instanceof GeoLocation ){
          setUserAttribute(key.trim(), (GeoLocation)attrValue);
          continue;
        }else if( attrValue instanceof Location ){
          setUserAttribute(key.trim(), (Location)attrValue);
          continue;
        }
        JSONObject userJson = new JSONObject();
        userJson.put(key.trim(), attrValue);
        mDispatcher.setUserAttribute(userJson);
      } catch (Exception e) {
        Logger.f("MoEHelper:setUserAttribute", e);
      }
    }
    return this;
  }

  /**
   * Set the user attributes with the attributes specified in the HashMap
   *
   * @param attributeMap The attribute map which needs to be set for the user
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull Map<String, Object> attributeMap) {
    if (attributeMap.isEmpty()) {
      Logger.e("MoEHelper:User attribute map cannot be null or empty");
      return this;
    }
    Set<String> keySet = attributeMap.keySet();
    if ( keySet.isEmpty() ) {
      return this;
    }
    for (String key : keySet) {
      try {
        if(TextUtils.isEmpty(key))continue;
        Object attrValue = attributeMap.get(key);
        if( attrValue instanceof Date){
          setUserAttribute(key.trim(), (Date)attrValue);
          continue;
        }else if( attrValue instanceof GeoLocation ){
          setUserAttribute(key.trim(), (GeoLocation)attrValue);
          continue;
        }else if( attrValue instanceof Location ){
          setUserAttribute(key.trim(), (Location)attrValue);
          continue;
        }
        JSONObject userJson = new JSONObject();
        userJson.put(key.trim(), attrValue);
        mDispatcher.setUserAttribute(userJson);
      } catch (Exception e) {
        Logger.f("MoEHelper:setUserAttribute", e);
      }
    }
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull Date value){
    if (name == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    mDispatcher.setCustomUserAttribute(new PayloadBuilder().putAttrDate(name, value).build());
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull String date, String dateFormat){
    if (name == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    mDispatcher.setCustomUserAttribute(new PayloadBuilder().putAttrDate(name, date, dateFormat).build());
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull Location location){
    if (name == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    mDispatcher.setCustomUserAttribute(new PayloadBuilder().putAttrLocation(name, location).build());
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull GeoLocation location){
    if (name == null) {
      Logger.e("MoEHelper:User attribute value cannot be null");
      return this;
    }
    mDispatcher.setCustomUserAttribute(new PayloadBuilder().putAttrLocation(name, location).build());
    return this;
  }

  /**
   * To set user location, call this method
   *
   * @param lat The latitude of the user location
   * @param lng The longitude of the user location
   */
  public void setUserLocation(double lat, double lng) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LOCATION, new GeoLocation(lat, lng));
  }

  /**
   * Returns the Inbox unread message count Don't call on main thread
   *
   * @return The count of unread messages
   */
  @WorkerThread
  public int getUnreadMessagesCount() {
    return mDispatcher.getUnreadMessageCount();
  }

  /**
   * <b>Advanced API not required for general usage.</b> Forces the SDK to
   * fetch in app messages from the server. This method does not need to be
   * called explicitly because the SDK periodically synchronizes data with the
   * server.
   * @param force force to show inapps
   */
  public void showInAppIfAny(boolean force) {
    mDispatcher.checkForInAppMessages(force);
  }

  /**
   * Tells the SDK whether this is a migration or a fresh installation.
   * <b>Not calling this method will STOP execution of INSTALL CAMPAIGNS</b>.
   * This is solely required for migration to MoEngage Platform
   *
   * @param existingUser true if it is an existing user else set false
   */
  public void setExistingUser(boolean existingUser) {
    mDispatcher.setExistingUser(existingUser, mContext);
  }
	/* ***** Client call backs ***** */

	/* ************************************************************************* */

  /**
   * Retrieves all Inbox messages from the DB<br>
   * <b>Note : Don't call on UI Thread</b>.
   * @param context An instance of the application {@link Context}
   * @return A {@link Cursor} with all the existing messages
   */
  @WorkerThread
  public static Cursor getAllMessages(Context context) {
    return MoEDispatcher.getInstance(context).getAllMessages();
  }

  /**
   * Logs that the message with the provided id was clicked
   *
   * @param context An instance of the application {@link Context}
   * @param id The id associated with the message which was clicked
   */
  @WorkerThread
  public static void setMessageClicked(Context context, long id) {
    MoEDispatcher.getInstance(context).setInboxMessageClicked(id);
  }

  /**
   * Invalidates the existing sessions and user attributes and treats all
   * actions performed by the user as a new user after this method is called<br>
   * If the Application is doing a self registration for gcm token then invalidate the token on
   * logout
   */
  public void logoutUser() {
    if (null == mContext) return;
    mDispatcher.logoutUser(false);
  }

  /**
   * Will request a data sync with MoEngage Platform.
   */
  public void syncInteractionDataNow() {
    mDispatcher.sendInteractionData();
  }

  /**
   * Set the notification time preference which will be kept into account
   * when notifications are being sent from the MoEngage Platform
   *
   * @param dontSendStart The Do Not Disturb Start Time
   * @param dontSendEnd The Do Not Disturb End Time
   * @param format The time format being used,
   * use any of these - hh:mm/hh:mm:ss
   * @param timezone The timezone in which the user is present.
   * The system expects IST, PST values
   */
  public void setNotificationPreference(String dontSendStart, String dontSendEnd, String timezone,
      String format) {
    try {
      JSONObject obj = new JSONObject();
      obj.put(MoEHelperConstants.USER_ATTRIBUTE_NOTIFICATION_START, dontSendStart);
      obj.put(MoEHelperConstants.USER_ATTRIBUTE_NOTIFICATION_END, dontSendEnd);
      obj.put(MoEHelperConstants.USER_ATTRIBUTE_ASSOCIATED_TIMEZONE, timezone);
      obj.put(MoEHelperConstants.USER_ATTRIBUTE_ASSOCIATED_TIME_FORMAT, format);
      JSONObject userAttrObj = new JSONObject();
      userAttrObj.put(MoEHelperConstants.USER_ATTRIBUTE_NOTIFICATION_PREF, obj.toString());
      mDispatcher.setUserAttribute(userAttrObj);
    } catch (Exception e) {
      Logger.f("MoEHelper: setNotificationPreference", e);
    }
  }


  /**
   * @param context Application context
   * @param optOut true if the user is opting out of Google Advertisement ID collection, false
   * otherwise
   */
  public void optOutOfAdIdCollection(Context context, boolean optOut) {
    ConfigurationProvider.getInstance(context).optOutOfAdIdCollection(optOut);
  }

  /**
   * Enables or disables location tracking
   * <b>Should be set in the Application subclass</b>
   * @param optOut true if the user wants to opt out of location tracking
   */
  public void optOutOfLocationTracking(boolean optOut){
    ConfigurationProvider.getInstance(mContext).optOutOfTrackLocation(optOut);
  }

  /**
   * Enables or disables setting of geo-fences
   * <b>Should be set in the Application subclass</b>
   * @param optOut true if the user wants to opt out of geo fencing
   */
  public void optOutOfGeoFences(boolean optOut){
    ConfigurationProvider.getInstance(mContext).optOutOfSetGeoFence(optOut);
  }

  public void trackNotificationClickedByTime(long gtime){
    mDispatcher.trackNotificationClicked(gtime);
  }

  public void playNotificationSound(boolean state){
    ConfigurationProvider.getInstance(mContext).saveNotificationSoundState(state);
  }

  /**
   * Method to auto-integrate with MoEngage without adding lifecycle callbacks.
   * <b>Note : Only for applications having minimum sdk version greater than or equal to 14</b>
   * @param application reference to {@link Application}
   */
  public void autoIntegrate(Application application){
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH){
      Logger.d("MoEHelper: Auto integration is enabled");
      mLifeCycleCallBacks = new MoEActivityLifeCycleCallBacks();
      application.registerActivityLifecycleCallbacks(mLifeCycleCallBacks);
      isAutoIntegration = true;
    }
  }

  private static boolean isAutoIntegration = false;

  private MoEActivityLifeCycleCallBacks mLifeCycleCallBacks;

  /**
   * Method to unregister lifecycle callbacks
   * @param application reference to {@link Application}
   */
  public void unregisterLifecycleCallbacks(@NonNull Application application){
    if (VERSION.SDK_INT >= VERSION_CODES.ICE_CREAM_SANDWICH){
      Logger.v( "MoEHelper unregisterLifecycleCallbacks() : will try to unregister lifecycle "
          + "callbacks");
      if (mLifeCycleCallBacks != null && application != null){
        Logger.v(TAG + " unregisterLifecycleCallbacks() : un-registering for lifecycle callbacks");
        application.unregisterActivityLifecycleCallbacks(mLifeCycleCallBacks);
        isAutoIntegration = false;
      }else {
        Logger.v(TAG + " unregisterLifecycleCallbacks() : either lifecycle callbacks or "
            + "application class instance is null cannot unregister.");
      }
    }
  }


  public MoEDispatcher getDelegate(){
    if( null == mDispatcher){
      mDispatcher = MoEDispatcher.getInstance(mContext);
    }
    return mDispatcher;
  }

  /**
   * Helper method to set user email
   *
   * @param value email of User passed by user
   */
  public void setEmail(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_EMAIL, value);
  }

  /**
   * Helper method to set user full name
   *
   * @param value Full Name of user passed by user
   */
  public void setFullName(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_NAME, value);
  }

  /**
   * Helper method to set user gender
   *
   * @param value Gender value passed by user
   */
  public void setGender(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_GENDER, value);
  }

  /**
   * Helper method to set User first Name
   *
   * @param value First Name value passed by user
   */
  public void setFirstName(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_FIRST_NAME, value);
  }

  /**
   * Helper method to set User Last Name
   *
   * @param value Last Name value passed by user
   */
  public void setLastName(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LAST_NAME, value);
  }

  /**
   * Helper method to set User BirthDate
   * Use {@link MoEHelper#setBirthDate(Date)}
   * @param value BirthDate value passed by user
   */
  @Deprecated
  public void setBirthDate(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_BDAY, value);
  }

  /**
   * Helper method to set User BirthDate<br>
   *
   * @param birthDate BirthDate value passed by user
   */
  public void setBirthDate(@NonNull Date birthDate){
    mDispatcher.setCustomUserAttribute(new PayloadBuilder().putAttrDate(MoEHelperConstants
        .USER_ATTRIBUTE_USER_BDAY, birthDate).build());
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(int value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(long value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(float value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(double value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Number
   *
   * @param value Number passed by user
   */
  public void setNumber(String value) {
    if (!TextUtils.isEmpty(value)) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_MOBILE, value);
    }
  }

  /**
   * Set log level for MoEngage SDK's logs
   * @param logLevel : log Level {@link Logger}
   */
  public void setLogLevel(int logLevel){
    Logger.setLogLevel(logLevel);
  }

  /**
   *Enables/Disables MoEngage logs.<br><b>Note : This API should be used only if logs are required
   * in production/signed builds.</b>
   * @param status true to enable logs,else false
   */
  public void setLogStatus(boolean status){
    Logger.setLogStatus(status);
  }

  /**
   * Redirect data to Indian servers.
   * Use {@link MoEHelper#redirectDataToRegion(int)}
   * @param value true if you want to redirect data to indian server
   */
  @Deprecated
  public void setDataRedirection(boolean value){
    ConfigurationProvider.getInstance(mContext).setRouteTraffic(value);
  }


  public void optOutOfAndroidIdCollection(@NonNull Context context, boolean optOut){
    ConfigurationProvider.getInstance(context).optOutOfAndroidIdCollection(optOut);
  }

  public void optOutOfOperatorNameCollection(@NonNull Context context, boolean optOut){
    ConfigurationProvider.getInstance(context).optOutOfOperatorNameCollection(optOut);
  }

  public void optOutOfIMEICollection(@NonNull Context context, boolean optOut){
    ConfigurationProvider.getInstance(context).optOutOfIMEICollection(optOut);
  }

  public void optOutOfDeviceAttributeCollection(@NonNull Context context, boolean optOut){
    ConfigurationProvider.getInstance(context).optOutOfDeviceAttributesCollection(optOut);
  }

  /**
   * Tracks device locale.
   * Usage : Call the method in the application class of the application
   */
  public void trackDeviceLocale(){
    mDispatcher.trackDeviceLocale();
  }

  public void setAlias(double currentId){
    try{
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      mDispatcher.setAlias(userJSON);
    } catch(Exception e){
      Logger.f("MoEHelper: setAlias() ", e);
    }
  }

  public void setAlias(String currentId){
    try{
      if (TextUtils.isEmpty(currentId)){
        Logger.e("Updated id cannot be null");
        return;
      }
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      mDispatcher.setAlias(userJSON);
    } catch(Exception e){
      Logger.f("MoEHelper: setAlias() ", e);
    }
  }

  public void setAlias(long currentId){
    try{
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      mDispatcher.setAlias(userJSON);
    } catch(Exception e){
      Logger.f("MoEHelper: setAlias() ", e);
    }
  }

  public void setAlias(int currentId) {
    try {
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      mDispatcher.setAlias(userJSON);
    } catch (Exception e) {
      Logger.f("MoEHelper: setAlias() ", e);
    }
  }

  /**
   * Set push preference for the user.
   * @param pushPreference push preference value true or false
   */
  public void trackUserPushPreference(boolean pushPreference){
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_PUSH_PREFERENCE, pushPreference);
  }

  /**
   * Redirect data to different regions if required.<br/>
   * Possible values : {@link MoEHelperConstants#MOE_REGION_DEFAULT},
   * {@link MoEHelperConstants#MOE_REGION_INDIA},
   * {@link MoEHelperConstants#MOE_REGION_EU},
   * @param regionConstant one of the possible values based on the region you want to redirect to.
   */
  public void redirectDataToRegion(int regionConstant){
    ConfigurationProvider.getInstance(mContext).setDataRegion(regionConstant);
  }

  private long flushInterval = -1;

  /**
   * Set the interval(in seconds) at which the SDK flushes out the data to the server.<br>
   *   Note: Value will only be accepted if it is greater than the SDK defined flush interval
   * @param flushInterval time interval to flush data. Value should be in seconds
   */
  public void setFlushInterval(long flushInterval){
    if (flushInterval < ConfigurationProvider.getInstance(mContext).getPeriodicFlushTime()){
      Logger.e("MoEHelper:setFlushInterval() cannot set interval less than threshold. Threshold "
          + "value: " + ConfigurationProvider.getInstance(mContext).getPeriodicFlushTime());
      return;
    }
    this.flushInterval = flushInterval;
  }

  public long getFlushInterval(){
    return flushInterval;
  }

  private boolean isPeriodicSyncEnabled = true;

  /**
   * Enable/Disable periodic flush made by the SDK. By default periodic flush is enabled
   * @param periodicFlushState true to enable, else false
   */
  public void setPeriodicFlushState(boolean periodicFlushState){
    isPeriodicSyncEnabled = periodicFlushState;
  }

  public boolean getPeriodicSyncState(){
    return isPeriodicSyncEnabled;
  }

  public void fetchDeviceTriggersIfRequired(){
    MoEDTManager.getInstance().syncDeviceTriggersIfRequired(mContext);
  }
}
