package com.moengage.inapp;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.WorkerThread;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.Logger;
import java.util.HashMap;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class InAppController {

  public enum NETWORK_CALL_TYPE {

    AUTO_TRIGGER_EVENT,
    SYNC_IN_APPS,
    SINGLE_FETCH
  }

  private InAppHandler inAppHandler;

  /**
   * orientation of the activity in which in-app is being shown
   */
  private int mActivityOrientation = -1;
  /**
   * Name of activity in which in-app was being shown
   */
  private String mActivityName = null;
  private static InAppController _INSTANCE;

  private InAppController() {
    //check for PushHandler on class path
    loadInAppHandler();
  }

  private void loadInAppHandler() {
    try {
      Class handler = Class.forName("com.moengage.inapp.InAppHandlerImpl");
      inAppHandler = (InAppHandler) handler.newInstance();
      Logger.v("InAppController:loadInAppHandler InApp Module present");
    } catch (Exception e) {
      Logger.e("InAppController : loadInAppHandler : InApp Module not present " + e.getMessage());
    }
  }

  public InAppHandler getInAppHandler() {
    return inAppHandler;
  }

  public static InAppController getInstance() {
    if (_INSTANCE == null) {
      _INSTANCE = new InAppController();
    }
    return _INSTANCE;
  }

  public String getActivityName() {
    return mActivityName;
  }

  public void setActivityName(String activityName) {
    mActivityName = activityName;
  }

  public int getActivityOrientation() {
    return mActivityOrientation;
  }

  public void setActivityOrientation(int activityOrientation) {
    mActivityOrientation = activityOrientation;
  }

  public interface InAppHandler {

    /**
     * (non-JavaDoc)
     * Checks for in-app messages
     *
     * @param context Application Context
     */
    void showInAppIfPossible(Context context);

    /**
     * (non-Javadoc)
     * Shows in-app on Configuration change(a.k.a rotation )
     *
     * @param context Application Context
     */
    void showInAppOnConfigurationChange(Context context);

    /**
     * (non-Javadoc)
     * Sync in-apps from the server
     *
     * @param context Application Context
     */
    void syncOrShowInApps(Context context);

    /**
     * (non-Javadoc)
     * Try to show in-app for a smart event
     *
     * @param context Application Context
     * @param responseObject json response for the smart trigger
     */
    @WorkerThread void tryShowAutoTriggerInApp(Context context, JSONObject responseObject);

    /**
     * (non-Javadoc)
     * Parses the response for in-app api calls.
     *
     * @param responseObject API response object
     * @param context Application Context
     */
    void parseAndSaveInApps(JSONObject responseObject, Context context);

    /**
     * (non-Javadoc)
     * Registers a smart event and tries to show in-app for it
     *
     * @param context Application Context
     * @param event event for which there is a trigger
     */
    void registerAutoTriggerEvent(Context context, Event event);

    /**
     * (non-Javadoc)
     * Register to show in-apps in the activity.<br>
     * Should be called from {@link Activity#onStart()}
     *
     * @param currentActivity Activity instance in which in-apps needs to be shown.
     */
    void registerInAppManager(Activity currentActivity);

    /**
     * (non-Javadoc)
     * Unregister to show in-apps in the activity.<br>
     * Should be called from {@link Activity#onStop()}
     *
     * @param currentActivity Activity instance in which in-apps needs to be shown.
     */
    void unregisterInAppManager(Activity currentActivity);

    /**
     * (non-Javadoc)
     * Set in-app's server sync status.
     *
     * @param state true if sync is complete, else false
     */
    void setInappsSynced(boolean state);

    /**
     * Show linked in-app
     * @param campaignId Linked in-app campaignId
     * @param context Application Context
     */
    void fetchLinkedInApp(Context context, String campaignId);

    /**
     * Show linked/test inapp
     * @param context Application Context
     * @param jsonObject InApp JSON
     * @param paramsMap Campaign Id map
     */
    void showLinkedInApp(Context context, JSONObject jsonObject, HashMap<String, String> paramsMap);

    /**
     * Show error dialog if test in-app fails
     * @param errorMessage : error message
     */
    void showTestInAppErrorDialog(String errorMessage);
  }
}
