package com.moengage.core;

import android.content.Context;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.inapp.InAppController;
import java.util.ArrayList;
import java.util.List;
import org.json.JSONObject;

/**
 * Central point for all event related processing, i.e. tracking, blacklisting, trigger-events
 * @author Umang Chamaria
 */

public class MoEEventManager{

  private static MoEEventManager _INSTANCE = null;

  private List<String> mBlackListedEvents = null;

  private ConfigurationProvider mConfigProvider = null;

  private Context mContext;

  private int mEventCounter = 0;

  private List<String> mTriggerEvents;

  private List<String> mFlushEvents;

  private MoEEventManager(Context context){
    mContext = context;
    mConfigProvider = ConfigurationProvider.getInstance(context);
    getBlackListedEvents();
    getTriggerEvents();
    getFlushEvents();
  }

  public static MoEEventManager getInstance(Context context){
    if (_INSTANCE == null){
      _INSTANCE = new MoEEventManager(context);
    }
    return _INSTANCE;
  }

  private boolean isEventBlackListed(String eventName) {
    return !mBlackListedEvents.isEmpty() && mBlackListedEvents.contains(eventName);
  }

  void getBlackListedEvents() {
    try {
      mBlackListedEvents = new ArrayList<>();
      String blackListEventsString = mConfigProvider.getBlackListEvents();
      if (!TextUtils.isEmpty(blackListEventsString)) {
        String[] blackListEventArrary = blackListEventsString.split(MoEConstants.EVENT_SEPERATOR);
        if (blackListEventArrary != null && blackListEventArrary.length > 0) {
          for (String blackEvent : blackListEventArrary) {
            mBlackListedEvents.add(blackEvent);
          }
        }
      }
    } catch (Exception e) {
      Logger.e("MoEEventManager: getBlackListedEvents() ", e);
    }
  }

  int getEventCounter() {
    return mEventCounter;
  }

  void incrementEventCounter() {
    mEventCounter++;
  }

  void setEventCounter(int counter) {
    mEventCounter = counter;
  }

  /**
   * Tracks an event and if the event is a smart trigger event makes a call to the backend.
   *
   * @param action event name
   * @param attrs event attributes
   */
  public void trackEvent(final String action, JSONObject attrs) {
    try {
      if (!mConfigProvider.isAppEnabled()) return;
      if (isEventBlackListed(action)) {
        Logger.e("MoEEventManager: Event Blacklisted : " + action);
        return;
      }
      Event event = new Event(MoEHelperUtils.getDatapointJSON(action.trim(), attrs));
      if (action.equals(MoEHelperConstants.EVENT_APP_INSTALL)) {
        MoEUtils.setInstallRegistered(mContext);
      }
      if (isSmartTriggerEvent(action)) {
        // This is a smart action
        Logger.v("MoEEventManager:acting on auto trigger");
        InAppController.InAppHandler inAppHandler = InAppController.getInstance().getInAppHandler();
        if (inAppHandler != null) {
          inAppHandler.registerAutoTriggerEvent(mContext, event);
        }
      }
      //write datapoint to storage
      MoEDispatcher.getInstance(mContext).writeDataPointToStorage(event);
      //check if flush event and flush
      if (isFlushEvent(action)){
        Logger.v("MoEEventManager: trackEvent() flush event, flushing events");
        MoEHelper.getInstance(mContext).syncInteractionDataNow();
      }
    } catch (Exception e) {
      Logger.f("MoEEventManager: trackEvent() ", e);
    }
  }

  /**
   * Checks whether a performed action is a smart trigger event or not
   * @param action The action which needs to be checked
   * @return true if the action is a smart trigger event, false otherwise
   */
  private boolean isSmartTriggerEvent(final String action) {
    return (mTriggerEvents != null && mTriggerEvents.contains(action)) || MoEHelperConstants
        .EVENT_APP_INSTALL.equals(action);
  }

  /**
   * Reads the trigger events list from storage and cache it.
   */
  public void getTriggerEvents(){
    try{
      String list = mConfigProvider.getSmartTriggerList();
      if (list == null){
        Logger.v("MoEEventManager:No smart triggers found");
        return;
      }
      String triggers[] = list.split(MoEConstants.EVENT_SEPERATOR);
      mTriggerEvents = new ArrayList<>(triggers.length);
      for (String event : triggers){
        mTriggerEvents.add(event);
      }
    } catch(Exception e){
      Logger.e("MoEEventManager: getTriggerEvents()", e);
    }
  }

  void getFlushEvents(){
    try {
      String events = mConfigProvider.getFlushEvents();
      if (TextUtils.isEmpty(events)){
        Logger.v("MoEEventManager: getFlushEvents() No flush events");
        return;
      }
      String[] flushEvents = events.split(MoEConstants.EVENT_SEPERATOR);
      mFlushEvents = new ArrayList<>(flushEvents.length);
      for (String event: flushEvents){
        mFlushEvents.add(event);
      }
    }catch (Exception e){
      Logger.e("MoEEventManager: getFlushEvents()");
    }
  }

  private boolean isFlushEvent(String eventName){
    return mFlushEvents != null && mFlushEvents.contains(eventName);
  }
}
