/* ************************************************************************
 *
 * MOENGAGE CONFIDENTIAL
 * __________________
 *
 *  [2014] - [2015] MoEngage Inc.
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moe.pushlibrary;

import android.location.Location;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * A helper class for builder the tracking payload
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3.02
 */
public class PayloadBuilder {

  public JSONObject generalAttrs;
  public JSONObject customAttrs;

  private void notNullCheck(String attrName) throws Exception {
    if(TextUtils.isEmpty(attrName))throw new Exception("Action name cannot be empty");
  }

  /**
   * Add an integer attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value
   * @return the Builder instance
   */
  public PayloadBuilder putAttrInt(String attrName, int attrValue) {
    try {
      notNullCheck(attrName);
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    } catch (Exception e) {
      Logger.f("PayloadBuilder: putAttrInt", e);
    }
    return this;
  }

  /**
   * Add an String attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value
   * @return the Builder instance
   */
  public PayloadBuilder putAttrString(String attrName, String attrValue) {
    try {
      notNullCheck(attrName);
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    } catch (Exception e) {
      Logger.f("PayloadBuilder: putAttrString", e);
    }
    return this;
  }

  /**
   * Add an boolean attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value
   * @return the Builder instance
   */
  public PayloadBuilder putAttrBoolean(String attrName, boolean attrValue) {
    try {
      notNullCheck(attrName);
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    } catch (Exception e) {
      Logger.f("PayloadBuilder: putAttrBoolean", e);
    }
    return this;
  }

  /**
   * Add an float attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value
   * @return the Builder instance
   */
  public PayloadBuilder putAttrFloat(String attrName, float attrValue) {
    try {
      notNullCheck(attrName);
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    } catch (Exception e) {
      Logger.f("PayloadBuilder: putAttrFloat", e);
    }
    return this;
  }

  /**
   * Add an double attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value
   * @return the Builder instance
   */
  public PayloadBuilder putAttrDouble(String attrName, double attrValue) {
    try {
      notNullCheck(attrName);
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    } catch (Exception e) {
      Logger.f("PayloadBuilder: putAttrDouble", e);
    }
    return this;
  }

  /**
   * Add an long attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value
   * @return the Builder instance
   */
  public PayloadBuilder putAttrLong(String attrName, long attrValue) {
    try {
      notNullCheck(attrName);
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    } catch (Exception e) {
      Logger.f("PayloadBuilder: putAttrLong", e);
    }
    return this;
  }

  /**
   * Add an Date attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value of type {@link Date}
   * @return the Builder instance
   */
  public PayloadBuilder putAttrDate(String attrName, Date attrValue) {
    try {
      notNullCheck(attrName);
      if (null == customAttrs) {
        customAttrs = new JSONObject();
      }
      JSONArray timeStamp;
      if (customAttrs.has(ATTR_TIMESTAMP)) {
        timeStamp = customAttrs.getJSONArray(ATTR_TIMESTAMP);
      } else {
        timeStamp = new JSONArray();
      }
      JSONObject attr = new JSONObject();
      attr.put(attrName.trim(), attrValue.getTime());
      timeStamp.put(attr);
      customAttrs.put(ATTR_TIMESTAMP, timeStamp);
    } catch (Exception e) {
      Logger.f("EventPayload: putAttrDate: ", e);
    }
    return this;
  }

  /**
   * @param attrName The Event attribute name
   * @param dateString String representation of Date
   * @param dateFormat Format of the date string which is being passed
   * @return the Builder instance
   */
  public PayloadBuilder putAttrDate(String attrName, String dateString, String dateFormat) {
    try{
      notNullCheck(attrName);
      DateFormat format = new SimpleDateFormat(dateFormat, Locale.ENGLISH);
      return putAttrDate(attrName.trim(), format.parse(dateString));
    }catch(Exception e){
      Logger.f("EventPayload: putAttrDate 2: ", e);
    }
    return this;
  }

  /**
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value of type {@link GeoLocation}
   * @return the Builder instance
   */
  public PayloadBuilder putAttrLocation(String attrName, GeoLocation attrValue) {
    try {
      notNullCheck(attrName);
      if (null == customAttrs) {
        customAttrs = new JSONObject();
      }
      JSONArray location;
      if (customAttrs.has(ATTR_LOCATION)) {
        location = customAttrs.getJSONArray(ATTR_LOCATION);
      } else {
        location = new JSONArray();
      }
      JSONObject attr = new JSONObject();
      attr.put(attrName.trim(), attrValue.latitude + "," + attrValue.longitude);
      location.put(attr);
      customAttrs.put(ATTR_LOCATION, location);
    } catch (Exception e) {
      Logger.f("EventPayload: putAttrLocation1: ", e);
    }
    return this;
  }

  /**
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value of type {@link Location}
   * @return the Builder instance
   */
  public PayloadBuilder putAttrLocation(String attrName, Location attrValue) {
    try {
      notNullCheck(attrName);
      if (null == customAttrs) {
        customAttrs = new JSONObject();
      }
      JSONArray location;
      if (customAttrs.has(ATTR_LOCATION)) {
        location = customAttrs.getJSONArray(ATTR_LOCATION);
      } else {
        location = new JSONArray();
      }
      JSONObject attr = new JSONObject();
      attr.put(attrName.trim(), attrValue.getLatitude() + "," + attrValue.getLongitude());
      location.put(attr);
      customAttrs.put(ATTR_LOCATION, location);
    } catch (Exception e) {
      Logger.f("EventPayload: putAttrLocation2: ", e);
    }
    return this;
  }

  /**
   * @param attrName The Event attribute name
   * @param latitude The latitude value of the geo location
   * @param longitude The longitude value of the geo location
   * @return the Builder instance
   */
  public PayloadBuilder putAttrLocation(String attrName, double latitude, double longitude) {
    try {
      notNullCheck(attrName);
      if (null == customAttrs) {
        customAttrs = new JSONObject();
      }
      JSONArray location;
      if (customAttrs.has(ATTR_LOCATION)) {
        location = customAttrs.getJSONArray(ATTR_LOCATION);
      } else {
        location = new JSONArray();
      }
      JSONObject attr = new JSONObject();
      attr.put(attrName.trim(), latitude + "," + longitude);
      location.put(attr);
      customAttrs.put(ATTR_LOCATION, location);
    } catch (Exception e) {
      Logger.f("EventPayload: putAttrLocation3: ", e);
    }
    return this;
  }

  /**
   * Add an Date attribute value to the payload
   *
   * @param attrName The Event attribute name
   * @param attrValue The Event attribute Value of type long
   * @return the Builder instance
   */
  public PayloadBuilder putAttrDateEpoch(String attrName, long attrValue) {
    try {
      notNullCheck(attrName);
      if (null == customAttrs) {
        customAttrs = new JSONObject();
      }
      JSONArray timeStamp;
      if (customAttrs.has(ATTR_TIMESTAMP)) {
        timeStamp = customAttrs.getJSONArray(ATTR_TIMESTAMP);
      } else {
        timeStamp = new JSONArray();
      }
      JSONObject attr = new JSONObject();
      attr.put(attrName.trim(), attrValue);
      timeStamp.put(attr);
      customAttrs.put(ATTR_TIMESTAMP, timeStamp);
    } catch (Exception e) {
      Logger.f("EventPayload: putAttrDate: ", e);
    }
    return this;
  }

  public PayloadBuilder putAttrJSONArray(@NonNull String attrName, @NonNull JSONArray attrValue){
    try {
      notNullCheck(attrName);
      if (attrValue == null) return this;
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    }catch (Exception e){
      Logger.f("EventPayload: putAttrJSONArray: ", e);
    }
    return this;
  }

  public PayloadBuilder putAttrJSONObject(@NonNull String attrName, @NonNull JSONObject attrValue){
    try {
      notNullCheck(attrName);
      if (attrValue == null) return this;
      if( null == generalAttrs)generalAttrs = new JSONObject();
      generalAttrs.put(attrName.trim(), attrValue);
    }catch (Exception e){
      Logger.f("EventPayload: putAttrJSONObject: ", e);
    }
    return this;
  }


  /**
   * Build the Event Payload and pass it to trackEvent
   *
   * @return populated EventPayload instance or null if nothing is set
   */
  public JSONObject build() {
    return MoEHelperUtils.getDatapointJSON(generalAttrs, customAttrs);
  }

  private static final String ATTR_TIMESTAMP = "timestamp";
  private static final String ATTR_LOCATION = "location";
}
