package com.moengage.core;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.models.UserAttribute;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import java.util.ArrayList;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * (non-JavaDoc)
 * Sends all the events to the server once the application is closed.
 * Service constant MSG_SEND_INTERACTION_DATA
 *
 * @author Umang Chamaria
 */
class SendInteractionDataTask extends SDKTask {
  SendInteractionDataTask(Context context) {
    super(context);
  }

  @Override public TaskResult execute() {
    try {
      if (!ConfigurationProvider.getInstance(mContext).isAppEnabled()) return null;
      Logger.v("SendInteractionDataTask : executing task");
      String referrer = MoEHelperUtils.getInstallReferrer(mContext);
      if (!TextUtils.isEmpty(referrer)) {
        JSONObject userJson = new JSONObject();
        try {
          userJson.put(MoEHelperConstants.PREF_KEY_INSTALL_REFERRER, referrer);
          MoEUtils.trackEventInternal(MoEConstants.EVENT_ACTION_USER_ATTRIBUTE, userJson, mContext);
        } catch (Exception e) {
          Logger.f("SendInteractionDataTask:setUserAttribute", e);
        }
        MoEHelperUtils.removeInstallReferrer(mContext);
      }
      mTaskResult.setIsSuccess(sendInteractionData());
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
        scheduleRetryDataSyncJob();
      }else {
        scheduleRetryDataSyncAlarm();
      }
      Logger.v("SendInteractionDataTask : completed task");
      return mTaskResult;
    } catch (Exception e) {
      Logger.f("SendInteractionData: execute() ", e);
    }
    return null;
  }

  @Override public String getTaskTag() {
    return TAG_SEND_INTERACTION_DATA;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private boolean sendInteractionData() {
    boolean result = false;
    ArrayList<Event> batchedData = null;
    for (; ; ) {
      batchedData = MoEDAO.getInstance(mContext).getBatchedInteactionData(100);
      Logger.d(
          "SendInteractionDataTask : sendInteractionData:Fetching interaction data in batches");
      if (null == batchedData || batchedData.isEmpty()) {
        Logger.d("SendInteractionDataTask : sendInteractionData: Found Nothing to send");
        return result;
      }
      String interactionData = convertEventsToJSON(batchedData);
      if (interactionData == null) {
        return true;
      }
      Logger.v("SendInteractionDataTask : sendInteractionData " + interactionData);
      try {
        String endpoint = MoEConstants.API_ENDPOINT_STORE_REPORTS_V2;
        if (shouldSendDataToTestServer()) {
          endpoint = MoEConstants.API_ENDPOINT_INTEGRATION_VERIFICATION_REPORT_ADD;
        }
        result = APIManager.sendInteractionReport(mContext, interactionData, endpoint);
      }catch (Exception e) {
        Logger.f("SendInteractionDataTask : API failed", e);
        return false;
      }
      if (result) {
        MoEDAO.getInstance(mContext).deleteBatchedInteractionData(batchedData, mContext);
        Logger.d("SendInteractionDataTask : Deleting interaction data in batches");
      } else {
          int retryCount = ConfigurationProvider.getInstance(mContext).getImmediateRetryCount();
          switch(retryCount){
            case 0:
              scheduleImmediateRetry(1);
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(++retryCount);
              break;
            case 1:
              scheduleImmediateRetry(3);
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(++retryCount);
              break;
            default:
              ConfigurationProvider.getInstance(mContext).setImmediateRetryCount(0);
          }
        break;
      }
      batchedData.clear();
    }
    return result;
  }

  private boolean shouldSendDataToTestServer() {
    boolean isDeviceRegisteredForVerification =
        ConfigurationProvider.getInstance(mContext).isDeviceRegisteredForVerification();
    long registrationTime =
        ConfigurationProvider.getInstance(mContext).getVerificationRegistrationTime();
    return isDeviceRegisteredForVerification && ((registrationTime + 3600 * 1000)
        > System.currentTimeMillis());
  }

  @Nullable private String convertEventsToJSON(ArrayList<Event> eventList) {
    if (null == eventList || eventList.isEmpty()) {
      return null;
    }
    JSONArray jsonInteractions = new JSONArray();
    for (Event event : eventList) {
      try {
        jsonInteractions.put(new JSONObject(event.details));
      } catch (Exception e) {
        Logger.f("MoEUtils:convertEventsToJSON", e);
      }
    }
    if (jsonInteractions.length() == 0) {
      return null;
    }
    JSONObject jsonInteractionsObject = new JSONObject();
    try {
      jsonInteractionsObject.put(MoEConstants.ATTR_INTEACTION_VIEWS_COUNT,
          jsonInteractions.length());
      jsonInteractionsObject.put(MoEConstants.ATTR_INTEACTION_VIEWS_INFO, jsonInteractions);
      JSONObject sdkIdentifiers = getIdentifiersJSON();
      if (sdkIdentifiers != null){
        jsonInteractionsObject.put(MoEConstants.ATTR_SDK_IDENTIFIERS, sdkIdentifiers);
      }
    } catch (Exception e) {
      Logger.f("MoEUtils:convertEventsToJSON", e);
      return null;
    }
    return jsonInteractionsObject.toString();
  }

  @Nullable
  private JSONObject getIdentifiersJSON(){
    try {
      JSONObject identifierJSON = new JSONObject();
      UserAttribute userAttributeUniqueId = MoEDAO.getInstance(mContext).getUserAttributesForKey
          (MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID);
      if (userAttributeUniqueId != null){
        identifierJSON.put(MoEConstants.ATTR_MOE_USER_ID, userAttributeUniqueId.userAttributeValue);
      }
      String segmentAnonymousId = ConfigurationProvider.getInstance(mContext)
          .getSegmentAnonymousId();
      if (!TextUtils.isEmpty(segmentAnonymousId)) {
        identifierJSON.put(MoEConstants.ATTR_SEGMENT_ID, segmentAnonymousId);
      }
     if (identifierJSON.length() !=0){
        return identifierJSON;
     }
    } catch (Exception e) {
      Logger.f("SendInteractionDataTask: getIdentifiersJSON() ", e);
    }
    return null;
  }

  private void scheduleRetryDataSyncAlarm(){
    Logger.v("Scheduling data sync retry");
    Intent alarmIntent = new Intent(mContext, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(mContext, 88888, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
    alarmManager.setInexactRepeating(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + ConfigurationProvider
        .getInstance(mContext).getRetrySyncTime(), ConfigurationProvider.getInstance
        (mContext).getRetrySyncTime(), pendingIntent);
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  private void scheduleRetryDataSyncJob(){
    Logger.v("Scheduling retry data sync job");
    ComponentName serviceComponent = new ComponentName(mContext, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(77777, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(System.currentTimeMillis() + ConfigurationProvider.getInstance
        (mContext).getRetrySyncTime() * 2);
    builder.setMinimumLatency( ConfigurationProvider.getInstance
        (mContext).getRetrySyncTime());
    JobScheduler jobScheduler =
        (JobScheduler) mContext.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    jobScheduler.schedule(builder.build());
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  private void scheduleImmediateRetrySyncJob(int minutes){
    Logger.v("Scheduling immediate retry data sync job");
    ComponentName serviceComponent = new ComponentName(mContext, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(666666, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(System.currentTimeMillis() + minutes * 2 * 60 * 1000);
    builder.setMinimumLatency(minutes * 60 * 1000);
    JobScheduler jobScheduler =
        (JobScheduler) mContext.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    jobScheduler.schedule(builder.build());
  }

  private void scheduleImmediateRetryAlarm(int minutes){
    Intent alarmIntent = new Intent(mContext, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(mContext, 55555, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
    alarmManager.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + (minutes * 60 * 1000),
        pendingIntent);
  }

  private void scheduleImmediateRetry(int minutes){
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
      scheduleImmediateRetrySyncJob(minutes);
    }else {
      scheduleImmediateRetryAlarm(minutes);
    }
  }
}
