/* ************************************************************************
 *
 * MOENGAGE CONFIDENTIAL
 * __________________
 *
 *  [2014] - [2015] MoEngage Inc.
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.core;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ActivityInfo;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.AndroidException;
import com.delight.pushlibrary.R;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.models.UserAttribute;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

public final class ConfigurationProvider {

  //Should have the Generic params and the commonly used params
  //SHOULD NOT have the
  private HashMap<String, Object> configMap;

  private boolean initialized = false;
  private boolean disableGCMRegistration = false;
  private static long INAPP_DELAY_DURATION = 15 * 60;
  private static final String PREF_NAME = "pref_moe";
  private static final String SENDER_ID = "SENDER_ID";
  private static final String APP_ID = "APP_ID";
  private static final String SKIP_GCM_REGISTRATION = "SKIP_GCM_REGISTRATION";
  private static final String NOTIFICATION_ICON = "NOTIFICATION_ICON";
  private static final String NOTIFICATION_LARGE_ICON = "NOTIFICATION_LARGE_ICON";
  private static final String NOTIFICATION_TONE = "NOTIFICATION_TONE";
  private static final String NOTIFICATION_COLOR = "NOTIFICATION_COLOR";
  private static final String NOTIFICATION_TYPE = "NOTIFICATION_TYPE";
  private static final String CURRENT_APP_VERSION = "APP_VERSION";
  private static final String PREF_KEY_LAST_CAMPAIGN_ID = "MOE_LAST_CAMPAIGN_ID";
  private static final String BAIDU_API_KEY = "api_key";
  /**
   * <b>Preference key</b>. Stores the value of the unique ID that was sent
   * from server if using the V1 api
   */
  private static final String PREF_KEY_MOE_GEN_UID = "APP_UNIQUE_ID_MOE";
  private static final String PREF_CHAT_LAST_UPD = "PREF_KEY_LAST_CHAT";
  /**
   * <b>Preference key</b>.The version associated with the app . This is the
   * Version Code specified in the application manifest
   */
  private static final String PREF_KEY_APP_VERSION = "appVersion";
  private static final String PREF_KEY_DEVICE_REGISTERED = "PREF_KEY_DEVICE_REGISTERED";

  private static final int NOTIFICATION_ID = 17987;
  private static final String PREF_KEY_LAST_NOTIFICATION_ID = "PREF_LAST_NOTIFICATION_ID";

  Context mContext;

  //constructor intentionally hidden
  private ConfigurationProvider(Context context) {
    if (context == null) {
      Logger.e("ConfigurationProvider : context passed is null");
      return;
    }
    mContext = context;
    init();
  }

  //A mutex to be used by all methods which need synchornization
  private final Object lock = new Object();

  private void init() {
    synchronized (lock) {
      //return if a call was waiting and now is no more required
      if (initialized) return;
      configMap = new HashMap<>();
      INAPP_DELAY_DURATION = getSharedPrefs().getInt(PREF_KEY_IN_APP_TIME_DIFF, 900);
      upgradeFromOldSharedPrefIfRequired();
      try {
        PackageInfo packageInfo =
            mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), 0);
        configMap.put(CURRENT_APP_VERSION, packageInfo.versionCode);
      } catch (PackageManager.NameNotFoundException e) {
        Logger.f("Could not get package name: ", e);
      } catch (Exception e) {
        Logger.f("Could not get package name: ", e);
      }
      //readSDKConfiguration();
      getSDKConfiguration();
      getOptedOutActivitiesFromManifest();
      getSentScreenNamesInit();
      initialized = true;
    }
  }

  private List<String> sentScreenNames;

  private void getSentScreenNamesInit() {
    try {
      sentScreenNames = new ArrayList<>();
      String savedScreens = getSentScreenNames();
      if (TextUtils.isEmpty(savedScreens))return;
      String[] savedScreenArray = savedScreens.split(MoEHelperConstants.EVENT_SEPERATOR);
      sentScreenNames.addAll(Arrays.asList(savedScreenArray));
      Logger.v("ConfigurationProvider: getSentScreenNamesInit: Saved screen : " + sentScreenNames
          .toString());
    }catch (Exception e){
      Logger.f("ConfigurationProvider: getSentScreenNamesInit: ", e);
    }

  }

  private static ConfigurationProvider _INSTANCE;

  public static ConfigurationProvider getInstance(Context context) {
    synchronized (ConfigurationProvider.class) {
      if (null == _INSTANCE) {
        _INSTANCE = new ConfigurationProvider(context);
      }
    }
    return _INSTANCE;
  }

  public boolean isGCMRegistrationEnabled() {
    return !disableGCMRegistration;
  }

  public void flush() {
    synchronized (ConfigurationProvider.class) {
      _INSTANCE = null;
    }
  }

  private SharedPreferences getSharedPrefs() {
    return mContext.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
  }

  public String getSenderIdIfAny() {
    return (String) configMap.get(SENDER_ID);
  }

  public String getAppId() {
    return (String) configMap.get(APP_ID);
  }

  public boolean isGCMRegistrationDisabled() {
    return disableGCMRegistration;
  }

  public int getNotificationSmallIcon() {
    return (Integer) configMap.get(NOTIFICATION_ICON);
  }

  // Method to update AppID passed through source like Segment API
  public void saveAppDetails(String senderId, String appId) {
    configMap.put(APP_ID, appId);
    if (senderId != null) {
      configMap.put(SENDER_ID, senderId);
      setSenderId(senderId);
    }
  }

  @Deprecated public String getUniqueId() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(PREF_KEY_MOE_GEN_UID, "");
  }

  public int getNotificationLargeIconIfAny() {
    return (Integer) configMap.get(NOTIFICATION_LARGE_ICON);
  }

  public String getNotificationToneIfAny() {
    return (String) configMap.get(NOTIFICATION_TONE);
  }

  public int getNotificationColor() {
    Object obj = configMap.get(NOTIFICATION_COLOR);
    if (null == obj) return -1;
    return (Integer) obj;
  }

  public int getNotificationDisplayType() {
    return (Integer) configMap.get(NOTIFICATION_TYPE);
  }

  private final Object gcmTokenLock = new Object();

  public String getGCMToken() {
    synchronized (gcmTokenLock) {
      final SharedPreferences prefs = getSharedPrefs();
      String registrationId = prefs.getString(PREF_KEY_PROPERTY_REG_ID, null);
      if (TextUtils.isEmpty(registrationId)) {
        Logger.v("ConfigurationProvider:getGCMToken: Registration not found.");
        return null;
      }
      return registrationId;
    }
  }

  public void setGCMToken(String token) {
    synchronized (gcmTokenLock) {
      final SharedPreferences prefs = getSharedPrefs();
      int appVersion = getAppVersion();
      Logger.v("ConfigurationProvider: SettingGCMToken : " + token);
      SharedPreferences.Editor editor = prefs.edit();
      editor.putString(PREF_KEY_PROPERTY_REG_ID, token).apply();
      editor.putInt(PREF_KEY_APP_VERSION, appVersion);
      editor.apply();
      setPushFailureCount(0);
    }
  }

  public int getAppVersion() {
    return (Integer) configMap.get(CURRENT_APP_VERSION);
  }

  public int getStoredAppVersion() {
    return getSharedPrefs().getInt(PREF_KEY_APP_VERSION, 0);
  }

  public void storeAppVersion(int version) {
    SharedPreferences pref = getSharedPrefs();
    pref.edit().putInt(PREF_KEY_APP_VERSION, version).apply();
  }

  /**
   * Get the SDK configuration
   */
  private void getSDKConfiguration() {
    ApplicationInfo info;
    try {
      info = mContext.getPackageManager()
          .getApplicationInfo(mContext.getPackageName(), PackageManager.GET_META_DATA);
    } catch (PackageManager.NameNotFoundException e1) {
      Logger.f("ConfigurationProvider:getSDKConfiguration", e1);
      return;
    } catch (Exception e) {
      Logger.f("ConfigurationProvider:getSDKConfiguration", e);
      return;
    }
    Bundle metaData = info.metaData;
    if (null == metaData) {
      Logger.f("ConfigurationProvider: How can meta be null");
      return;
    }

    if (metaData.containsKey(APP_ID)) {
      String appId = metaData.getString(APP_ID);
      if (!TextUtils.isEmpty(appId)) {
        configMap.put(APP_ID, appId);
      } else {
        {
          throw new IllegalStateException("No 'appId' added in manifest application meta");
        }
      }
    }

    if (metaData.containsKey(SENDER_ID)) {
      String senderId = metaData.getString(SENDER_ID);
      if (!TextUtils.isEmpty(senderId)) {
        senderId = senderId.trim();
        if (senderId.startsWith("id:")) {
          senderId = senderId.substring(3);
        }
        String oldSenderId = getStoredSenderId();
        Logger.i("ConfigurationProvider:getSDKConfig: old Sender Id: " + oldSenderId);
        configMap.put(SENDER_ID, senderId);
        setSenderId(senderId);
        Logger.i("ConfigurationProvider:getSDKConfig: Sender Id: " + senderId);
        if (!TextUtils.isEmpty(oldSenderId) && !oldSenderId.equals(senderId)) {
          Logger.e("ConfigurationProvider:getSDKConfig: change in Sender Id");
          setGCMToken(null);
          setDeviceRegistered(false);
        }
      } else {
        disableGCMRegistration = true;
        Logger.d(
            "ConfigurationProvider: App does not specify sender ID will not be able to register "
                + "for GCM");
      }
    }

    int resNotificationIcon = 0;
    if (metaData.containsKey(NOTIFICATION_ICON)) {
      resNotificationIcon = getDrawable(metaData.get(NOTIFICATION_ICON));
    }
    if (0 == resNotificationIcon || !MoEHelperUtils.isValidResourceId(mContext,
        resNotificationIcon)) {
      resNotificationIcon = getAppIcon();
    }
    Logger.v("ConfigurationProvider:getSDKConfig: Notification Icon: " + resNotificationIcon);
    configMap.put(NOTIFICATION_ICON, resNotificationIcon);

    //this is also optional as per the framework
    int largeIcon = 0;
    if (metaData.containsKey(NOTIFICATION_LARGE_ICON)) {
      largeIcon = getDrawable(metaData.get(NOTIFICATION_LARGE_ICON));
    }
    if (0 == largeIcon || !MoEHelperUtils.isValidResourceId(mContext, largeIcon)) {
      largeIcon = getAppIcon();
    }
    Logger.v("ConfigurationProvider:getSDKConfig: Notification Large Icon: " + largeIcon);
    configMap.put(NOTIFICATION_LARGE_ICON, largeIcon);

    if (metaData.containsKey(NOTIFICATION_TONE)) {
      try {
        String notificationTone = metaData.getString(NOTIFICATION_TONE);
        String tone = null;
        if (!TextUtils.isEmpty(notificationTone)) {
          if (notificationTone.contains("res")) {
            //String contents because @raw was used
            String[] resourcePathParts = notificationTone.split("/");
            String resName = resourcePathParts[resourcePathParts.length - 1];
            if (resName.contains(".")) {
              tone = resName.substring(resName.lastIndexOf("/") + 1, resName.lastIndexOf("."));
            } else {
              tone = resName.substring(resName.lastIndexOf("/") + 1);
            }
          } else {
            tone = notificationTone;
          }
        } else {
          tone = notificationTone;
        }
        Logger.d("ConfigurationProvider:getSDKConfig: Tone is: " + tone);
        configMap.put(NOTIFICATION_TONE, tone);
      } catch (Exception e) {
        Logger.f("ConfigurationProvider:getSDKConfig: tone", e);
      }
    }

    if (metaData.containsKey(NOTIFICATION_COLOR)) {
      try {
        int resNotificationColor;
        Object notifyObj = metaData.get(NOTIFICATION_COLOR);
        {
          Logger.d("ConfigurationProvider:getSDKConfig: Notification Color: " + notifyObj);
        }
        if (notifyObj instanceof Integer) {
          resNotificationColor = (Integer) notifyObj;
        } else {
          String color = (String) notifyObj;
          if (null != color && color.contains("res/color")) {
            resNotificationColor = mContext.getResources()
                .getIdentifier(color.substring(color.lastIndexOf("/") + 1), "color",
                    mContext.getPackageName());
          } else {
            resNotificationColor =
                mContext.getResources().getIdentifier(color, "color", mContext.getPackageName());
          }
        }

        if (0 < resNotificationColor) {
          configMap.put(NOTIFICATION_COLOR, resNotificationColor);
        }
      } catch (Exception e) {
        Logger.f("ConfigurationProvider:getSDKConfig: Color", e);
      }
    }

    try {
      int fallbackColor = mContext.getResources()
          .getIdentifier("moe_notification_color", "color", mContext.getPackageName());
      if (0 < fallbackColor) {
        configMap.put(NOTIFICATION_COLOR, fallbackColor);
      }
    } catch (Exception e1) {
      Logger.f("ConfigurationProvider:getSDKConfig: Color", e1);
    }

    if (metaData.containsKey(NOTIFICATION_TYPE)) {
      try {
        configMap.put(NOTIFICATION_TYPE, metaData.getInt(NOTIFICATION_TYPE));
      } catch (ClassCastException e) {
        Logger.f("ConfigurationProvider:getSDKConfig: notification type", e);
      }
    } else {
      //TODO test this
      configMap.put(NOTIFICATION_TYPE,
          mContext.getResources().getInteger(R.integer.notification_type_single));
    }

    if (metaData.containsKey(BAIDU_API_KEY)) {
      configMap.put(BAIDU_API_KEY, metaData.getString(BAIDU_API_KEY));
    }

    if (metaData.containsKey(SKIP_GCM_REGISTRATION)) {
      disableGCMRegistration = metaData.getBoolean(SKIP_GCM_REGISTRATION);
    }
    Logger.i("ConfigurationProvider: SDK initialized. MoEngage SDK version: "
        + MoEHelperConstants.LIB_VERSION);
  }

  private int getAppIcon() {
    int appIcon = 0;
    try {
      appIcon = mContext.getPackageManager().getApplicationInfo(mContext.getPackageName(), 0).icon;
    } catch (PackageManager.NameNotFoundException e) {
      Logger.f("ConfigurationProvider:getAppIcon: nameNotFoundException", e);
    } catch (Exception e) {
      Logger.f("ConfigurationProvider:getAppIcon: Exception", e);
    }
    return appIcon;
  }

  public String getLastPushCampaignId() {
    return getSharedPrefs().getString(PREF_KEY_LAST_CAMPAIGN_ID, null);
  }

  public void setLastPushCampaignId(String campaign_id) {
    getSharedPrefs().edit().putString(PREF_KEY_LAST_CAMPAIGN_ID, campaign_id).apply();
  }

  private final Object userLock = new Object();

  /**
   * Gets the Unique ID from the shared preference file. If it is not found
   * then generates a Unique ID and stores it before returning the newly
   * generated UniqueId
   *
   * @return The Unique ID associated with this app
   */
  public String getCurrentUserId() {
    synchronized (userLock) {
      UserAttribute uniqueId = MoEDAO.getInstance(mContext).getUserAttributesForKey
          (PREF_KEY_APP_UUID_MOE);
      //only if it is using API v2 generate and store the UUID
      if (uniqueId == null || TextUtils.isEmpty(uniqueId.userAttributeValue)) {
        uniqueId = generateAndSaveUniqueId();
      }
      return uniqueId.userAttributeValue;
    }
  }

  public void setLastChatUpdatedNow() {
    SharedPreferences pref = getSharedPrefs();
    pref.edit().putLong(PREF_CHAT_LAST_UPD, System.currentTimeMillis() / 1000).apply();
  }

  public long getLastChatUpdatedAt() {
    SharedPreferences pref = getSharedPrefs();
    return pref.getLong(PREF_CHAT_LAST_UPD, 0);
  }

  public int getNotificationId() {
    return getSharedPrefs().getInt(PREF_KEY_LAST_NOTIFICATION_ID, NOTIFICATION_ID);
  }

  public void updateNotificationId(int notificationId) {
    if (notificationId - NOTIFICATION_ID >= 100) {
      notificationId = NOTIFICATION_ID;
    }
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putInt(PREF_KEY_LAST_NOTIFICATION_ID, notificationId).apply();
  }

  private static final String PREF_KEY_APP_UUID_MOE = "APP_UUID";

  /**
   * Generates a unique ID using {@link UUID#randomUUID()}, stores the same in
   * the {@link SharedPreferences} file and returns the same
   *
   * @return the generated UUID
   */
  private UserAttribute generateAndSaveUniqueId() {
    String uid = generateUUID();
    UserAttribute uniqueId = new UserAttribute(PREF_KEY_APP_UUID_MOE, uid);
    MoEDAO.getInstance(mContext).addOrUpdateUserAttribute(uniqueId);
    return uniqueId;
  }

  /**
   * Get a randomly generated UID
   *
   * @return A randomly generated Sting representation of {@link UUID}
   */
  private String generateUUID() {
    UUID generatedUID = UUID.randomUUID();
    return generatedUID.toString();
  }

  public void setDeviceRegistered(boolean state) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putBoolean(PREF_KEY_DEVICE_REGISTERED, state).apply();
  }

  public boolean isDeviceRegistered() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getBoolean(PREF_KEY_DEVICE_REGISTERED, false);
  }

  public void upgradeFromOldSharedPrefIfRequired() {
    SharedPreferences oldPref = PreferenceManager.getDefaultSharedPreferences(mContext);
    SharedPreferences newPref = getSharedPrefs();
    if (oldPref.contains(PREF_KEY_SENDER_ID)) {
      Logger.v("Migrating from old shared pref");
      //migrate stuff
      String registrationId = oldPref.getString(PREF_KEY_PROPERTY_REG_ID, null);
      if (null != registrationId) {
        newPref.edit().putString(PREF_KEY_PROPERTY_REG_ID, registrationId).apply();
        oldPref.edit().remove(PREF_KEY_PROPERTY_REG_ID).apply();
      }
      String uid = oldPref.getString(PREF_KEY_MOE_GEN_UID, null);
      if (null != uid) {
        newPref.edit().putString(PREF_KEY_MOE_GEN_UID, uid).apply();
        oldPref.edit().remove(PREF_KEY_MOE_GEN_UID).apply();
      }

      String uuid = oldPref.getString(PREF_KEY_APP_UUID_MOE, null);
      if (null != uuid) {
        newPref.edit().putString(PREF_KEY_APP_UUID_MOE, uuid).apply();
        oldPref.edit().remove(PREF_KEY_APP_UUID_MOE).apply();
      }
      int duration = oldPref.getInt(PREF_KEY_IN_APP_TIME_DIFF, -1);
      if (duration != -1) {
        newPref.edit().putInt(PREF_KEY_IN_APP_TIME_DIFF, duration).apply();
        oldPref.edit().remove(PREF_KEY_IN_APP_TIME_DIFF).apply();
      }

      boolean initialized = oldPref.getBoolean(PREF_KEY_APP_INIT, false);
      if (initialized) {
        newPref.edit().putBoolean(PREF_KEY_APP_INIT, true).apply();
        oldPref.edit().remove(PREF_KEY_APP_INIT).apply();
      }
    }
  }

  /**
   * <b>Preference key</b>.The GCM registration ID retrieved after registering
   * with GCM
   */
  static final String PREF_KEY_PROPERTY_REG_ID = "registration_id";
  /**
   * <b>Preference key</b>. The 12 digit Google Project ID
   */
  static final String PREF_KEY_SENDER_ID = "SENDER_ID";
  /**
   * <b>Preference key</b>. Tells us that the SDK was initialized and does not
   * need to be initialized again
   */
  static final String PREF_KEY_APP_INIT = "APP_INITIALIZED_MOE";
  /**
   * <b>Preference key</b> and <b>User attribute</b> key which denotes whether
   * the user has enabled limited ad tracking or not.
   */
  static final String PREF_KEY_MOE_ISLAT = "PREF_KEY_MOE_ISLAT";
  /**
   * <b>Preference key</b> and <b>User attribute</b> key for the Google
   * Advertisement ID associated with the Google user.
   */
  static final String PREF_KEY_MOE_GAID = "PREF_KEY_MOE_GAID";
  /**
   * <b>Preference key</b>. Stores The minimum delay duration between two
   * consecutive in app messages as set by the user
   */
  static final String PREF_KEY_IN_APP_TIME_DIFF = "inapp_delay_dur";
  /**
   * <b>Preference key</b>. Stores the service name which needs to be started
   * for Push message received
   */
  private static final String PREF_KEY_MOE_PASS_SERVICE_NAME = "PREF_KEY_MOE_PASS_SERVICE_NAME";

  /**
   * Store the Google Advertisement ID
   *
   * @param gaid The Google Advertisement ID
   */
  public void storeGAID(String gaid) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putString(PREF_KEY_MOE_GAID, gaid).apply();
  }

  /**
   * Get the stored Google AdvertisementID
   *
   * @return the stored google advertisement ID
   */
  public String getStoredGAID() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(PREF_KEY_MOE_GAID, "");
  }

  /**
   * Store the isLimitedAdTracking flag
   *
   * @param isLAT the integer representation of the limited ad tracking flag
   */
  public void storeISLAT(int isLAT) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putInt(PREF_KEY_MOE_ISLAT, isLAT).apply();
  }

  /**
   * Get the stored isLimitedAdTracking flag
   *
   * @return the stored limited ad tracking flag
   */
  public int getStoredISLAT() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getInt(PREF_KEY_MOE_ISLAT, 2);
  }

  /**
   * Set the minimum delay between 2 in app messages
   *
   * @param seconds The delay duration which will be used for configuring the
   * minimum time for which 2 in apps can be shown. It has to be
   * greater than 0
   */
  public void setInAppDelayDuration(int seconds) {
    if (seconds <= 0) {
      Logger.f("Ignoring the supplied value. Minimum delay should be greater than 0");
      return;
    }
    INAPP_DELAY_DURATION = seconds;
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putInt(PREF_KEY_IN_APP_TIME_DIFF, seconds).apply();
  }

  /**
   * Get the minimum in app delay between 2 consecutive in app messages
   *
   * @return Returns the minimum delay duration which will be used to show 2
   * in app messages
   */
  public static long getInAppDelayDuration() {
    return INAPP_DELAY_DURATION * 1000;
  }

  /**
   * Removes the user related configuration which is stored int he system
   */
  public void removeUserConfigurationOnLogout() {
    SharedPreferences sp = getSharedPrefs();
    //remove all in app specific time stamps
    SharedPreferences.Editor editor = sp.edit();
    editor.remove(PREF_KEY_DEVICE_REGISTERED);
    editor.remove(PREF_KEY_INAPP_LAST_SYNC_TIME);
    editor.remove(PREF_KEY_LAST_CAMPAIGN_ID);
    editor.remove(PREF_KEY_LAST_NOTIFICATION_CLICKED);
    editor.remove(PREF_KEY_INAPP_LAST_SHOWN_TS);
    editor.remove(PREF_CHAT_LAST_UPD);
    editor.remove(PREF_KEY_APP_UUID_MOE);
    editor.remove(PREF_KEY_MOE_GEN_UID);
    editor.apply();
  }

  public void optOutOfAdIdCollection(boolean optOut) {
    configMap.put(PREF_KEY_GAID_COLLECTION, optOut);
  }

  public boolean isAdIdCollectionProhibitted() {
    return configMap.containsKey(PREF_KEY_GAID_COLLECTION)
        && configMap.get(PREF_KEY_GAID_COLLECTION) == Boolean.TRUE;
  }

  private int getDrawable(Object value) {
    int resource = 0;
    try {
      if (value instanceof Integer) {
        resource = (Integer) value;
        return resource;
      } else {
        String strObj = (String) value;
        if (strObj.contains("res")) {
          String name = strObj.substring(strObj.lastIndexOf("/") + 1, strObj.lastIndexOf("."));
          if (strObj.contains("drawable")) {
            resource =
                mContext.getResources().getIdentifier(name, "drawable", mContext.getPackageName());
          } else if (strObj.contains("mipmap")) {
            resource =
                mContext.getResources().getIdentifier(name, "mipmap", mContext.getPackageName());
          } else {
            resource = getAppIcon();
          }
        } else {
          String notifyIcon = (String) value;
          resource = mContext.getResources()
              .getIdentifier(notifyIcon, "drawable", mContext.getPackageName());
          if (0 == resource) {
            resource = mContext.getResources()
                .getIdentifier(notifyIcon, "mipmap", mContext.getPackageName());
          }
        }
      }
    } catch (Exception e) {
      Logger.f("ConfigurationProvider:getDrawable: ", e);
    }
    return resource;
  }

  public void setNewDBVersion(int version) {
    getSharedPrefs().edit().putInt(KEY_DB_VERSION, version).apply();
  }

  public int getDBVersion() {
    return getSharedPrefs().getInt(KEY_DB_VERSION, -1);
  }

  public void storeLastKnownLocation(GeoLocation location) {
    getSharedPrefs().edit()
        .putString(PREF_KEY_USER_LOCATION, location.latitude + "," + location.longitude)
        .commit();
  }

  public GeoLocation getLastKnownUserLocation() {
    try {
      String location = getSharedPrefs().getString(PREF_KEY_USER_LOCATION, null);
      if (null != location) {
        String[] latLong = location.split(",");
        return new GeoLocation(Double.parseDouble(latLong[0]), Double.parseDouble(latLong[1]));
      }
    } catch (Exception e) {
      Logger.f("ConfigurationProvider: getLastKnownUserLocation", e);
    }
    return null;
  }

  public long getLastInAppupdate() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getLong(PREF_KEY_INAPP_LAST_SYNC_TIME, 0);
  }

  public void setLastInappUpdateTime(long value) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putLong(PREF_KEY_INAPP_LAST_SYNC_TIME, value).apply();
  }

  /**
   * <b>Preference key</b>. Stores when the last in app message was shown to
   * the user.
   */
  private static final String PREF_KEY_INAPP_LAST_SHOWN_TS = "MOE_LAST_IN_APP_SHOWN_TIME";
  /**
   * <b>Preference key</b>. Denotes when the last in app sync occurred.
   */
  private static final String PREF_KEY_INAPP_LAST_SYNC_TIME = "MOE_LAST_IN_APP_UPDATE_TIME";

  /**
   * <b>Preference key</b>. Stores the value of the last notification clicked
   * time stamp
   */
  private static final String PREF_KEY_LAST_NOTIFICATION_CLICKED = "MOE_LAST_PUSH_CLICK_TIME";

  private static final String PREF_KEY_GAID_COLLECTION = "pref_key_isCollectGAID";

  private static final String PREF_KEY_USER_LOCATION = "key_geoinfo";

  private static final String KEY_DB_VERSION = "key_dbversion";

  public String getAppVersionName() {
    if (configMap.get("app_version_name") == null) {
      setAppVersionName();
    }
    return (String) configMap.get("app_version_name");
  }

  private void setAppVersionName() {
    PackageInfo pInfo = null;
    try {
      pInfo = mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), 0);
      configMap.put("app_version_name", pInfo.versionName);
    } catch (PackageManager.NameNotFoundException e) {
      Logger.f("ConfigurationProvider#setAppVersionName : Package Name not found", e);
    }
  }

  public void saveGeoIDList(String list) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putString(MoEHelperConstants.PREF_KEY_GEO_FENCE_LIST, list).apply();
  }

  public String getGeoIDList() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(MoEHelperConstants.PREF_KEY_GEO_FENCE_LIST, null);
  }

  public void saveNotificationSoundState(boolean state) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putBoolean(PREF_KEY_NOTIFICATION_SOUND, state).apply();
  }

  public boolean isNotificationSoundEnabled() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getBoolean(PREF_KEY_NOTIFICATION_SOUND, true);
  }

  private static final String PREF_KEY_NOTIFICATION_SOUND = "key_notification_sound";

  public long getLastInAppShownTime() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getLong(PREF_KEY_INAPP_LAST_SHOWN_TS, 0);
  }

  public void setLastInAppShownTime(long currentTimeMillis) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putLong(PREF_KEY_INAPP_LAST_SHOWN_TS, currentTimeMillis).apply();
  }

  public void saveSmartTriggerList(String actionList) {
    if (null == actionList) return;
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putString(PREF_KEY_SMART_ACTIONS, actionList).apply();
  }

  @Nullable
  String getSmartTriggerList() {
    //TODO move to provider cache
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(PREF_KEY_SMART_ACTIONS, null);
  }

  /**
   * <b>Preference key</b>. Stores the smart triggers event list. Each event
   * is separated by { EVENT_SEPERATOR}
   */
  static final String PREF_KEY_SMART_ACTIONS = "smart_actions";

  private static final String KEY_TRACK_LOCATION = "key_track_location";
  private static final String KEY_SET_GEO_FENCE = "key_set_geo_fence";

  public void optOutOfTrackLocation(boolean optOut) {
    configMap.put(KEY_TRACK_LOCATION, optOut);
  }

  public void optOutOfSetGeoFence(boolean optOut) {
    configMap.put(KEY_SET_GEO_FENCE, optOut);
  }

  public boolean isTrackLocationProhibited() {
    return configMap.containsKey(KEY_TRACK_LOCATION)
        && configMap.get(KEY_TRACK_LOCATION) == Boolean.TRUE;
  }

  public boolean isSetGeoFenceProhibited() {
    return configMap.containsKey(KEY_SET_GEO_FENCE)
        && configMap.get(KEY_SET_GEO_FENCE) == Boolean.TRUE;
  }

  static final String PREF_KEY_USER_LOGOUT = "logout";

  public boolean isUserLoggedOut() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getBoolean(PREF_KEY_USER_LOGOUT, false);
  }

  public void setUserLoggedOut(boolean logoutStatus) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putBoolean(PREF_KEY_USER_LOGOUT, logoutStatus).apply();
  }

  private final Object senderIdLock = new Object();

  public void setSenderId(String senderId) {
    synchronized (senderIdLock) {
      final SharedPreferences prefs = getSharedPrefs();
      Logger.d("ConfigurationProvider: SettingSenderID : " + senderId);
      SharedPreferences.Editor editor = prefs.edit();
      editor.putString(PREF_KEY_SENDER_ID, senderId);
      editor.apply();
    }
  }

  public String getStoredSenderId() {
    if (getSharedPrefs().contains(SENDER_ID)) {
      return getSharedPrefs().getString(SENDER_ID, null);
    }
    return null;
  }

  private final String KEY_OPTED_OUT_ACTIVITIES = "opted_out_activities";

  public List<String> getOptedOutActivities() {
    if (configMap.containsKey(KEY_OPTED_OUT_ACTIVITIES)) {
      return (List<String>) configMap.get(KEY_OPTED_OUT_ACTIVITIES);
    }
    return null;
  }

  void setOptedOutActivities(List<String> optedOutActivities) {
    configMap.put(KEY_OPTED_OUT_ACTIVITIES, optedOutActivities);
  }

  private void getOptedOutActivitiesFromManifest() {
    String OPTOUT_ACTIVITY_KEY = "OPT_OUT_TRACKING";
    List<String> optedOutActivities = new ArrayList<>();
    try {
      PackageManager pm = mContext.getPackageManager();
      PackageInfo info = pm.getPackageInfo(mContext.getPackageName(),
          PackageManager.GET_ACTIVITIES | PackageManager.GET_META_DATA);
      ActivityInfo[] activityInfoList = info.activities;
        for (ActivityInfo activityInfo : activityInfoList) {
          Bundle bundle = activityInfo.metaData;
          if (bundle != null) {
            if (bundle.containsKey(OPTOUT_ACTIVITY_KEY)) {
              optedOutActivities.add(activityInfo.name);
            }
          }
        }
      Logger.v("ConfigurationProvider : Opted out activities : " + optedOutActivities.toString());
      setOptedOutActivities(optedOutActivities);
    } catch (PackageManager.NameNotFoundException pe) {
      Logger.f("ConfigurationProvider#getOptedOutActivitiesFromManifest " + pe.getMessage());
    } catch (AndroidException ae) {
      Logger.f("ConfigurationProvider#getOptedOutActivitiesFromManifest " + ae.getMessage());
    } catch (Exception e) {
      Logger.f("ConfigurationProvider#getOptedOutActivitiesFromManifest " + e.getMessage());
    }
  }

  private static final String XML_APP_ID = "moe_app_id";
  private static final String XML_SENDER_ID = "moe_sender_id";
  private static final String XML_SKIP_GCM_REGISTRATION = "moe_skip_registration";
  private static final String XML_NOTIFICATION_ICON = "moe_small_icon";
  private static final String XML_NOTIFICATION_LARGE_ICON = "moe_large_icon";
  private static final String XML_NOTIFICATION_TONE = "moe_notification_tone";
  private static final String XML_NOTIFICATION_COLOR = "moe_notif_color";
  private static final String XML_NOTIFICATION_TYPE = "moe_notification_type";
  private static final String XML_BACKSTACK_OPT_OUT = "moe_backstack_opt_out";

/*  private void readSDKConfiguration() {
    if (!readXMLConfiguration()) {
      getSDKConfiguration();
    }
  }*/

/*  private boolean readXMLConfiguration() {
    int resourceId = getResourceId(XML_APP_ID, "string");
    if (resourceId != 0) {
      configMap.put(APP_ID, mContext.getString(resourceId));
    }
    resourceId = getResourceId(XML_SENDER_ID, "string");
    if (resourceId != 0) {
      configMap.put(SENDER_ID, mContext.getString(resourceId));
    }
    resourceId = getResourceId(XML_SKIP_GCM_REGISTRATION, "boolean");
    if (resourceId != 0) {
      configMap.put(SKIP_GCM_REGISTRATION, mContext.getResources().getBoolean(resourceId));
    }
    resourceId = getResourceId(XML_NOTIFICATION_ICON, "string");
    if (resourceId != 0) {
      String resourceName = mContext.getString(resourceId);
      int drawableId = getDrawable(resourceName);
      if (drawableId == 0 || !MoEHelperUtils.isValidResourceId(mContext, drawableId)) {
        drawableId = getAppIcon();
      }
      configMap.put(NOTIFICATION_ICON, drawableId);
    }
    resourceId = getResourceId(XML_NOTIFICATION_LARGE_ICON, "string");
    if (resourceId != 0) {
      String resourceName = mContext.getString(resourceId);
      int drawableId = getDrawable(resourceName);
      if (drawableId == 0 || !MoEHelperUtils.isValidResourceId(mContext, drawableId)) {
        drawableId = getAppIcon();
      }
      configMap.put(NOTIFICATION_LARGE_ICON, drawableId);
    }
    resourceId = getResourceId(XML_NOTIFICATION_TONE, "string");
    if (resourceId != 0) {
      String notificationTone = mContext.getString(resourceId);
      String tone = null;
      if (!TextUtils.isEmpty(notificationTone)) {
        if (notificationTone.contains("res")) {
          //String contents because @raw was used
          String[] resourcePathParts = notificationTone.split("/");
          String resName = resourcePathParts[resourcePathParts.length - 1];
          if (resName.contains(".")) {
            tone = resName.substring(resName.lastIndexOf("/") + 1, resName.lastIndexOf("."));
          } else {
            tone = resName.substring(resName.lastIndexOf("/") + 1);
          }
        } else {
          tone = notificationTone;
        }
      } else {
        tone = notificationTone;
      }
      configMap.put(NOTIFICATION_TONE, tone);
    }
    resourceId = getResourceId(XML_NOTIFICATION_COLOR, "color");
    if (resourceId != 0) {
      configMap.put(NOTIFICATION_COLOR, resourceId);
    }
    resourceId = getResourceId(XML_NOTIFICATION_TYPE, "integer");
    if (resourceId != 0) {
      configMap.put(NOTIFICATION_TYPE, mContext.getResources().getInteger(resourceId));
    } else {
      configMap.put(NOTIFICATION_TYPE, R.integer.notification_type_single);
    }
    resourceId = getResourceId(XML_BACKSTACK_OPT_OUT, "boolean");
    if (resourceId != 0) {
      configMap.put(BACK_STACK_OPT_OUT, mContext.getResources().getBoolean(resourceId));
    } else {
      configMap.put(BACK_STACK_OPT_OUT, false);
    }

    return !TextUtils.isEmpty((String) configMap.get(APP_ID));
  }*/

  private int getResourceId(String resourceName, String resourceType) {
    int resourceId = 0;
    switch (resourceType) {
      case "string":
        resourceId = mContext.getResources()
            .getIdentifier(resourceName, "string", mContext.getPackageName());
        break;
      case "boolean":
        resourceId = mContext.getResources()
            .getIdentifier(resourceName, "boolean", mContext.getPackageName());
        break;
      case "int":
        resourceId =
            mContext.getResources().getIdentifier(resourceName, "int", mContext.getPackageName());
        break;
      case "color":
        resourceId =
            mContext.getResources().getIdentifier(resourceName, "color", mContext.getPackageName());
        break;
    }
    return resourceId;
  }

  private static final String PREF_REG_FAIL_COUNT = "push_fail_count";

  public int getPushRegistrationFailureCount() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      int failCount = sp.getInt(PREF_REG_FAIL_COUNT, 0);
      sp.edit().putInt(PREF_REG_FAIL_COUNT, ++failCount).apply();
      return failCount;
    }
    return 1;
  }

  private void setPushFailureCount(int count) {
    final SharedPreferences prefs = getSharedPrefs();
    prefs.edit().putInt(PREF_REG_FAIL_COUNT, count).apply();
  }

  private static final String PREF_NAV_BAR_OPT_OUT = "opt_out_nav_bar";

  public void setNavBarOptOut(boolean hasNavBar) {
    final SharedPreferences prefs = getSharedPrefs();
    prefs.edit().putBoolean(PREF_NAV_BAR_OPT_OUT, hasNavBar).apply();
  }

  public boolean hasOptedOutNavBar() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getBoolean(PREF_NAV_BAR_OPT_OUT, false);
    }
    return false;
  }

  private static final String PREF_UNITY_SDK_VERSION = "unity_sdk_ver";

  public void setUnityVersion(String sdkVersion) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putString(PREF_UNITY_SDK_VERSION, sdkVersion).apply();
  }

  public String getUnityVersion() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getString(PREF_UNITY_SDK_VERSION, null);
    }
    return null;
  }

  private static final String PREF_LOG_ENTRY_KEY = "log_entry_key";

  public void setLogEntryKey(String logEntryKey) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      sp.edit().putString(PREF_LOG_ENTRY_KEY, logEntryKey).apply();
    }
  }

  public String getLogEntryKey() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getString(PREF_LOG_ENTRY_KEY, "031df6f2-907b-46a4-9654-440991e39380");
    }
    return "031df6f2-907b-46a4-9654-440991e39380";
  }

  private static final String PREF_LOG_ENTRY_ENABLED = "log_entry_enabled";

  public void setLogEntryEnable(boolean logStatus) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      sp.edit().putBoolean(PREF_LOG_ENTRY_ENABLED, logStatus).apply();
    }
  }

  public boolean isLogEntryEnabled() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getBoolean(PREF_LOG_ENTRY_ENABLED, false);
    }
    return false;
  }

  private static final String PREF_LAST_CONFIG_SYNC_TIME = "last_config_sync_time";

  public void setLastConfigSyncTime(long syncTime) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      sp.edit().putLong(PREF_LAST_CONFIG_SYNC_TIME, syncTime).apply();
    }
  }

  public long getLastConfigSyncTime() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getLong(PREF_LAST_CONFIG_SYNC_TIME, 0);
    }
    return 0;
  }

  private static final String PREF_SHOULD_ROUTE_TRAFFIC = "route_traffic";

  public void setRouteTraffic(boolean shouldRoute) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putBoolean(PREF_SHOULD_ROUTE_TRAFFIC, shouldRoute).apply();
  }

  public boolean shouldRouteTraffic() {
    SharedPreferences sp = getSharedPrefs();
    return sp != null && sp.getBoolean(PREF_SHOULD_ROUTE_TRAFFIC, false);
  }

  public String getBaiduApiKey() {
    return (String) configMap.get(BAIDU_API_KEY);
    //return String.valueOf(apiKey);
  }

  private static final String PREF_LAST_MESSAGE_FETCH_TIME = "last_message_sync";

  public void setLastMessageFetchTime(long time) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) sp.edit().putLong(PREF_LAST_MESSAGE_FETCH_TIME, time).commit();
  }

  public long getLastMessageFetchTime() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getLong(PREF_LAST_MESSAGE_FETCH_TIME, 0);
    }
    return 0;
  }

  private static final String PREF_MESSAGE_FETCH_TIME_DELAY = "last_message_sync_time_difference";

  public void setMessageFetchDelayDuration(long time) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) sp.edit().putLong(PREF_MESSAGE_FETCH_TIME_DELAY, time).apply();
  }

  public long getMessageFetchDelayDuration() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getLong(PREF_MESSAGE_FETCH_TIME_DELAY, 10800 * 1000);
    }
    return 10800 * 1000;
  }

  private static final String PREF_VERIFICATION_REGISTRATION_TIME = "verfication_registration_time";

  public void setVerificationRegistrationTime(long time) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) sp.edit().putLong(PREF_VERIFICATION_REGISTRATION_TIME, time).apply();
  }

  public long getVerificationRegistrationTime() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getLong(PREF_VERIFICATION_REGISTRATION_TIME, 0);
    }
    return 0;
  }

  private static final String PREF_HAS_REGISTERED_FOR_VERIFICATION =
      "has_registered_for_verification";

  public void setVerificationRegistration(boolean hasVerificationRegistered) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      sp.edit().putBoolean(PREF_HAS_REGISTERED_FOR_VERIFICATION, hasVerificationRegistered).apply();
    }
  }

  public boolean isDeviceRegisteredForVerification() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getBoolean(PREF_HAS_REGISTERED_FOR_VERIFICATION, false);
    }
    return false;
  }

  private static final String PREF_IS_SEGMENT_INTEGRATION = "is_segment_integration";

  public void setSegmentEnabledFlag(boolean isSegmentEnabled) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) sp.edit().putBoolean(PREF_IS_SEGMENT_INTEGRATION, isSegmentEnabled).apply();
  }

  public boolean isSegmentEnabled() {
    SharedPreferences sp = getSharedPrefs();
    return sp != null && sp.getBoolean(PREF_IS_SEGMENT_INTEGRATION, false);
  }

  private static final String PREF_KEY_ANDROID_ID_COLLECTION = "pref_key_android_id_collection";

  public void optOutOfAndroidIdCollection(boolean optOut) {
    configMap.put(PREF_KEY_ANDROID_ID_COLLECTION, optOut);
  }

  public boolean isAndroidIdCollectionProhibited() {
    return configMap.containsKey(PREF_KEY_ANDROID_ID_COLLECTION)
        && configMap.get(PREF_KEY_ANDROID_ID_COLLECTION) == Boolean.TRUE;
  }

  private static final String PREF_KEY_OPERATOR_NAME_COLLECTION =
      "pref_key_operator_name_collection";

  public void optOutOfOperatorNameCollection(boolean optOut) {
    configMap.put(PREF_KEY_OPERATOR_NAME_COLLECTION, optOut);
  }

  public boolean isOperatorNameCollectionProhibited() {
    return configMap.containsKey(PREF_KEY_OPERATOR_NAME_COLLECTION)
        && configMap.get(PREF_KEY_OPERATOR_NAME_COLLECTION) == Boolean.TRUE;
  }

  private static final String PREF_KEY_IMEI_COLLECTION = "pref_key_imei_collection";

  public void optOutOfIMEICollection(boolean optOut) {
    configMap.put(PREF_KEY_IMEI_COLLECTION, optOut);
  }

  public boolean isIMEICollectionProhibited() {
    return configMap.containsKey(PREF_KEY_IMEI_COLLECTION)
        && configMap.get(PREF_KEY_IMEI_COLLECTION) == Boolean.TRUE;
  }

  private static final String PREF_KEY_DEVICE_ATTRIBUTE_COLLECTION =
      "pref_key_device_attribute_collection";

  public void optOutOfDeviceAttributesCollection(boolean optOut) {
    configMap.put(PREF_KEY_DEVICE_ATTRIBUTE_COLLECTION, optOut);
  }

  public boolean isDeviceAttributesCollectionProhibited() {
    return configMap.containsKey(PREF_KEY_DEVICE_ATTRIBUTE_COLLECTION)
        && configMap.get(PREF_KEY_DEVICE_ATTRIBUTE_COLLECTION) == Boolean.TRUE;
  }

  private static final String PREF_KEY_ACTIVITY_SENT_LIST = "activity_sent_list";

  private void setSentScreenNames(String sentScreenNames){
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putString(PREF_KEY_ACTIVITY_SENT_LIST, sentScreenNames).apply();
  }

  private String getSentScreenNames(){
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(PREF_KEY_ACTIVITY_SENT_LIST, null);
  }

  public void addScreenToSentList(String screenName){
    sentScreenNames.add(screenName);
  }

  @Nullable
  public List<String> getSentScreenList(){
    return sentScreenNames;
  }

  public void storeSentScreenList(){
    try {
      StringBuilder sentScreenBuilder = new StringBuilder();
      int size = sentScreenNames.size();
      for (int i = 0; i<size; i++){
        sentScreenBuilder.append(sentScreenNames.get(i));
        if (i <= size -2){
          sentScreenBuilder.append(MoEHelperConstants.EVENT_SEPERATOR);
        }
      }
      setSentScreenNames(sentScreenBuilder.toString());
    }catch (Exception e){
      Logger.f("ConfigurationProvider: storeSentScreenList: ", e);
    }
  }

  private static final  String PREF_KEY_ENABLE_DEBUG_LOGS = "enable_logs";

  public void setDebugLogStatus(boolean status){
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putBoolean(PREF_KEY_ENABLE_DEBUG_LOGS, status).apply();
  }

  public boolean isDebugLogEnabled(){
    SharedPreferences sp = getSharedPrefs();
    return sp.getBoolean(PREF_KEY_ENABLE_DEBUG_LOGS, false);
  }

  private static final String PREF_KEY_BLACKLIST_EVENTS = "black_list_events";

  public void saveBlackListEventList(@NonNull String blackListEvents){
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putString(PREF_KEY_BLACKLIST_EVENTS, blackListEvents).apply();
  }

  @Nullable
  public String getBlackListEvents(){
    return getSharedPrefs().getString(PREF_KEY_BLACKLIST_EVENTS, null);
  }

  private static final String PREF_KEY_APP_STATUS = "app_status";

  public void saveAppState(boolean status) {
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putBoolean(PREF_KEY_APP_STATUS, status).apply();
  }

  public boolean isAppEnabled() {
    return getSharedPrefs().getBoolean(PREF_KEY_APP_STATUS, true);
  }

  private static final String PREF_KEY_IN_APP_STATUS = "inapp_status";

  public void saveInAppState(boolean status) {
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putBoolean(PREF_KEY_IN_APP_STATUS, status).apply();
  }

  public boolean isInAppEnabled() {
    return getSharedPrefs().getBoolean(PREF_KEY_IN_APP_STATUS, true);
  }

  private static final String PREF_KEY_GEO_STATE = "geo_state";

  public void saveGeoState(boolean status) {
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putBoolean(PREF_KEY_GEO_STATE, status).apply();
  }

  public boolean isGeoEnabled() {
    return getSharedPrefs().getBoolean(PREF_KEY_GEO_STATE, true);
  }

  private static final String PREF_KEY_INBOX_STATE = "inbox_state";

  public void saveInboxState(boolean status) {
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putBoolean(PREF_KEY_INBOX_STATE, status).apply();
  }

  public boolean isInboxEnabled() {
    return getSharedPrefs().getBoolean(PREF_KEY_INBOX_STATE, true);
  }

  private static final String PREF_KEY_EVENT_BATCH_COUNT = "event_count";

  public void saveEventBatchCount(int count) {
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putInt(PREF_KEY_EVENT_BATCH_COUNT, count).apply();
  }

  public int getEventBatchCount() {
    return getSharedPrefs().getInt(PREF_KEY_EVENT_BATCH_COUNT, 30);
  }

  private static final String PREF_KEY_SEGMENT_ANONYMOUS_ID = "segment_anonymous_id";

  public void saveSegmentAnonymousId(String anonymousId) {
    getSharedPrefs().edit().putString(PREF_KEY_SEGMENT_ANONYMOUS_ID, anonymousId).apply();
  }

  @Nullable public String getSegmentAnonymousId() {
    return getSharedPrefs().getString(PREF_KEY_SEGMENT_ANONYMOUS_ID, null);
  }

  private static final String PREF_KEY_DATA_REGION = "data_region";

  public void setDataRegion(int region){
    getSharedPrefs().edit().putInt(PREF_KEY_DATA_REGION, region).apply();
  }

  public int getDataRegion(){
    return getSharedPrefs().getInt(PREF_KEY_DATA_REGION, -999);
  }

  private static final String PREF_KEY_RETRY_SYNC_TIME = "retry_sync_time";

  void setRetrySyncTime(long retrySyncTime){
    getSharedPrefs().edit().putLong(PREF_KEY_RETRY_SYNC_TIME, retrySyncTime).apply();
  }

  long getRetrySyncTime(){
    return getSharedPrefs().getLong(PREF_KEY_RETRY_SYNC_TIME, 1800 * 1000);
  }

  private static final String PREF_KEY_IMMEDIATE_RETRY_SYNC_COUNT = "retry_count";

  void setImmediateRetryCount(int count){
    getSharedPrefs().edit().putInt(PREF_KEY_IMMEDIATE_RETRY_SYNC_COUNT, count).apply();
  }

  int getImmediateRetryCount(){
    return getSharedPrefs().getInt(PREF_KEY_IMMEDIATE_RETRY_SYNC_COUNT, 0);
  }

  private static final String PREF_KEY_FLUSH_EVENTS = "flush_events";

  void saveFlushEventList(@NonNull String flushEvents){
    getSharedPrefs().edit().putString(PREF_KEY_FLUSH_EVENTS, flushEvents).apply();
  }

  @Nullable
  String getFlushEvents(){
    return getSharedPrefs().getString(PREF_KEY_FLUSH_EVENTS, null);
  }

  private static final String PREF_KEY_PERIODIC_FLUSH_STATE = "periodic_flush_state";

  public void savePeriodicFlushState(boolean state){
    getSharedPrefs().edit().putBoolean(PREF_KEY_PERIODIC_FLUSH_STATE, state).apply();
  }

  boolean isPeriodicFlushEnabled(){
    return getSharedPrefs().getBoolean(PREF_KEY_PERIODIC_FLUSH_STATE, true);
  }

  private static final String PREF_KEY_PERIODIC_FLUSH_TIME = "periodic_flush_time";

  public void savePeriodicFlushTime(long time) {
    getSharedPrefs().edit().putLong(PREF_KEY_PERIODIC_FLUSH_TIME, time).apply();
  }

  public long getPeriodicFlushTime() {
    return getSharedPrefs().getLong(PREF_KEY_PERIODIC_FLUSH_TIME, 60);
  }

  private static final String PREF_KEY_CAMPAIGN_ID_TTL = "cid_ttl";

  void saveCampaignIdTTL(long ttl){
    getSharedPrefs().edit().putLong(PREF_KEY_CAMPAIGN_ID_TTL, ttl).apply();
  }

  public long getCampaignIdTTL(){
    return getSharedPrefs().getLong(PREF_KEY_CAMPAIGN_ID_TTL, 2419200000L);
  }
}
