/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.firebase.internal

import android.content.Context
import android.os.Looper
import com.moe.pushlibrary.MoEHelper
import com.moe.pushlibrary.utils.MoEHelperConstants
import com.moengage.core.*
import com.moengage.firebase.MoEFireBaseHelper
import com.moengage.push.PushManager
import org.json.JSONObject

/**
 * @author Arshiya Khanum
 * Date: 2020/07/18
 */

internal object TokenHandler {

    private const val tag = "${MODULE_TAG}TokenHandler"
    private const val ATTR_REGISTRATION_BY = "registered_by"
    private val lock = Any()

    /**
     * Process Push Token
     *
     * @param context instance of [Context]
     * @param pushToken Push Token
     * @param pushRegisteredBy Push registration is handled by App or MoEngage
     */
    @Suppress("SENSELESS_COMPARISON")
    fun processToken(context: Context, pushToken: String?, pushRegisteredBy: String) {
        if (pushToken == null || pushToken.trim().isEmpty() || context == null) return
        Logger.v("$tag processToken() : Will try to process push token. Token: $pushToken registered by: $pushRegisteredBy")
        try {
            synchronized(lock) {
                val token = ripMultiplexingExtras(pushToken)
                notifyListeners(pushToken)
                val repository = Injection.getRepository(context)
                val oldRegId = repository.getPushToken()
                val update = tokenRefreshRequired(token, oldRegId)
                if (update) {
                    repository.savePushToken(token)
                    MoEDispatcher.getInstance(context).deviceAddManager.registerFcmToken(context)
                    trackTokenGeneration(pushRegisteredBy, context)
                }

                Logger.v(
                    "$tag processToken() oldId: = $oldRegId token = $token " +
                            "--updating[true/false]: $update"
                )
            }
        } catch (e: Exception) {
            Logger.e("$tag processToken() : Exception ", e)
        }
    }

    private fun trackTokenGeneration(pushRegisteredBy: String, context: Context) {
        val properties = Properties()
        properties.addAttribute(ATTR_REGISTRATION_BY, pushRegisteredBy)
        properties.setNonInteractive()
        MoEHelper.getInstance(context).trackEvent(MoEHelperConstants.TOKEN_EVENT, properties)
        trackDeviceAttributeForRegistration(context, pushRegisteredBy)
    }

    private fun notifyListeners(token: String) {
        PushManager.getInstance().notifyTokenChange(token)
        val handler = android.os.Handler(Looper.getMainLooper())
        handler.post {
            try {
                MoEFireBaseHelper.getInstance().getEventListener()?.onTokenAvailable(token)
            } catch (ex: Exception) {
                Logger.e("$tag notifyListeners: Exception: ", ex)
            }
        }
    }

    private const val ID_PREFIX = "|ID|"

    private fun ripMultiplexingExtras(token: String): String {
        return if (!MoEUtils.isEmptyString(token) && token.startsWith(ID_PREFIX))
            token.substring(7)
        else token
    }

    private fun tokenRefreshRequired(newToken: String, oldToken: String?): Boolean {
        return if (!MoEUtils.isEmptyString(newToken)) {
            MoEUtils.isEmptyString(oldToken) || !newToken.equals(oldToken)
        } else {
            false
        }
    }

    private fun trackDeviceAttributeForRegistration(context: Context, pushRegisteredBy: String) {
        try {
            val attribute = JSONObject()
            attribute.put(MoEConstants.PUSH_REGISTRATION_ATTRIBUTE, pushRegisteredBy)
            MoEDispatcher.getInstance(context).setDeviceAttribute(attribute)
        } catch (e: Exception) {
            Logger.e("$tag trackDeviceAttributeForRegistration() : ", e)
        }
    }
}