/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.firebase.internal

import android.content.Context
import android.os.Looper
import androidx.annotation.WorkerThread
import com.google.android.gms.tasks.OnCompleteListener
import com.google.firebase.iid.FirebaseInstanceId
import com.google.firebase.messaging.RemoteMessage
import com.moe.pushlibrary.MoEHelper
import com.moengage.core.*
import com.moengage.core.executor.TaskManager
import com.moengage.core.listeners.AppBackgroundListener
import com.moengage.firebase.MoEFireBaseHelper
import java.io.IOException
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.TimeUnit

/**
 * @author Umang Chamaria
 * Date: 27/03/20
 */
internal class FcmController private constructor() : AppBackgroundListener {

    private val tag = "${MODULE_TAG}FcmController"

    private var scheduler: ScheduledExecutorService? = null

    fun getPushToken(context: Context) {
        try {
            Logger.v("$tag getPushToken() : Will try to register for push.")
            if (!shouldRegisterForPush(context)) return
            if (!MoEUtils.isEmptyString(SdkConfig.getConfig().pushConfig.fcm.senderId)) {
                Logger.v("$tag getPushToken() : Sender-id is set by the client, will use it for push token registration.")
                TaskManager.getInstance().execute {
                    processPushTokenForSenderId(context)
                }
                return
            }
            Logger.v("$tag getPushToken() : Regular app registration.")
            FirebaseInstanceId.getInstance().instanceId.addOnCompleteListener(OnCompleteListener { task ->
                try {
                    if (!task.isSuccessful) {
                        Logger.e(
                            "$tag onComplete() : Task<InstanceIdResult> failed. ",
                            task.exception
                        )
                        scheduleTokenRegistrationRetry(context)
                        return@OnCompleteListener
                    }
                    val token = task.result?.token
                    if (MoEUtils.isEmptyString(token)) {
                        scheduleTokenRegistrationRetry(context)
                        return@OnCompleteListener
                    }
                    TokenHandler.processToken(
                        context,
                        token,
                        MoEConstants.ATTRIBUTE_TOKEN_REGISTERED_MOE
                    )
                } catch (e: Exception) {
                    Logger.e("$tag onComplete() : ", e)
                    scheduleTokenRegistrationRetry(context)
                }
            })
        } catch (e: Exception) {
            Logger.e("$tag getPushToken() : ", e)
        }
    }

    @Throws(IOException::class)
    @WorkerThread
    private fun processPushTokenForSenderId(context: Context) {
        try {
            val token = SdkConfig.getConfig().pushConfig.fcm.senderId?.let { senderId ->
                FirebaseInstanceId.getInstance().getToken(senderId, "FCM")
            }
            if (MoEUtils.isEmptyString(token)) {
                Logger.v("$tag processPushTokenForSenderId() : Empty token returned. Scheduling a" +
                        " retry.")
                scheduleTokenRegistrationRetry(context)
                return
            }
            Logger.v("$tag processPushTokenForSenderId() : Token: $token")
            TokenHandler.processToken(context, token, MoEConstants.ATTRIBUTE_TOKEN_REGISTERED_MOE)
        } catch (e: Exception) {
            Logger.e("$tag processPushTokenForSenderId() : ", e)
            scheduleTokenRegistrationRetry(context)
        }
    }

    private fun scheduleTokenRegistrationRetry(context: Context) {
        if (MoEHelper.isAppInBackground()) return
        Logger.v(
            "$tag scheduleTokenRegistrationRetry() : Will schedule token registration retry."
        )
        if (scheduler == null || scheduler?.isShutdown == true) {
            scheduler = Executors.newScheduledThreadPool(1)
        }
        val runnable = Runnable {
            Logger.v("$tag run() : Will try attempt to register for token.")
            getPushToken(context)
        }
        scheduler?.schedule(
            runnable, SdkConfig.getConfig().pushConfig.tokenRetryInterval,
            TimeUnit.SECONDS
        )
    }

    override fun onAppBackground(context: Context) {
        try {
            Logger.v("$tag goingToBackground() : Will shutdown scheduler.")
            if (scheduler != null && (scheduler?.isShutdown == false)) {
                scheduler?.shutdownNow()
            }
        } catch (e: Exception) {
            Logger.e("$tag goingToBackground() : ", e)
        }
    }

    /**
     * This method is invoked whenever a new push message is received via callback from
     * [MoEFireBaseMessagingService.onMessageReceived]
     *
     * @param message Push Message [com.google.firebase.messaging.RemoteMessage]
     */
    fun onNonMoEngagePushReceived(
        context: Context,
        message: RemoteMessage
    ) {
        val handler = android.os.Handler(Looper.getMainLooper())

        handler.post {
            try {
                MoEFireBaseHelper.getInstance().getEventListener()
                    ?.onNonMoEngageMessageReceived(message)
            } catch (e: Exception) {
                Logger.e("$tag notifyListeners: Exception: ", e)
            }
        }
        //Note: This callback will be removed
        MoEFireBaseHelper.getInstance()
            .getNonMoEngagePushReceivedListener()?.onPushReceived(context, message)
    }

    /**
     * This method is invoked whenever a new token is generated via callback from
     * [MoEFireBaseMessagingService.onNewToken]
     *
     * @param context Application Context
     * @param token Push token
     */
    internal fun onNewToken(context: Context, token: String?) {
        try {
            if (!shouldRegisterForPush(context)) {
                Logger.v("$tag onNewToken() : Ignoring token as MoEngage is not registering for " +
                        "push")
                return
            }
            if (token.isNullOrBlank()) {
                Logger.v("$tag onNewToken() : Generated token is empty, returning")
                return
            } else {
                TokenHandler.processToken(
                    context,
                    token,
                    MoEConstants.ATTRIBUTE_TOKEN_REGISTERED_MOE
                )
            }
        } catch (e: Exception) {
            Logger.e("$tag onNewToken() : ", e)
        }
    }

    private fun shouldRegisterForPush(context: Context): Boolean {
        return SdkConfig.getConfig().pushConfig.fcm.isRegistrationEnabled
                && !Injection.getRepository(context).isPushNotificationOptedOut()
    }

    companion object {
        private var instance: FcmController? = null

        fun getInstance(): FcmController {
            if (instance == null) {
                synchronized(FcmController::class.java) {
                    if (instance == null) instance = FcmController()
                }
            }
            return instance as FcmController
        }
    }

    init {
        MoECallbacks.getInstance().addAppBackgroundListener(this)
    }
}