/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.firebase

import android.content.Context
import android.os.Bundle
import com.moengage.core.Logger
import com.moengage.core.MoEConstants
import com.moengage.core.MoEUtils
import com.moengage.firebase.internal.FcmController
import com.moengage.firebase.internal.Injection
import com.moengage.firebase.internal.MODULE_TAG
import com.moengage.firebase.internal.TokenHandler
import com.moengage.firebase.listener.FirebaseEventListener
import com.moengage.firebase.listener.OnNonMoEngagePushReceivedListener
import com.moengage.pushbase.MoEPushHelper

/**
 * Helper class for MoEngage Firebase Module.
 *
 * @author Umang Chamaria
 * Date: 13/03/19
 * @since 9.4.00
 */
class MoEFireBaseHelper private constructor() {

    private val tag = "${MODULE_TAG}MoEFireBaseHelper"

    private var listener: OnNonMoEngagePushReceivedListener? = null
    private var eventListener: FirebaseEventListener? = null

    @Suppress("SENSELESS_COMPARISON")
    @Deprecated(
        "This api will be removed.",
        ReplaceWith(
            "MoEFireBaseHelper.getInstance().setEventListener()",
            "com.moengage.firebase.listener.FirebaseEventListener"
        ),
        DeprecationLevel.WARNING
    )
    fun setOnNonMoEngagePushReceivedListener(pushReceivedListener:
                                             OnNonMoEngagePushReceivedListener) {
        try {
            if (pushReceivedListener == null) return
            this.listener = pushReceivedListener
        } catch (ex: Exception) {
            Logger.e("$tag setOnNonMoEngagePushReceivedListener() : Exception: ", ex)
        }
    }

    internal fun getNonMoEngagePushReceivedListener(): OnNonMoEngagePushReceivedListener? {
        return this.listener
    }

    @Suppress("SENSELESS_COMPARISON")
    fun setEventListener(listener: FirebaseEventListener) {
        try {
            if (listener == null) return
            eventListener = listener
        } catch (ex: Exception) {
            Logger.e("$tag setEventListener() : Exception: ", ex)
        }
    }

    internal fun getEventListener(): FirebaseEventListener? {
        return eventListener
    }

    /**
     * Helper API to pass the push token to the MoEngage platform.
     *
     * @param context instance of [Context]
     * @param token Generated Push Token.
     * @since 10.3.00
     */
    @Suppress("SENSELESS_COMPARISON")
    fun passPushToken(context: Context, token: String) {
        try {
            if (context == null || MoEUtils.isEmptyString(token)){
                Logger.w("$tag passPushToken() : Either Context is null or Token is empty/null.")
                return
            }
            if (Injection.getRepository(context).isPushNotificationOptedOut()) {
                Logger.w("$tag passPushToken() : Push notification opted out rejecting token.")
                return
            }
            TokenHandler.processToken(context, token, MoEConstants.ATTRIBUTE_TOKEN_REGISTERED_APP)
        } catch (ex: Exception) {
            Logger.e("$tag passPushToken() : Exception: ", ex)
        }
    }

    /**
     * Pass the payload received from FCM to the MoEngage SDK for showing Push Notification.
     * @param context instance of [Context]
     * @param payload instance of [Bundle]
     */
    @Suppress("SENSELESS_COMPARISON")
    fun passPushPayload(context: Context, payload: Bundle) {
        try {
            if (context == null || payload == null) return
            MoEPushHelper.getInstance().handlePushPayload(context, payload)
        } catch (ex: Exception) {
            Logger.e("$tag passPushPayload() : Exception: ", ex)
        }
    }

    /**
     * Pass the payload received from FCM to the MoEngage SDK for showing Push Notification.
     * @param context instance of [Context]
     * @param payload instance of [Map]
     */
    @Suppress("SENSELESS_COMPARISON")
    fun passPushPayload(context: Context, payload: Map<String, String>) {
        try {
            if (context == null || payload == null) return
            MoEPushHelper.getInstance().handlePushPayload(context, payload)
        } catch (ex: Exception) {
            Logger.e("$tag passPushPayload() : Exception: ", ex)
        }
    }

    /**
     * API to register for FCM Token. Call this API only if you are delaying the SDK
     * initialisation else this is not required.
     *
     * <b>Note:</b> If the application has opted out of token registration this API will not
     * register a push token.
     *
     * @param context instance of [Context]
     * @since 10.3.00
     */
    @Suppress("SENSELESS_COMPARISON")
    fun registerForToken(context: Context) {
        try {
            if (context == null) return
            Logger.v("$tag registerForToken() : Will try to register for push.")
            FcmController.getInstance().getPushToken(context)
        } catch (e: Exception) {
            Logger.e("$tag registerForToken() : ", e)
        }
    }

    companion object {
        private var instance: MoEFireBaseHelper? = null

        @JvmStatic
        fun getInstance(): MoEFireBaseHelper {
            if (instance == null) {
                synchronized(MoEFireBaseHelper::class.java) {
                    if (instance == null) instance = MoEFireBaseHelper()
                }
            }

            return instance as MoEFireBaseHelper
        }
    }
}