/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.geofence.internal.repository;

import androidx.annotation.Nullable;
import com.google.android.gms.location.Geofence;
import com.moe.pushlibrary.models.GeoLocation;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.rest.Response;
import com.moengage.geofence.internal.LocationConstants;
import com.moengage.geofence.internal.model.GeoCampaign;
import com.moengage.geofence.internal.model.GeofenceFetchResponse;
import com.moengage.geofence.internal.model.GeofenceHitResponse;
import java.util.ArrayList;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2020-02-04
 */
public class ResponseParser {

  private static final String TAG = "ResponseParser";

  GeofenceFetchResponse parseFetchResponse(Response response) {
    try {
      if (response == null) return new GeofenceFetchResponse(false);
      if (response.responseCode != 200) return new GeofenceFetchResponse(false);
      if (!isValidAPIResponse(response.responseBody)) return new GeofenceFetchResponse(false);
      JSONObject responseJson = new JSONObject(response.responseBody);
      JSONArray fenceInfo = responseJson.getJSONArray(FENCES_INFO);
      if (fenceInfo.length() == 0) {
        return new GeofenceFetchResponse(true,
            new ArrayList<GeoCampaign>());
      }
      List<GeoCampaign> campaignList = new ArrayList<>();
      for (int i = 0; i < fenceInfo.length(); i++) {
        JSONObject campaignJson = fenceInfo.getJSONObject(i);
        GeoCampaign campaign = campaignFromJson(campaignJson);
        if (campaign != null) {
          campaignList.add(campaign);
        }
      }
      return new GeofenceFetchResponse(true, campaignList);
    } catch (Exception e) {
      Logger.e(TAG + " parseFetchResponse() : ", e);
    }
    return new GeofenceFetchResponse(false);
  }

  @Nullable GeoCampaign campaignFromJson(JSONObject campaignJson) {
    try {
      GeoCampaign campaign = new GeoCampaign(
          getTransitionTypeFromString(campaignJson.getString(TRANSITION_TYPE)),
          new GeoLocation(campaignJson.getDouble(LATITUDE), campaignJson.getDouble(LONGITUDE)),
          (float) campaignJson.getDouble(DISTANCE),
          campaignJson.optInt(EXPIRY, -1),
          campaignJson.optInt(LOITERING_DELAY, -1),
          campaignJson.optInt(RESPONSIVENESS),
          campaignJson.getString(GEOFENCE_ID),
          campaignJson.optString(CAMPAIGN_ID),
          campaignJson.getString(GEOFENCE_ID)
              + LocationConstants.GEO_ID_SEPARATOR
              + campaignJson.optString(CAMPAIGN_ID)
      );
      if (campaign.transitionType == Geofence.GEOFENCE_TRANSITION_DWELL
          && campaign.loiteringDelay == -1) {
        return null;
      }
      return campaign;
    } catch (Exception e) {
      Logger.e(TAG + " campaignFromJson() : ", e);
    }
    return null;
  }

  private int getTransitionTypeFromString(String transitionString) throws IllegalArgumentException {
    switch (transitionString) {
      case LocationConstants.TRANSITION_ENTER:
        return Geofence.GEOFENCE_TRANSITION_ENTER;
      case LocationConstants.TRANSITION_DWELL:
        return Geofence.GEOFENCE_TRANSITION_DWELL;
      case LocationConstants.TRANSITION_EXIT:
        return Geofence.GEOFENCE_TRANSITION_EXIT;
      default:
        throw new IllegalArgumentException("Not a valid transition type");
    }
  }

  boolean isValidAPIResponse(String response) throws JSONException {
    if (MoEUtils.isEmptyString(response)) return false;
    JSONObject jsonResponse = new JSONObject(response);
    String result = jsonResponse.getString(RESPONSE_ATTR_RESULT);
    return RESULT_OK.equals(result);
  }

  GeofenceHitResponse parseHitResponse(Response response) {
    try {
      if (response == null) return new GeofenceHitResponse(false);
      if (response.responseCode != 200) return new GeofenceHitResponse(false);
      return new GeofenceHitResponse(isValidAPIResponse(response.responseBody));
    } catch (Exception e) {
      Logger.e( TAG + " parseHitResponse() : ", e);
    }
    return new GeofenceHitResponse(false);
  }

  private static final String DISTANCE = "distance";
  private static final String CAMPAIGN_ID = "cid";
  private static final String TRANSITION_TYPE = "transitionType";
  private static final String RESPONSIVENESS = "responsiveness_time";
  private static final String LATITUDE = "lat";
  private static final String LONGITUDE = "lng";
  private static final String GEOFENCE_ID = "geoId";
  private static final String LOITERING_DELAY = "ldelay";
  private static final String EXPIRY = "expiry";
  private static final String FENCES_INFO = "fencesInfo";

  static final String RESPONSE_ATTR_RESULT = "result";

  static final String RESULT_OK = "OK";
}
