/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.geofence.internal.repository;

import com.moe.pushlibrary.models.GeoLocation;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.geofence.internal.model.GeoCampaign;
import com.moengage.geofence.internal.model.GeofenceFetchRequest;
import com.moengage.geofence.internal.model.GeofenceFetchResponse;
import com.moengage.geofence.internal.model.GeofenceHitRequest;
import java.util.List;
import org.json.JSONException;

/**
 * @author Umang Chamaria
 * Date: 2020-02-04
 */
public class LocationRepository {

  private static final String TAG = "LocationRepository";

  private LocalRepository localRepository;
  private RemoteRepository remoteRepository;

  public LocationRepository(LocalRepository localRepository,
      RemoteRepository remoteRepository) {
    this.localRepository = localRepository;
    this.remoteRepository = remoteRepository;
  }

  public GeofenceFetchResponse fetchGeofence(GeoLocation lastKnownLocation, boolean isForeground)
      throws JSONException {
    GeofenceFetchRequest request = new GeofenceFetchRequest(localRepository.baseRequest(),
        lastKnownLocation, isForeground);
    GeofenceFetchResponse response = remoteRepository.fetchGeofence(request);

    Logger.v(TAG + " fetchGeofence() : Api response: " + response);
    if (!response.isSuccess) return response;

    localRepository.saveLastSyncTime(MoEUtils.currentMillis());
    return response;
  }

  public void geofenceHit(GeoLocation triggeringLocation, String geoId,
      String transitionType, boolean isForeground) {
    try {
      GeofenceHitRequest request = new GeofenceHitRequest(localRepository.baseRequest(),
          isForeground, triggeringLocation, transitionType, geoId, localRepository.getPushId());
      remoteRepository.geofenceHit(request);
    } catch (Exception e) {
      Logger.e(TAG + " geofenceHit() : ", e);
    }
  }

  public List<String> getSavedGeoIds() {
    return localRepository.getSavedGeoIds();
  }

  public long getLastSyncTime() {
    return localRepository.getLastSyncTime();
  }

  public void saveLocation(GeoLocation geoLocation) {
    localRepository.saveLocation(geoLocation);
  }

  public GeoLocation getLastLocation() {
    return localRepository.getSavedLocation();
  }

  public void saveRequestIdList(List<GeoCampaign> campaignList) {
    if (campaignList == null || campaignList.isEmpty()) return;
    localRepository.saveRequestIdList(campaignList);
  }
}
