/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.geofence.internal.repository;

import android.content.Context;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.MoEUtils;
import com.moengage.core.model.BaseRequest;
import com.moengage.core.RestUtils;
import com.moengage.geofence.internal.model.GeoCampaign;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.json.JSONException;

/**
 * @author Umang Chamaria
 * Date: 2020-02-04
 */
public class LocalRepository {

  private Context context;

  public LocalRepository(Context context) {
    this.context = context;
  }

  void saveLastSyncTime(long lastSyncTime) {
    ConfigurationProvider.getInstance(context).setLastGeoFenceSyncTime(lastSyncTime);
  }

  void saveLocation(GeoLocation geoLocation) {
    ConfigurationProvider.getInstance(context).saveLocation(geoLocation);
  }

  void saveRequestIdList(List<GeoCampaign> geofenceMetaList) {
    StringBuilder geoIDBuilder = new StringBuilder();
    int size = geofenceMetaList.size();
    for (int index = 0; index < size; index++) {
      GeoCampaign meta = geofenceMetaList.get(index);
      geoIDBuilder.append(meta.requestId);
      geoIDBuilder.append(MoEHelperConstants.EVENT_SEPERATOR);
    }
    ConfigurationProvider.getInstance(context).saveGeoIDList(geoIDBuilder.toString());
  }

  BaseRequest baseRequest() throws JSONException {
    return RestUtils.getBaseRequest(context);
  }

  List<String> getSavedGeoIds() {
    String geoList = ConfigurationProvider.getInstance(context).getGeoIDList();
    List<String> oldGeoFenceList = null;
    if (MoEUtils.isEmptyString(geoList)) return new ArrayList<>();

    if (geoList.contains(MoEHelperConstants.EVENT_SEPERATOR)) {
      String[] geoIds = geoList.split(MoEHelperConstants.EVENT_SEPERATOR);
      oldGeoFenceList = Arrays.asList(geoIds);
    } else {
      //only one geo fence found
      oldGeoFenceList = new ArrayList<>();
      oldGeoFenceList.add(geoList);
    }

    return oldGeoFenceList;
  }

  String getPushId(){
    return ConfigurationProvider.getInstance(context).getFcmToken();
  }

  long getLastSyncTime(){
    return ConfigurationProvider.getInstance(context).getLastGeoFenceSyncTime();
  }

  GeoLocation getSavedLocation(){
    return ConfigurationProvider.getInstance(context).getSavedLocation();
  }
}
