/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.geofence.internal.repository;

import android.net.Uri;
import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.rest.RequestBuilder;
import com.moengage.core.rest.RequestBuilder.RequestType;
import com.moengage.core.rest.Response;
import com.moengage.core.rest.RestClient;
import com.moengage.core.utils.JsonBuilder;
import com.moengage.core.RestUtils;
import com.moengage.geofence.internal.LocationConstants;
import com.moengage.geofence.internal.model.GeofenceFetchRequest;
import com.moengage.geofence.internal.model.GeofenceHitRequest;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2020-02-04
 */
public class ApiManager {

  private static final String TAG = LocationConstants.MODULE_TAG + "ApiManager";

  @Nullable Response fetchGeofence(GeofenceFetchRequest request) {
    try {
      Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
          .appendEncodedPath(GEOFENCE_FETCH);

      RequestBuilder requestBuilder =
          RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST,
              request.appId);
      JsonBuilder requestParams = request.defaultParams;
      requestParams.putString(LATITUDE, String.valueOf(request.location.latitude))
          .putString(LONGITUDE, String.valueOf(request.location.longitude))
          .putBoolean(IS_FOREGROUND, request.isForeground);

      JSONObject jsonBody = new JSONObject();
      jsonBody.put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, requestParams.build());
      requestBuilder.addBody(jsonBody);

      return new RestClient(requestBuilder.build()).executeRequest();
    } catch (Exception e) {
      Logger.e(TAG + " fetchGeofence() : ", e);
    }
    return null;
  }

  @Nullable Response geofenceHit(GeofenceHitRequest request) {
    try {
      Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
          .appendEncodedPath(GEOFENCE_HIT);

      RequestBuilder requestBuilder =
          RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST,
              request.appId);

      JsonBuilder requestParams = request.defaultParams;
      requestParams.putString(CURRENT_LATITUDE, String.valueOf(request.triggeringLocation.latitude))
          .putString(CURRENT_LONGITUDE, String.valueOf(request.triggeringLocation.longitude))
          .putString(GEO_ID, request.geoId)
          .putBoolean(IS_FOREGROUND, request.isForeground)
          .putString(TRANSITION_TYPE, request.transitionType)
          .putString(PUSH_ID, request.pushId);

      JSONObject jsonBody = new JSONObject();
      jsonBody.put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, requestParams.build());
      requestBuilder.addBody(jsonBody);

      return new RestClient(requestBuilder.build()).executeRequest();
    } catch (Exception e) {
      Logger.e(TAG + " geofenceHit() : ", e);
    }
    return null;
  }

  private static final String GEOFENCE_FETCH = "v1/geoFences";
  private static final String GEOFENCE_HIT = "v1/geoFenceHit";
  // params geofence fetch
  private static final String LATITUDE = "lat";
  private static final String LONGITUDE = "lng";
  private static final String IS_FOREGROUND = "isForeground";
  // params geofence hit
  private static final String CURRENT_LATITUDE = "curr_lat";
  private static final String CURRENT_LONGITUDE = "curr_long";
  private static final String GEO_ID = "geoIds";
  private static final String PUSH_ID = "push_id";
  private static final String TRANSITION_TYPE = "transitionType";
}
