/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.geofence.internal;

import android.Manifest;
import android.Manifest.permission;
import android.annotation.SuppressLint;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.os.Build;
import android.os.Build.VERSION_CODES;
import android.text.TextUtils;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.Geofence.Builder;
import com.google.android.gms.location.GeofenceStatusCodes;
import com.google.android.gms.location.GeofencingEvent;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.android.gms.tasks.Task;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.core.SdkConfig;
import com.moengage.core.model.MoEJobParameters;
import com.moengage.geofence.MoEGeofenceHelper;
import com.moengage.geofence.listener.OnGeofenceHitListener;
import com.moengage.geofence.internal.model.GeoCampaign;
import com.moengage.geofence.internal.repository.LocationRepository;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Umang Chamaria
 * Date: 2020-02-04
 */
public class LocationController {

  private static final String TAG = LocationConstants.MODULE_TAG + "LocationController";

  public boolean isSynced;

  private Context context;

  private static LocationController instance;

  private LocationController(Context context) {
    this.context = context;
  }

  public static LocationController getInstance(Context context) {
    if (instance == null) {
      synchronized (LocationController.class) {
        if (instance == null) {
          instance = new LocationController(context);
        }
      }
    }
    return instance;
  }

  void setSynced(boolean synced) {
    isSynced = synced;
  }

  @SuppressWarnings("MissingPermission")
  void setGeofence(Context context, List<GeoCampaign> campaignList) {
    try {
      if (campaignList.isEmpty()) return;
      List<Geofence> geofenceList = new ArrayList<>(campaignList.size());
      for (GeoCampaign campaign : campaignList) {
        Geofence.Builder builder = new Builder();
        builder.setCircularRegion(campaign.location.latitude, campaign.location.longitude,
            campaign.radius)
            .setRequestId(campaign.requestId)
            .setTransitionTypes(campaign.transitionType);
        builder.setExpirationDuration(campaign.expiryDuration);
        if (campaign.loiteringDelay != -1) {
          builder.setLoiteringDelay(campaign.loiteringDelay);
        }
        if (campaign.responsiveness != -1) {
          builder.setNotificationResponsiveness(campaign.responsiveness);
        }
        geofenceList.add(builder.build());
      }

      removeGeofence();

      Intent intent = new Intent(context, GeoFenceBroadcastReceiver.class);
      PendingIntent geoPending =
          PendingIntent.getBroadcast(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
      GeofencingRequest.Builder builder = new GeofencingRequest.Builder();
      builder.addGeofences(geofenceList)
          .setInitialTrigger(
              Geofence.GEOFENCE_TRANSITION_DWELL | Geofence.GEOFENCE_TRANSITION_ENTER);

      LocationServices.getGeofencingClient(context).addGeofences(builder.build(), geoPending)
          .addOnSuccessListener(new OnSuccessListener<Void>() {
            @Override public void onSuccess(Void aVoid) {
              Logger.v(TAG + " onSuccess() : Fences set successfully.");
            }
          })
          .addOnFailureListener(new OnFailureListener() {
            @Override public void onFailure(@NonNull Exception e) {
              Logger.v(TAG + " onFailure() : Fences could not be set.");
            }
          });
    } catch (Exception e) {
      Logger.e(TAG + " setGeofences() : ", e);
    }
  }

  boolean hasRequiredPermission(Context context) {
    return MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)
        || MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION);
  }

  public void removeGeofence() {
    List<String> existingFenceIdList =
        LocationInjector.getInstance().getRepository(context).getSavedGeoIds();
    if (existingFenceIdList == null || existingFenceIdList.isEmpty()) return;
    LocationServices.getGeofencingClient(context).removeGeofences(existingFenceIdList);
  }

  void updateFenceAndLocation(MoEJobParameters jobParameters) {
    Logger.v(TAG + " updateFenceAndLocation() : Will try to update fence and location");
    if (!hasRequiredPermission(context)) {
      Logger.v(TAG + " updateFenceAndLocation() : App does not have location permission cannot "
          + "fetch location");
      return;
    }
    triggerLocationFetch(context, jobParameters);
  }

  @SuppressLint("MissingPermission")void scheduleGeofenceFetchIfRequired() {
    Logger.v(TAG + " scheduleGeofenceFetchIfRequired() : ");
    if (!SdkConfig.getConfig().isGeofenceBackgroundSyncEnabled) return;
    if (Build.VERSION.SDK_INT >= VERSION_CODES.LOLLIPOP) {
      ComponentName serviceComponent = new ComponentName(context, CampaignSyncJob.class);
      JobInfo.Builder builder = new JobInfo.Builder(LocationConstants.CAMPAIGN_SYNC_JOB_ID, serviceComponent);
      builder.setOverrideDeadline(MoEUtils.currentMillis() + LocationConstants.BACKGROUND_SYNC_DELAY + (3600 * 1000));
      builder.setMinimumLatency(LocationConstants.BACKGROUND_SYNC_DELAY);
      builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
      if (MoEHelperUtils.hasPermission(context, permission.RECEIVE_BOOT_COMPLETED)) {
        builder.setPersisted(true);
      }
      JobScheduler jobScheduler =
          (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
      if (jobScheduler != null) {
        jobScheduler.schedule(builder.build());
      }
    } else {
      Intent alarmIntent = new Intent(context, CampaignSyncAlarm.class);
      PendingIntent pendingIntent =
          PendingIntent.getBroadcast(context, LocationConstants.CAMPAIGN_SYNC_ALARM_ID,
              alarmIntent, PendingIntent.FLAG_UPDATE_CURRENT);
      AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
      if (alarmManager != null) {
        alarmManager.set(AlarmManager.RTC_WAKEUP, MoEUtils.currentMillis() + LocationConstants.BACKGROUND_SYNC_DELAY,
            pendingIntent);
      }
    }
  }

  @SuppressWarnings("MissingPermission")
  private void triggerLocationFetch(Context context, final MoEJobParameters jobParameters) {
    Logger.v(TAG + " triggerLocationFetch() : Fetching last known location.");
    FusedLocationProviderClient client =
        LocationServices.getFusedLocationProviderClient(context);
    client.getLastLocation().addOnCompleteListener(new OnCompleteListener<Location>() {
      @SuppressWarnings("ConstantConditions") @Override
      public void onComplete(@NonNull Task<Location> task) {
        try {
          Logger.v(TAG + " onComplete() : Location fetch completed.");
          if (task == null) return;
          GeoLocation geoLocation = new GeoLocation(0.0, 0.0);
          Location lastLocation = task.getResult();
          if (lastLocation != null) {
            geoLocation.latitude = lastLocation.getLatitude();
            geoLocation.longitude = lastLocation.getLongitude();
          }
          Logger.v(TAG + " onComplete() : Location: " + geoLocation);
          updateLastKnownLocation(geoLocation);
          updateFencesIfRequired(geoLocation, jobParameters);
        } catch (Exception e) {
          Logger.e(TAG + " onComplete() : ", e);
        }
      }
    });
  }

  private void updateFencesIfRequired(GeoLocation location, MoEJobParameters jobParameters) {
    if (!SdkConfig.getConfig().isGeofenceTrackingOptedOut) {
      MoEDispatcher.getInstance(context).startTask(new GeofenceFetchTask(context, location, jobParameters));
    }
  }

  private void updateLastKnownLocation(GeoLocation geoLocation) {
    if (SdkConfig.getConfig().isLocationTrackingOptedOut) return;
    LocationRepository repository = LocationInjector.getInstance().getRepository(context);
    GeoLocation storedLocation = repository.getLastLocation();
    if (!geoLocation.equals(storedLocation)) {
      repository.saveLocation(geoLocation);
      MoEHelper.getInstance(context)
          .setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LOCATION, geoLocation);
    }
  }

  void onGeofenceHit(Intent geoFenceIntent) {
    try {
      if (geoFenceIntent == null) return;
      GeofencingEvent geofencingEvent = GeofencingEvent.fromIntent(geoFenceIntent);

      if (geofencingEvent.hasError()) {
        Logger.v(TAG
            + " onGeofenceHit() : Received geofence transition intent has error.Error: "
            + GeofenceStatusCodes
            .getStatusCodeString(geofencingEvent.getErrorCode()));
        return;
      }

      OnGeofenceHitListener listener = MoEGeofenceHelper.getInstance().getListener();
      if (listener != null && listener.geofenceHit(geoFenceIntent)) {
        Logger.v(TAG + " geoFenceHitInternal() : Geo-fence hit consumed by the client. SDK will "
            + "not process hit intent.");
        return;
      }
      Logger.v(
          TAG + " onGeofenceHit() : Trigger Fences: " + geofencingEvent.getTriggeringGeofences());

      List<Geofence> triggeringGeoFences = geofencingEvent.getTriggeringGeofences();
      Location location = geofencingEvent.getTriggeringLocation();
      GeoLocation triggeringLocation =
          new GeoLocation(location.getLatitude(), location.getLongitude());
      if (triggeringGeoFences == null || triggeringGeoFences.isEmpty()) return;
      String transitionType = getTransitionString(geofencingEvent.getGeofenceTransition());
      for (Geofence fence : triggeringGeoFences) {
        Logger.v(TAG + " onGeofenceHit() : Processing Fence: " + fence);
        LocationInjector.getInstance().getRepository(context).geofenceHit(triggeringLocation,
            getGeoIdFromRequestId(fence.getRequestId()), transitionType,
            MoEHelper.isAppInForeground());
        trackGeoFenceHitEvent(context, transitionType, fence,
            triggeringLocation);
      }
    } catch (Exception e) {
      Logger.e(TAG + " onGeofenceHit() : ", e);
    }
  }

  private void trackGeoFenceHitEvent(Context context, String transitionType, Geofence fence,
      GeoLocation lastLocation) {
    Properties attributes = new Properties();
    String campaignId = getCampaignIdFromRequestId(fence.getRequestId());
    if (TextUtils.isEmpty(campaignId)) return;
    attributes.addAttribute(LocationConstants.EVENT_ATTRIBUTE_CAMPAIGN_ID, campaignId);
    attributes.addAttribute(LocationConstants.EVENT_ATTRIBUTE_TRIGGER_LOCATION, lastLocation);
    if (!TextUtils.isEmpty(transitionType)) {
      attributes.addAttribute(LocationConstants.EVENT_ATTRIBUTE_TRANSITION_TYPE, transitionType);
    }
    String geoId = getGeoIdFromRequestId(fence.getRequestId());
    if (!TextUtils.isEmpty(geoId)) {
      attributes.addAttribute(LocationConstants.EVENT_ATTRIBUTE_GEO_ID, geoId);
    }
    attributes.setNonInteractive();
    MoEHelper.getInstance(context)
        .trackEvent(MoEHelperConstants.EVENT_GEO_FENCE_HIT, attributes);
  }

  @Nullable
  private String getGeoIdFromRequestId(String requestId) {
    if (MoEUtils.isEmptyString(requestId)) return null;
    if (!requestId.contains(LocationConstants.GEO_ID_SEPARATOR)) return requestId;
    String[] splitString = requestId.split(LocationConstants.GEO_ID_SEPARATOR);
    if (splitString != null) {
      return splitString[0];
    }
    return null;
  }

  @Nullable
  private String getCampaignIdFromRequestId(String requestId) {
    if (MoEUtils.isEmptyString(requestId)) return null;
    if (requestId.contains(LocationConstants.GEO_ID_SEPARATOR)) {
      String[] splitString = requestId.split(LocationConstants.GEO_ID_SEPARATOR);
      if (splitString != null && splitString.length != 0) {
        return splitString[splitString.length - 1];
      }
    }
    return null;
  }

  @Nullable
  private String getTransitionString(int geoFenceTransition) {
    switch (geoFenceTransition) {
      case Geofence.GEOFENCE_TRANSITION_ENTER:
        return LocationConstants.TRANSITION_ENTER;
      case Geofence.GEOFENCE_TRANSITION_DWELL:
        return LocationConstants.TRANSITION_DWELL;
      case Geofence.GEOFENCE_TRANSITION_EXIT:
        return LocationConstants.TRANSITION_EXIT;
      default:
        return null;
    }
  }

  void syncFencesAndScheduleFetch(MoEJobParameters jobParameters) {
    if (SdkConfig.getConfig().isBackgroundLocationFetchEnabled){
      updateFenceAndLocation(jobParameters);
    }else {
      updateFencesIfRequired(
          LocationInjector.getInstance().getRepository(context).getLastLocation(), jobParameters);
    }
    scheduleGeofenceFetchIfRequired();
  }

  void syncFencesAndScheduleFetch() {
    syncFencesAndScheduleFetch(null);
  }
}
