/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.geofence.internal;

import android.content.Context;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.GeoLocation;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.MoEJobParameters;
import com.moengage.geofence.internal.model.GeofenceFetchResponse;
import com.moengage.geofence.internal.repository.LocationRepository;

/**
 * @author Umang Chamaria
 * Date: 05/02/20
 */
public class GeofenceFetchTask extends SDKTask {
  private static final String TAG = LocationConstants.MODULE_TAG + "GeofenceFetchTask";

  private GeoLocation currentLocation;
  private MoEJobParameters jobParameters;

  GeofenceFetchTask(Context context, GeoLocation currentLocation) {
    this(context, currentLocation, null);
  }

  GeofenceFetchTask(Context context, GeoLocation location, MoEJobParameters jobParameters) {
    super(context);
    this.currentLocation = location;
    this.jobParameters = jobParameters;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " execute() : Executing geofence fetch");
      LocationRepository repository = LocationInjector.getInstance().getRepository(context);
      if (!new LocationEvaluator().isSyncRequired(LocationController.getInstance(context)
              .isSynced,
          repository.getLastSyncTime(),
          MoEUtils.currentMillis())) {
        Logger.v(TAG + " execute() : Geofence sync not required.");
        return taskResult;
      }
      GeofenceFetchResponse response = repository.fetchGeofence(currentLocation,
          MoEHelper.isAppInForeground());
      if (!response.isSuccess) return taskResult;
      taskResult.setIsSuccess(true);
      LocationController.getInstance(context).setGeofence(context, response.campaignList);
      LocationController.getInstance(context).setSynced(true);
      repository.saveRequestIdList(response.campaignList);

      if (jobParameters != null) {
        jobParameters.jobCompleteListener.jobComplete(jobParameters);
      }
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    Logger.v(TAG + " execute() : Completed fetch");
    return taskResult;
  }

  @Override public String getTaskTag() {
    return TAG_GEOFENCE_FETCH;
  }

  @Override public boolean isSynchronous() {
    return true;
  }
}
