package com.moengage.locationlibrary;

import android.Manifest;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofenceStatusCodes;
import com.google.android.gms.location.GeofencingClient;
import com.google.android.gms.location.GeofencingEvent;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.GeoTask;
import com.moengage.core.Logger;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEEventManager;
import com.moengage.core.MoEUtils;
import com.moengage.location.GeoManager;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

public class LocationHandlerImpl
    implements GeoManager.LocationHandler, OnCompleteListener<Location>{

  private Context context;
  private int mode = -1;
  private GeoLocation geoLocation;

  private String geoResponse;

  private Intent geoFenceIntent;

  boolean isLocationSynced = false;

  private final static int SET_GEO_FENCE = 1;
  private final static int GEO_HIT = 2;
  private final static int UPDATE_FENCE = 3;

  @Override public void setGeoFences(@NonNull Context context, String response){
    Logger.v("LocationHandlerImpl: inside setGeoFences()");
    this.context = context;
    mode = SET_GEO_FENCE;
    geoResponse = response;
    setGeoFenceInternal();
  }

  @Override public void onGeoFenceHit(@NonNull Context context, Intent intent){
    Logger.v("LocationHandlerImpl: inside onGeoFenceHit()");
    this.context = context;
    mode = GEO_HIT;
    this.geoFenceIntent = intent;
    triggerLastLocationFetch();
  }

  @Override public void updateFenceAndLocation(@NonNull Context context){
    Logger.v("LocationHandlerImpl: inside updateFenceAndLocation()");
    this.context = context;
    mode = UPDATE_FENCE;
    if (isSyncRequired(context) && (!ConfigurationProvider.getInstance(context)
        .isTrackLocationProhibited() || !ConfigurationProvider.getInstance(context)
        .isSetGeoFenceProhibited())) {
      triggerLastLocationFetch();
    }
  }

  @Override public void removeGeoFences(Context context) {
    this.context = context;
    removeGeoFencesInternal();
  }

  private GeofencingClient getGeoFencingClient(){
    return LocationServices.getGeofencingClient(context);
  }

  private void triggerLastLocationFetch(){
    try{
      if (MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)
          || MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION)){
        FusedLocationProviderClient client =
            LocationServices.getFusedLocationProviderClient(context);
        client.getLastLocation().addOnCompleteListener(this);
      }
    } catch(Exception e){
      Logger.f("LocationHandlerImpl: triggerLastLocationFetch() ", e);
    }
  }

  @Override public void onComplete(@NonNull Task<Location> task){
    try {
      if (task != null){
        Location lastLocation = task.getResult();
        if (lastLocation != null){
          geoLocation = new GeoLocation(lastLocation.getLatitude(), lastLocation.getLongitude());
        } else{
          geoLocation = new GeoLocation(0.0, 0.0);
        }
        switch(mode){
          case SET_GEO_FENCE:
            setGeoFenceInternal();
            break;
          case GEO_HIT:
            geoFenceHitInternal();
            break;
          case UPDATE_FENCE:
            updateFenceAndLocationInternal();
            break;
        }
      }
    } catch (Exception e) {
      Logger.f("LocationHandlerImpl: onComplete()", e);
    }
  }

  private void geoFenceHitInternal(){
    try{
      if (geoFenceIntent == null) return;
      GeofencingEvent geofencingEvent = GeofencingEvent.fromIntent(geoFenceIntent);
      if (geofencingEvent == null){
        Logger.e("LocationHandlerImpl : Null geo fence transition event");
        return;
      }
      if (geofencingEvent.hasError()){
        Logger.e("LocationHandlerImpl : Received geo fence transition intent with error"
            + GeofenceStatusCodes.getStatusCodeString(geofencingEvent.getErrorCode()));
        return;
      }

      Logger.v("LocationHandlerImpl geoFenceHitInternal() triggering Fence :" + geofencingEvent
          .getTriggeringGeofences().toString());
      Logger.v("LocationHandlerImpl geoFenceHitInternal() transition: " + geofencingEvent
          .getGeofenceTransition());
      Logger.v("LocationHandlerImpl geoFenceHitInternal() : Received geo fence transition intent");
      int geoFenceTransition = geofencingEvent.getGeofenceTransition();

      if (geoFenceTransition == Geofence.GEOFENCE_TRANSITION_ENTER
          || geoFenceTransition == Geofence.GEOFENCE_TRANSITION_EXIT
          || geoFenceTransition == Geofence.GEOFENCE_TRANSITION_DWELL){
        List<Geofence> triggeringGeoFences = geofencingEvent.getTriggeringGeofences();
        if (triggeringGeoFences != null && !triggeringGeoFences.isEmpty()){
          Logger.v("LocationHandlerImpl : geoFenceHitInternal() received geo fences count: " +
              triggeringGeoFences.size());
          for (Geofence fence : triggeringGeoFences){
            Logger.v("LocationHandlerImpl : geoFenceHitInternal() registering geo fencing hit for "
                + "GeoId: " + fence.getRequestId());
            HashMap<String, String> paramsMap =
                buildRequestParams(geoFenceTransition, fence, geoLocation);
            GeoTask geoFenceHitTask = new GeoTask(context,
                MoEUtils.getAPIRoute(context) + LocationConstants.API_ENDPOINT_GEOFENCEHIT,
                paramsMap, GeoManager.TASK_TYPE.GEOFENCE_HIT);
            MoEDispatcher.getInstance(context).addTaskToQueue(geoFenceHitTask);
            trackGeoFenceHitEvent(geoFenceTransition, fence, geoLocation);
          }
        }
      } else{
        Logger.e("LocationHandlerImpl : geoFenceHitInternal() : Transition type was not in our "
            + "interest: " + geoFenceTransition);
      }
    } catch(Exception e){
      Logger.f("LocationHandlerImpl : geoFenceHitInternal()", e);
    }
  }

  /**
   * Removes an existing geo-fence and adds any active geo-fence
   */
  private void setGeoFenceInternal(){
    try {
      ArrayList<Geofence> tFences = parseGeoFences();
      if (tFences != null && !tFences.isEmpty()){
        //get existing fences list
        removeGeoFencesInternal();
        if (tFences.isEmpty()){
          Logger.v("LocationHandlerImpl: setGeoFenceInternal(): No new geo fences found");
          return;
        }
        //store the geo ids on the client side
        StringBuilder geoIDBuilder = new StringBuilder();
        int size = tFences.size();
        for (int index = 0; index < size; index++){
          Geofence fence = tFences.get(index);
          geoIDBuilder.append(fence.getRequestId());
          if (index < size - 2){
            geoIDBuilder.append(MoEHelperConstants.EVENT_SEPERATOR);
          }
        }
        ConfigurationProvider.getInstance(context).saveGeoIDList(geoIDBuilder.toString());

        Intent intent = new Intent(context, GeofenceIntentService.class);
        PendingIntent geoPending =
            PendingIntent.getService(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        GeofencingRequest.Builder builder = new GeofencingRequest.Builder();
        builder.addGeofences(tFences);
        try{
          getGeoFencingClient().addGeofences(builder.build(), geoPending);
        } catch(Exception e){
          Logger.f("LocationHandlerImpl: setGeoFenceInternal()", e);
        }
      } else{
        Logger.v("LocationHandlerImpl: setGeoFenceInternal(): no fences to set");
      }
    } catch (Exception e) {
      Logger.f("LocationHandlerImpl: setGeoFenceInternal() ", e);
    }
  }

  /**
   * update last known location and fetch active geo-fences if location tracking and setting
   * geo-fences are not prohibited.
   */
  private void updateFenceAndLocationInternal(){
    HashMap<String, String> paramsMap = new HashMap<>();
    if (!ConfigurationProvider.getInstance(context).isTrackLocationProhibited()){
      updateLastKnownLocation();
    }
    if (!ConfigurationProvider.getInstance(context).isSetGeoFenceProhibited()){
      GeoLocation geoLocation =
          ConfigurationProvider.getInstance(context).getLastKnownUserLocation();
      if (geoLocation == null) geoLocation = new GeoLocation(0.0, 0.0);
      paramsMap.put(LocationConstants.PARAM_LAT, Double.toString(geoLocation.latitude));
      paramsMap.put(LocationConstants.PARAM_LNG, Double.toString(geoLocation.longitude));
      MoEDispatcher.getInstance(context)
          .addTaskToQueue(new GeoTask(context,
              MoEUtils.getAPIRoute(context) + LocationConstants.API_ENDPOINT_GEOFENCES, paramsMap,
              GeoManager.TASK_TYPE.GET_GEOFENCE));
    }
    isLocationSynced = true;
  }

  /**
   * update last known-location
   */
  private void updateLastKnownLocation(){
    Logger.v("LocationHandlerImpl: inside updateLastKnownLocation()");
    GeoLocation storedLocation =
        ConfigurationProvider.getInstance(context).getLastKnownUserLocation();
    if (!geoLocation.equals(storedLocation)){
      ConfigurationProvider.getInstance(context).storeLastKnownLocation(geoLocation);
      MoEHelper.getInstance(context)
          .setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LOCATION, geoLocation);
    }
  }

  /**
   * Parse geo-fence api response and marshall them into {@link Geofence} objects
   * @return list of {@link Geofence}
   */
  private ArrayList<Geofence> parseGeoFences(){

    JSONObject jsonResponse;
    if (!TextUtils.isEmpty(geoResponse)){
      try{
        jsonResponse = new JSONObject(geoResponse);
        JSONArray jsonPlaces =
            jsonResponse.getJSONArray(LocationConstants.RESPONSE_ATTR_FENCES_INFO);
        int placesLength = jsonPlaces.length();
        ArrayList<Geofence> newFences = new ArrayList<>(placesLength);

        for (int i = 0; i < placesLength; i++){
          try{
            JSONObject jsonFence = jsonPlaces.getJSONObject(i);
            String reqTransition =
                jsonFence.getString(LocationConstants.RESPONSE_ATTR_TRANSITION_TYPE);
            if (TextUtils.isEmpty(reqTransition)){
              continue;
            }
            int transitionType = Geofence.GEOFENCE_TRANSITION_ENTER;
            if (reqTransition.equals(LocationConstants.TRANSITION_TYPE_EXIT)){
              transitionType = Geofence.GEOFENCE_TRANSITION_EXIT;
            } else if (reqTransition.equals(LocationConstants.TRANSITION_TYPE_DWELL)){
              transitionType = Geofence.GEOFENCE_TRANSITION_DWELL;
            }
            Geofence.Builder builder = new Geofence.Builder().setRequestId(getGeoId(jsonFence))
                .setCircularRegion(jsonFence.getDouble(LocationConstants.PARAM_LAT),
                    jsonFence.getDouble(LocationConstants.PARAM_LNG),
                    (float) jsonFence.getDouble(LocationConstants.RESPONSE_ATTR_DISTANCE))
                .setExpirationDuration(Geofence.NEVER_EXPIRE)
                .setTransitionTypes(transitionType);

            if (jsonFence.has(LocationConstants.RESPONSE_ATTR_FENCE_LDELAY)){
              builder.setLoiteringDelay(Integer.parseInt(
                  jsonFence.getString(LocationConstants.RESPONSE_ATTR_FENCE_LDELAY)));
            }
            if (jsonFence.has(LocationConstants.RESPONSE_ATTR_FENCE_EXPIRY)){
              builder.setExpirationDuration(Long.parseLong(
                  jsonFence.getString(LocationConstants.RESPONSE_ATTR_FENCE_EXPIRY)));
            }
            if (jsonFence.has(LocationConstants.RESPONSE_ATTR_FENCE_RESPONSIVENESS)){
              builder.setNotificationResponsiveness(jsonFence.getInt(LocationConstants
                  .RESPONSE_ATTR_FENCE_RESPONSIVENESS));
            }
            newFences.add(builder.build());
          } catch(Exception e){
            Logger.f("Location: parseFencesInfo() - INNER", e);
          }
        }
        return newFences;
      } catch(Exception e){
        Logger.f("LocationHandlerImpl: parseFencesInfo()", e);
      }
    }
    return null;
  }

  private HashMap<String, String> buildRequestParams(int geoFenceTransition, Geofence fence,
      GeoLocation lastLocation){
    HashMap<String, String> paramsMap = new HashMap<>();
    String transitionType = getTransitionString(geoFenceTransition);
    if (!TextUtils.isEmpty(transitionType)){
      paramsMap.put(LocationConstants.RESPONSE_ATTR_TRANSITION_TYPE, transitionType);
    }
    if (lastLocation != null){
      paramsMap.put(LocationConstants.CURRENT_LATITUDE, String.valueOf(lastLocation.latitude));
      paramsMap.put(LocationConstants.CURRENT_LONGITUDE,
          String.valueOf(lastLocation.longitude));
    }
    paramsMap.put(LocationConstants.PARAM_GEOIDS, getGeoIdFromRequestId(fence.getRequestId()));
    return paramsMap;
  }

  @Nullable
  private List<String> getSavedGeoIds(){
    String geoList = ConfigurationProvider.getInstance(context).getGeoIDList();
    Logger.v("LocationHandlerImpl: setGeoFenceInternal(): Existing fences: " + geoList);
    List<String> oldGeoFenceList = null;
    if (!TextUtils.isEmpty(geoList)){
      if (geoList.contains(MoEHelperConstants.EVENT_SEPERATOR)){
        String geoIds[] = geoList.split(MoEHelperConstants.EVENT_SEPERATOR);
        oldGeoFenceList = Arrays.asList(geoIds);
      } else{
        //only one geo fence found
        oldGeoFenceList = new ArrayList<>();
        oldGeoFenceList.add(geoList);
      }
    }
    return oldGeoFenceList;
  }

  private boolean isSyncRequired(Context context) {
    long nextUpdate = ConfigurationProvider.getInstance(context).getLastGeoFenceSyncTime() + 900000;
    long currentTime = MoEUtils.currentTime();
    Logger.v("Location: isSyncRequired: Next server sync will happen in "
        + (nextUpdate - currentTime) / 1000
        + " seconds");
    return (!isLocationSynced || nextUpdate < currentTime);
  }

  private String getGeoId(JSONObject jsonFence) throws JSONException {
    String geoId = jsonFence.getString(LocationConstants.RESPONSE_ATTR_GEOID);
    if (jsonFence.has(LocationConstants.RESPONSE_ATTR_FENCE_CAMPAIGN_ID)){
      String campaignId = jsonFence.getString(LocationConstants.RESPONSE_ATTR_FENCE_CAMPAIGN_ID);
      if (!TextUtils.isEmpty(campaignId) && !campaignId.equals("null")){
        geoId = geoId.concat(LocationConstants.GEO_ID_SEPARATOR).concat(campaignId);
      }
    }
    return geoId;
  }

  private void trackGeoFenceHitEvent(int geoFenceTransition, Geofence fence,
      GeoLocation lastLocation) {
    PayloadBuilder attributes = new PayloadBuilder();
    String campaignId = getCampaignIdFromRequestId(fence.getRequestId());
    if (TextUtils.isEmpty(campaignId)) return;
    attributes.putAttrString(LocationConstants.EVENT_ATTRIBUTE_CAMPAIGN_ID, campaignId);
    attributes.putAttrLocation(LocationConstants.EVENT_ATTRIBUTE_TRIGGER_LOCATION, lastLocation);
    String transitionType = getTransitionString(geoFenceTransition);
    if (!TextUtils.isEmpty(transitionType)){
      attributes.putAttrString(LocationConstants.EVENT_ATTRIBUTE_TRANSITION_TYPE, transitionType);
    }
    String geoId = getGeoIdFromRequestId(fence.getRequestId());
    if (!TextUtils.isEmpty(geoId)){
      attributes.putAttrString(LocationConstants.EVENT_ATTRIBUTE_GEO_ID, geoId);
    }
    MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_GEO_FENCE_HIT, attributes
        .build());
  }

  @Nullable
  private String getTransitionString(int geoFenceTransition){
    switch (geoFenceTransition){
      case 1:
        return LocationConstants.TRANSITION_TYPE_ENTER;
      case 4:
        return LocationConstants.TRANSITION_TYPE_DWELL;
      case 2:
        return LocationConstants.TRANSITION_TYPE_EXIT;
    }
    return null;
  }

  @Nullable
  private String getCampaignIdFromRequestId(String requestId){
    try {
      if (TextUtils.isEmpty(requestId)) return null;
      if (requestId.contains(LocationConstants.GEO_ID_SEPARATOR)){
        String[] splitString = requestId.split(LocationConstants.GEO_ID_SEPARATOR);
        if (splitString != null && splitString.length != 0){
          return splitString[splitString.length-1];
        }
      }
    } catch (Exception e) {
      Logger.f("LocationHandlerImpl: getCampaignIdFromRequestId() ", e);
    }
    return null;
  }

  @Nullable
  private String getGeoIdFromRequestId(String requestId){
    try {
      if (TextUtils.isEmpty(requestId)) return null;
      if (!requestId.contains(LocationConstants.GEO_ID_SEPARATOR)) return requestId;
      String[] splitString = requestId.split(LocationConstants.GEO_ID_SEPARATOR);
      if (splitString != null){
        return splitString[0];
      }
    }catch (Exception e){
      Logger.f("LocationHandlerImpl getGeoIdFromRequestId() ", e);
    }
    return null;
  }

  private void removeGeoFencesInternal(){
    List<String> oldGeoFenceList = getSavedGeoIds();
    if (oldGeoFenceList != null){
      Logger.v("LocationHandlerImpl: removeGeoFencesInternal(): Removing all existing geo fences");
      getGeoFencingClient().removeGeofences(oldGeoFenceList);
    }
  }
}
