/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.widgets;

import android.app.Activity;
import android.content.Context;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import com.moengage.core.Logger;
import com.moengage.core.executor.OnTaskCompleteListener;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.executor.TaskProcessor;
import com.moengage.core.executor.TaskResult;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.InAppController;
import com.moengage.inapp.model.NudgeViewResult;
import com.moengage.inapp.tasks.BuildNudgeViewTask;
import java.lang.ref.WeakReference;
import java.util.Observable;
import java.util.Observer;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.moengage.core.executor.SDKTask.TAG_BUILD_NUDGE_VIEW_TASK;

/**
 * This is a Widget which represents a Nudge (A small text message with a close
 * button).
 *
 * @author MoEngage (abhishek@moenegage.com)
 * @version 5.0
 * @since 5.0
 *
 */
public class NudgeView extends LinearLayout implements OnTaskCompleteListener {

  private static final String TAG = InAppConstants.MODULE_TAG + "NudgeView";

  public NudgeView(Context contextList) {
    this(contextList, null);
  }

  private Context context;
  private Activity nudgeView;
  private String requestId;

  public NudgeView(Context contextList, AttributeSet attrs) {
    super(contextList, attrs);
    LayoutParams layoutParams =
        new LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
    setLayoutParams(layoutParams);
    setVisibility(INVISIBLE);
    setOrientation(VERTICAL);
    context = contextList;
    requestId = UUID.randomUUID().toString();
  }

  @Override protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();
  }

  public void initialiseNudgeView(Activity activity) {
    Logger.v("NudgeView: inside initialiseNudgeView()");
    this.nudgeView = activity;
    queryForNudge();
  }

  @Override public void onTaskComplete(String tag, TaskResult taskResult) {
    if (TAG_BUILD_NUDGE_VIEW_TASK.equals(tag)) {
      Logger.v(TAG + " onTaskComplete() : Building nudge view completed.");
      wip.set(false);
      if (taskResult.isSuccess()
          && taskResult.getPayload() != null
          && taskResult.getPayload() instanceof NudgeViewResult) {
        NudgeViewResult nudgeViewResult = (NudgeViewResult) taskResult.getPayload();
        if (!nudgeViewResult.requestId.equals(requestId)){
          Logger.v( TAG + " onTaskComplete() : Nudge view not for this request.");
          return;
        }
        addNudge(nudgeViewResult);
      }
    }
  }

  private class InAppCacheObserver implements Observer {
    @Override public void update(Observable observable, Object data) {
      //query and show nudge
      queryForNudge();
    }
  }

  private InAppCacheObserver inAppCacheObserver = new InAppCacheObserver();

  private WeakReference<OnTaskCompleteListener> completeListener;

  @Override protected void onWindowVisibilityChanged(int visibility) {
    super.onWindowVisibilityChanged(visibility);
    Logger.v(TAG + " onWindowVisibilityChanged() : Visibility: " + visibility);
    if (VISIBLE != visibility) {
      if (alreadyObserving) {
        InAppController.getInstance().unregisterSyncObserver(inAppCacheObserver);
        if (completeListener != null) {
          TaskProcessor.getInstance().removeOnTaskCompleteListener(completeListener);
        }
        alreadyObserving = false;
      }
    } else {
      InAppController.getInstance().registerSyncObserver(inAppCacheObserver);
      completeListener =
          TaskProcessor.getInstance().setOnTaskCompleteListener(this);
      alreadyObserving = true;
    }
  }

  boolean alreadyObserving = false;

  private final Object queryLock = new Object();

  private AtomicBoolean wip = new AtomicBoolean(false);

  void addNudge(final NudgeViewResult nudgeViewResult) {
    try {
      Logger.v(TAG + " addNudge() : Will attempt to show nudge view.");
      InAppController.getInstance().mainThreadHandler.post(new Runnable() {
        @Override public void run() {
          try {
            Logger.v(TAG + " run() : Adding nudge to layout.");
            addView(nudgeViewResult.view);
            InAppController.getInstance().onInAppShown(context, nudgeViewResult.campaignPayload);
            setVisibility(VISIBLE);
          } catch (Exception e) {
            Logger.e(TAG + " run() : Exception ", e);
          }
        }
      });
    } catch (Exception e) {
      Logger.e("NudgeView : addNudge ", e);
    }
  }

  private void queryForNudge() {
    if (wip.get()) {
      return;
    }
    synchronized (queryLock) {
      if (nudgeView != null) {
        if (getVisibility() == View.VISIBLE) {
          Logger.v(TAG + " queryForNudge() : Already showing a nudge");
          return;
        }
        TaskManager.getInstance().startTask(new BuildNudgeViewTask(context, requestId));
        wip.set(true);
      }
    }
  }
}
