/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.widgets;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.graphics.Shader;
import androidx.annotation.Keep;
import android.util.AttributeSet;
import android.widget.RatingBar;
import com.moengage.inapp.R;

@SuppressLint("AppCompatCustomView") @Keep
public class MoERatingBar extends RatingBar {

  private int color = Color.rgb(0x61, 0x61, 0x61);
  private int colorFillPressedOff = Color.TRANSPARENT;
  private int polygonVertices = 5;
  private int strokeWidth = -1; //width of the outline
  private final Paint paintInside = new Paint();
  private final Paint paintOutline = new Paint();
  private Path path = new Path();
  private final RectF rectangle = new RectF();
  private float interiorAngleModifier = 2.2F;
  private final float dp = getResources().getDisplayMetrics().density;
  private float starSize;
  private Bitmap colorsJoined;

  public MoERatingBar(Context context) {
    this(context, null);
  }

  public MoERatingBar(Context context, AttributeSet attrs) {
    super(context, attrs);
    getXmlAttrs(context, attrs);
    init();
  }

  public MoERatingBar(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    getXmlAttrs(context, attrs);
    init();
  }

  private void init() {
    paintInside.setAntiAlias(true);
    paintOutline.setStrokeWidth(strokeWidth);
    paintOutline.setStyle(Paint.Style.STROKE);
    paintOutline.setStrokeJoin(Paint.Join.ROUND); //Remove this line to create pointy stars
    paintOutline.setAntiAlias(true);
  }

  @Override protected synchronized void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
    int desiredWidth = (int) (40 * dp * getNumStars());
    int widthMode = MeasureSpec.getMode(widthMeasureSpec);
    int widthSize = MeasureSpec.getSize(widthMeasureSpec);
    int heightMode = MeasureSpec.getMode(heightMeasureSpec);
    int heightSize = MeasureSpec.getSize(heightMeasureSpec);

    int width;
    int height;

    //Measure Width
    if (widthMode == MeasureSpec.EXACTLY) {
      //Must be this size
      width = widthSize;
    } else if (widthMode == MeasureSpec.AT_MOST) {
      //Can't be bigger than...
      width = Math.min(desiredWidth, widthSize);
    } else {
      //Be whatever you want
      width = desiredWidth;
    }

    //Measure Height
    if (heightMode == MeasureSpec.EXACTLY) {
      //Must be this size
      height = heightSize;
    } else if (heightMode == MeasureSpec.AT_MOST) {
      //Can't be bigger than...
      height = Math.min(heightSize, width / getNumStars());
    } else {
      //Be whatever you want
      height = width / getNumStars();
    }

    //If starSize matches getHeight, the tips of the star can get cut off due to strokeWidth being added to the polygon size.
    //Make it a bit smaller to avoid this. Also decrease star size and spread them out rather than cutting them off if the
    //height is insufficient for the width.
    starSize = Math.min(height, width / getNumStars());
    if (strokeWidth < 0) strokeWidth = (int) (starSize / 15);
    starSize -= strokeWidth;

    //MUST CALL THIS
    setMeasuredDimension(width, height);
  }

  //Create a star polygon with any number of vertices, down to 2, which creates a diamond
  //If you enter 0 vertices, you get a circle
  private Path createStarBySize(float size, int steps) {
    //draw a simple circle if steps == 0
    if (steps == 0) {
      path.addOval(new RectF(0, 0, size, size), Path.Direction.CW);
      path.close();
      return path;
    }
    float halfSize = size / 2.0F;
    float radius = halfSize / interiorAngleModifier; //Adjusts "pointiness" of stars
    float degreesPerStep = (float) Math.toRadians(360.0F / (float) steps);
    float halfDegreesPerStep = degreesPerStep / 2.0F;
    path.setFillType(Path.FillType.EVEN_ODD);
    float max = (float) (2.0F * Math.PI);
    path.moveTo(halfSize, 0);
    for (double step = 0; step < max; step += degreesPerStep) {
      path.lineTo((float) (halfSize - halfSize * Math.sin(step)),
          (float) (halfSize - halfSize * Math.cos(step)));
      path.lineTo((float) (halfSize - radius * Math.sin(step + halfDegreesPerStep)),
          (float) (halfSize - radius * Math.cos(step + halfDegreesPerStep)));
    }
    path.close();
    return path;
  }

  @Override protected void onDraw(Canvas canvas) {

    //Default RatingBar changes color when pressed. This replicates the effect.
    BitmapShader shaderFillPressed = updateShader(color, colorFillPressedOff);
    paintInside.setShader(shaderFillPressed);

    path.rewind();
    path = createStarBySize(starSize, polygonVertices);

    for (int i = 0; i < getNumStars(); ++i) {
      //Default RatingBar only shows fractions in the interior, not the outline.
      paintOutline.setColor(color);

      path.computeBounds(rectangle, true);
      path.offset((i + .5F) * getWidth() / getNumStars() - rectangle.centerX(),
          getHeight() / 2 - rectangle.centerY());
      canvas.drawPath(path, paintInside);
      canvas.drawPath(path, paintOutline);
    }
  }

  //Create a BitmapShader, which is used to show fractions of stars
  private BitmapShader updateShader(int colorOn, int colorOff) {

    //Bitmap of width 0 will cause a crash. Make sure it's a positive number.
    int ratingWidth = (int) (getRating() * getWidth() / getNumStars());

    if (ratingWidth <= 0 || getWidth() - ratingWidth <= 0) {
      colorsJoined = Bitmap.createBitmap(getWidth(), getHeight(), Bitmap.Config.ARGB_8888);
      colorsJoined.eraseColor(ratingWidth <= 0 ? colorOff : colorOn);
    } else {
      Bitmap colorLeft = Bitmap.createBitmap(ratingWidth, getHeight(), Bitmap.Config.ARGB_8888);
      colorLeft.eraseColor(colorOn);
      Bitmap colorRight =
          Bitmap.createBitmap(getWidth() - ratingWidth, getHeight(), Bitmap.Config.ARGB_8888);
      colorRight.eraseColor(colorOff);
      colorsJoined = combineBitmaps(colorLeft, colorRight);
    }
    return new BitmapShader(colorsJoined, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
  }

  //Combine two bitmaps side by side for use as a BitmapShader
  private Bitmap combineBitmaps(Bitmap leftBitmap, Bitmap rightBitmap) {
    colorsJoined =
        Bitmap.createBitmap(leftBitmap.getWidth() + rightBitmap.getWidth(), leftBitmap.getHeight(),
            Bitmap.Config.ARGB_8888);

    Canvas comboImage = new Canvas(colorsJoined);
    comboImage.drawBitmap(leftBitmap, 0f, 0f, null);
    comboImage.drawBitmap(rightBitmap, leftBitmap.getWidth(), 0f, null);

    return colorsJoined;
  }

  //Set any XML attributes that may have been specified
  private void getXmlAttrs(Context context, AttributeSet attrs) {
    TypedArray a =
        context.getTheme().obtainStyledAttributes(attrs, R.styleable.MoERatingBar, 0, 0);
    try {
      color =
          a.getInteger(R.styleable.MoERatingBar_starColor, Color.rgb(0x61, 0x61, 0x61));
    } finally {
      a.recycle();
    }
  }

  public void setColor(int color){
    this.color = color;
  }
}
