package com.moengage.inapp.tasks;

import android.content.Context;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.InAppInjector;
import com.moengage.inapp.model.enums.StateUpdateType;
import com.moengage.inapp.model.meta.CampaignState;
import com.moengage.inapp.model.meta.InAppCampaign;
import com.moengage.inapp.repository.InAppRepository;

/**
 * Writes campaign stats to storage and updates the cache.
 * @author Umang Chamaria
 * Date: 2019-09-05
 */
public class UpdateCampaignStateTask extends SDKTask {

  private static final String TAG = InAppConstants.MODULE_TAG + "UpdateCampaignStateTask";

  private String campaignId;
  private StateUpdateType updateType;
  private boolean isSelfHandled;

  public UpdateCampaignStateTask(Context context, StateUpdateType updateType, String campaignId) {
    this(context, campaignId, updateType, false);
  }

  public UpdateCampaignStateTask(Context context, String campaignId,
      StateUpdateType updateType, boolean isSelfHandled) {
    super(context);
    this.campaignId = campaignId;
    this.updateType = updateType;
    this.isSelfHandled = isSelfHandled;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " execute() : Will log updated in-app state " + updateType + " for campaign "
          + "id " + campaignId);
      long currentTime = MoEUtils.currentSeconds();
      InAppRepository repository = InAppInjector.getInstance().getInAppRepository(context);
      InAppCampaign campaign = repository.localRepository.getCampaignById(campaignId);
      if (campaign == null) {
        Logger.v(TAG + " execute() : Campaign not found. Cannot update stats.");
        return taskResult;
      }
      if (isSelfHandled && !campaign.campaignMeta.templateType.equals(
          InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED)) {
        Logger.v(TAG + " execute() : Expected template type was self-handled. Not a self handled "
            + "campaign will ignore update.");
        return taskResult;
      }
      CampaignState currentState = campaign.campaignState;
      CampaignState updatedState = null;
      switch (updateType) {
        case SHOWN:
          updatedState = new CampaignState(currentState.showCount + 1, currentTime,
              currentState.isClicked);
          repository.localRepository.updateLastShowTime(currentTime);
          break;
        case CLICKED:
          updatedState = new CampaignState(currentState.showCount, currentState.lastShowTime, true);
          break;
      }

      int updateCount = repository.localRepository.updateCampaignState(updatedState,
          campaign.campaignMeta.campaignId);

      repository.updateCache();

      if (updateCount > 0) {
        taskResult.setIsSuccess(true);
      }
      Logger.v(TAG
          + " execute() : Updated in-app state for campaign id: "
          + campaignId
          + "updated campaign: "
          + updateCount);
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    return taskResult;
  }

  @Override public String getTaskTag() {
    return TAG_UPDATE_CAMPAIGN_STATE_TASK;
  }

  @Override public boolean isSynchronous() {
    return false;
  }
}
