/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.tasks;

import android.content.Context;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.evaluator.ConditionEvaluator;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.InAppController;
import com.moengage.inapp.InAppEvaluator;
import com.moengage.inapp.InAppInjector;
import com.moengage.inapp.InAppUtils;
import com.moengage.inapp.model.CampaignPayload;
import com.moengage.inapp.model.CampaignRequest;
import com.moengage.inapp.model.TriggerRequestMeta;
import com.moengage.inapp.model.meta.InAppCampaign;
import com.moengage.inapp.model.meta.Trigger;
import com.moengage.inapp.repository.InAppRepository;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class ShowTriggerInAppTask extends SDKTask {

  private static final String TAG = InAppConstants.MODULE_TAG + "ShowTriggerInAppTask";

  private Event event;
  private InAppController controller;

  public ShowTriggerInAppTask(Context context, Event event) {
    super(context);
    this.event = event;
    this.controller = InAppController.getInstance();
  }

  @Override public TaskResult execute() {
    Logger.v(TAG + " execute() : started execution");
    try {
      if (!controller.isModuleEnabled(context)) {
        Logger.v(TAG + " execute() : InApp Module is disabled. Cannot show in-app.");
        return taskResult;
      }
      if (!controller.isInAppSynced()) {
        Logger.v(TAG + " execute() : Cannot show trigger in-app as sync is pending");
        return null;
      }
      InAppUtils.logCurrentInAppState(context);
      InAppRepository repository = InAppInjector.getInstance().getInAppRepository(context);
      InAppEvaluator evaluator = new InAppEvaluator();

      Set<String> triggerEvent =
          repository.cache.triggerEvents;
      if (triggerEvent == null || !triggerEvent.contains(event.eventName)) {
        Logger.v(TAG
            + " execute() : Given event is not a trigger event, event name: "
            + event.eventName);
        return null;
      }

      List<InAppCampaign> triggerCampaigns =
          repository.localRepository.getCampaignsForEvent(event.eventName);

      if (triggerCampaigns == null) {
        Logger.v(TAG + " execute() : No campaign for given event, This is strange.");
        return null;
      }

      List<InAppCampaign> filteredCampaigns = new ArrayList<InAppCampaign>();

      for (InAppCampaign campaign : triggerCampaigns) {
        JSONObject eventAttributesTransformed =
            MoEHelperUtils.transformEventAttributesForEvaluationPackage(event.attributes);
        if (evaluateCondition(campaign.campaignMeta.trigger, eventAttributesTransformed)) {
          filteredCampaigns.add(campaign);
        }
      }

      if (filteredCampaigns.size() == 0) {
        Logger.v(TAG + " execute() : No campaign satisfies the filter condition.");
        return taskResult;
      }

      if (!controller.canShowInAppForConfig(context, filteredCampaigns)) return taskResult;

      InAppCampaign suitableCampaign = evaluator.getEligibleCampaignFromList(filteredCampaigns,
          repository.localRepository.getGlobalState(),
          MoEHelper.getInstance(context).getAppContext());

      if (suitableCampaign == null) {
        Logger.v(TAG + " execute() : Did not find any suitable in-app");
        return taskResult;
      }
      // fetch in-app
      CampaignPayload campaignPayload =
          repository.fetchCampaignPayload(
              new CampaignRequest(repository.localRepository.baseRequest(),
                  suitableCampaign.campaignMeta.campaignId, controller.getCurrentActivityName(),
                  MoEHelper.getInstance(context).getAppContext(),
                  new TriggerRequestMeta(event.eventName,
                      MoEHelperUtils.transformEventAttributesForEvaluationPackage(event.attributes),
                      MoEUtils.currentISOTime())),
              suitableCampaign.campaignMeta.deliveryControl.persistent);

      if (campaignPayload == null) {
        Logger.v(TAG + " execute() : Campaign Payload is empty. Cannot show campaign.");
        return taskResult;
      }
      //build and show in-app
      taskResult.setIsSuccess(true);
      if (campaignPayload.templateType.equals(InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED)) {
        InAppController.getInstance().onSelfHandledAvailable(campaignPayload);
      } else {
        controller.buildAndShowInApp(context, suitableCampaign, campaignPayload);
      }

      Logger.v(TAG + " execute() : execution completed");
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    return taskResult;
  }

  private boolean evaluateCondition(Trigger condition,
      JSONObject eventAttributesTransformed) {
    boolean conditionEvaluatorResult = false;
    try {
      Logger.v(TAG
          + " execute() : Attribute JSON for evaluation"
          + eventAttributesTransformed.toString());
      if (condition.primaryCondition.attributes == null
          || condition.primaryCondition.attributes.length() == 0) {
        return true;
      }
      conditionEvaluatorResult =
          new ConditionEvaluator(condition.primaryCondition.attributes,
              eventAttributesTransformed).evaluate();
    } catch (Exception e) {
      Logger.e(TAG + " evaluateCondition() : ", e);
    }
    Logger.v(TAG + " evaluateCondition() : Evaluation result: " + conditionEvaluatorResult);
    return conditionEvaluatorResult;
  }

  @Override public String getTaskTag() {
    return TAG_CHECK_TRIGGER_BASED_IN_APP_TASK;
  }

  @Override public boolean isSynchronous() {
    return false;
  }
}
